#!/bin/sh -e
# ==============================================================================
# portsreinstall library script
# - Reinstallation processes -
# Copyright (C) 2013 Mamoru Sakaue, MwGhennndo, All Rights Reserved.
# This software is distributed under the 2-Clause BSD License.
# ==============================================================================

# ============= Variables =============
# The following variables are available only within reinstall_exec and functions invoked by it.
REINSTALL_PKGTAG=	# Tag indicating version changes by reinstallation of the current port
REINSTALL_ORIGPKGTAG=	# Tag indicating the port origin and version changes by reinstallation of the current port
REINSTALL_CURRENTPKG=	# Currently installed package name of the current port
REINSTALL_ORIGIN=	# Port origin of the current port
REINSTALL_DBNODE_DIR=	# Path of the "requires" section of the temporary database for the current port
REINSTALL_IS_CURRENTPKG_LATEST=	# Currently installed package name of the current port
REINSTALL_DBSUFFIX=	# Database suffix for the evaluation method of dependencies

# ============= Skip reinstallation by showing messages =============
reinstall_skip ()
{
	local message
	message=$1
	message_echo "($message)"
	message_target_relations "$REINSTALL_ORIGIN"
	message_echo
	fileedit_rm_a_line "$REINSTALL_ORIGIN" "${DBDIR}/stage.loop_list/reinst_todo.remain"
}

# ============= Restore a package if it is temporarily deinstalled =============
reinstall_restore_if_temporarily_deinstalled ()
{
	local backup_pkg
	[ -n "$REINSTALL_CURRENTPKG" ] && return
	backup_pkg=`cat "$REINSTALL_DBNODE_DIR/backedup_pkgfile" 2> /dev/null` || return 0
	[ -z "$backup_pkg" -o ! -e "$backup_pkg" ] && return
	message_echo "INFO: Temporarily deinstalled, so restoring from the backup."
	pkg_add_fF "$backup_pkg" && return
	message_echo "WARNING: Failed to restore the backup, but continuing anyway." >&2
}

# ============= Skip reinstallation by showing messages if an origin is in a list =============
reinstall_skip_if_in_a_list ()
{
	local message list mode origin_regexp_esc
	message=$1
	list=$2
	mode=$3
	origin_regexp_esc=`str_escape_regexp "$REINSTALL_ORIGIN"`
	grep -m 1 -E "^$origin_regexp_esc$" "${DBDIR}/$list" > /dev/null 2>&1 || return
	[ "x$mode" = xrestore ] && reinstall_restore_if_temporarily_deinstalled
	reinstall_skip "$message" || :
}

# ============= Get the make arguments =============
reinstall_setup_make_args ()
{
	local mode
	mode=$1
	{
		for key in LOCALBASE LINUXBASE PORT_DBDIR PORTSDIR DISTDIR PACKAGES PKGREPOSITORY
		do
			eval echo $key=\$$key
		done
		echo 'FORCE_PKG_REGISTER=yes'
		[ $opt_avoid_vulner = yes ] || echo 'DISABLE_VULNERABILITIES=yes'
		case $mode in
		anymode);;
		''|distinct)
			[ $opt_batch_ports_only = yes ] && echo 'BATCH=yes'
			[ $opt_interactive_ports_only = yes ] && echo 'INTERACTIVE=yes'
			;;
		esac
		cat "${DBDIR}/requires/$REINSTALL_ORIGIN/MARG.conf" 2> /dev/null || :
	} | tr '\n' ' '
}

# ============= Get the make environment variables =============
reinstall_setup_make_envs ()
{
	cat "${DBDIR}/requires/$REINSTALL_ORIGIN/MENV.conf" 2> /dev/null | tr '\n' ' '
}

# ============= Execute make command without restricting for BATCH or INTERACTIVE ports =============
reinstall_make_anymode ()
{
	local MAKE_ARGS MAKE_ENVS
	MAKE_ARGS=`reinstall_setup_make_args anymode`
	MAKE_ENVS=`reinstall_setup_make_envs`
	env $MAKE_ENVS make -C "${PORTSDIR}/$REINSTALL_ORIGIN" "$@" $MAKE_ARGS
}

# ============= Execute make command =============
reinstall_make ()
{
	local MAKE_ARGS MAKE_ENVS
	MAKE_ARGS=`reinstall_setup_make_args`
	MAKE_ENVS=`reinstall_setup_make_envs`
	env $MAKE_ENVS make -C "${PORTSDIR}/$REINSTALL_ORIGIN" "$@" $MAKE_ARGS
}

# ============= Error process during reinstallation =============
reinstall_errproc ()
{
	local position msg
	position=$1
	msg=$2
	database_build_update_pkgname "$REINSTALL_ORIGIN" > /dev/null
	message_echo "ERROR: In $position." >&2
	message_echo "The port/package in concern is $REINSTALL_ORIGPKGTAG." >&2
	[ -n "$msg" ] && message_echo "($msg)" >&2
	message_echo >&2
	echo "$position" > $REINSTALL_DBNODE_DIR/note_failtre
	database_record_failure "$REINSTALL_ORIGIN" noclean
	fileedit_rm_a_line "$REINSTALL_ORIGIN" "${DBDIR}/stage.loop_list/reinst_todo.remain"
	message_report_failure_for_a_port "$REINSTALL_ORIGIN"
}

# ============= Check the latest stage for a port =============
reinstall_chk_stage ()
{
	local stagetag
	stagetag=$1
	[ -e "${DBDIR}/status.ports/$REINSTALL_ORIGIN/$stagetag" ]
}

# ============= Check completion of a stage for a port =============
reinstall_chk_stage_complete ()
{
	local stagetag
	stagetag=$1
	[ -e "${DBDIR}/status.ports/$REINSTALL_ORIGIN/complete/$stagetag" ]
}

# ============= Register the latest stage for a port =============
reinstall_register_stage ()
{
	local stagetag parentdir
	stagetag=$1
	parentdir=${DBDIR}/status.ports/$REINSTALL_ORIGIN
	[ -d "$parentdir" ] || mkdir -p "$parentdir"
	touch "$parentdir/$stagetag"
}

# ============= Register completion of a stage for a port =============
reinstall_register_stage_complete ()
{
	local stagetag parentdir
	stagetag=$1
	parentdir=${DBDIR}/status.ports/$REINSTALL_ORIGIN/complete
	[ -d "$parentdir" ] || mkdir -p "$parentdir"
	touch "$parentdir/$stagetag"
}

# ============= Deregister the latest stage for a port =============
reinstall_deregister_stage ()
{
	local stagetag
	stagetag=$1
	rm -f "${DBDIR}/status.ports/$REINSTALL_ORIGIN/$stagetag"
}

# ============= Deregister completion of a stage for a port =============
reinstall_deregister_stage_complete ()
{
	local stagetag
	stagetag=$1
	rm -f "${DBDIR}/status.ports/$REINSTALL_ORIGIN/complete/$stagetag"
}

# ============= Back up and delete conflicts =============
# Package names of conflicts are given via stdin. 
reinstall_backup_and_delete_conflicts ()
{
	local pkg origin origin_regexp_esc backup_pkgdir backup_pkg
	message_echo "INFO: Deinstalling conflicting packages for $REINSTALL_ORIGPKGTAG."
	while read pkg
	do
		origin=`pkg_info_qo "$pkg"`
		message_echo "INFO: Backing up and deleting a conflict, $origin ($pkg)."
		origin_regexp_esc=`str_escape_regexp "$origin"`
		if [ -d "${DBDIR}/requires/$origin" ]
		then
			backup_pkgdir=${DBDIR}/backup_packages
		else
			backup_pkgdir=${PKGREPOSITORY}
		fi
		[ -d "$backup_pkgdir" ] || mkdir -p "$backup_pkgdir"
		if backup_pkg=`pkgsys_get_backup_pkg "$origin"`
		then
			message_echo "INFO: backup package already exists as $backup_pkg"
		elif ! backup_pkg=`pkgsys_create_backup_pkg "$pkg" "$backup_pkgdir"`
		then
			message_echo "WARNING: Failed to create the backup package, the conflict is kept installed." >&2
			continue
		fi
		grep -v -E "^${origin_regexp_esc}[[:space:]]" "${DBDIR}/deleted_conflicts" \
			> ${DBDIR}/deleted_conflicts.tmp 2> /dev/null || :
		mv "${DBDIR}/deleted_conflicts.tmp" "${DBDIR}/deleted_conflicts"
		printf '%s\t%s\n' "$origin" "$pkg" >> ${DBDIR}/deleted_conflicts
		pkg_delete_f "$pkg" || \
		{
			message_echo "WARNING: Failed to deinstall $REINSTALL_CURRENTPKG by $PKGSYS_CMD_PKG_DELETE." >&2
		}
	done
	cat "${DBDIR}/deleted_conflicts" 2> /dev/null | sort -u > ${DBDIR}/deleted_conflicts.tmp
	mv "${DBDIR}/deleted_conflicts.tmp" "${DBDIR}/deleted_conflicts"
}

# ============= Restoration of backed up conflicts =============
reinstall_restore_conflicts ()
{
	local origin_current tmpsrc
	origin_current=$1
	[ $opt_dry_run = yes ] && return
	[ -e "${DBDIR}/deleted_conflicts" ] || return 0
	tmpsrc=${TMPDIR}/reinstall_restore_conflicts::deleted_conflicts
	cp "${DBDIR}/deleted_conflicts" "$tmpsrc"
	while read origin pkg
	do
		pkg_regexp_esc=`str_escape_regexp "$pkg"`
		origin_orig=`echo "$origin" \
			| sed -E -f "${DBDIR}/REVERSE_REPLACE.complete_sed_pattern"`
		[ "x$origin_orig" = "x$origin" ] && origin_orig=
		origin_replace=`echo "$origin" \
			| sed -E -f "${DBDIR}/REPLACE.complete_sed_pattern"`
		[ "x$origin_replace" = "x$origin" ] && origin_replace=
		if [ -n "$origin_current" -a "x$origin" = "x$origin_current" ] || \
			pkg_info_e "$pkg" || pkg_info_eO "$origin" || \
			{ [ -n "$origin_orig" ] && pkg_info_eO "$origin_orig"; } || \
			{ [ -n "$origin_replace" ] && pkg_info_eO "$origin_replace"; }
		then
			grep -v -E "[[:space:]]$pkg_regexp_esc$" "${DBDIR}/deleted_conflicts" \
				> ${DBDIR}/deleted_conflicts.tmp || :
			mv "${DBDIR}/deleted_conflicts.tmp" "${DBDIR}/deleted_conflicts"
			continue
		fi
		message_echo "INFO: Restoring a conflict, $origin ($pkg)."
		if ! backup_pkg=`pkgsys_get_backup_pkg "$origin"`
		then
			message_echo "WARNING: No backup exists, gave up." >&2
			continue
		fi
		if pkg_add_fF "$backup_pkg"
		then
			grep -v -E "[[:space:]]$pkg_regexp_esc$" "${DBDIR}/deleted_conflicts" \
				> ${DBDIR}/deleted_conflicts.tmp || :
			mv "${DBDIR}/deleted_conflicts.tmp" "${DBDIR}/deleted_conflicts"
		else
			message_echo "WARNING: Failed to restore. Note that your system may experience troubles by this error." >&2
		fi
	done < $tmpsrc
}

# ============= Get the all requirements ready for a port by restarting them if deinstalled =============
reinstall_chk_and_restore_requirements ()
{
	local tmp_restore
	tmp_restore=${TMPDIR}/reinstall_setup_requirements:restore
	tmp_isfailed=${TMPDIR}/reinstall_setup_requirements:isfailed
	cp /dev/null "$tmp_restore"
	rm -f "$tmp_isfailed"
	cat "$REINSTALL_DBNODE_DIR/requirements.all.direct" 2> /dev/null \
		| while read origin
	do
		pkg_info_eO "$origin" && continue
		origin_orig=`echo "$origin" \
			| sed -E -f "${DBDIR}/REVERSE_REPLACE.complete_sed_pattern"`
		[ "x$origin_orig" != "x$origin" ] \
			&& pkg_info_eO "$origin_orig" && continue
		if ! pkgarc=`pkgsys_get_backup_pkg "$origin"` && \
			! pkgarc=`pkgsys_get_backup_pkg "$origin_orig"`
		then
			origin_regexp=`str_escape_regexp "$origin"`
			if grep -m 1 -E "^$origin_regexp$" "${DBDIR}/failed.list" > /dev/null 2>&1
			then
				touch "$tmp_isfailed"
				break
			fi
		fi
		printf '%s\t%s\n' "$origin_orig" "$pkgarc" >> $tmp_restore
	done
	[ -e "$tmp_isfailed" ] && return 1
	while read origin pkgarc
	do
		pkg=`pkgsys_pkgarc_to_pkgname "$pkgarc"`
		pkg_regexp_esc=`str_escape_regexp "$pkg"`
		message_echo "INFO: Restoring a backed-up requirement, $origin ($pkg)."
		if pkg_add_fF "$pkgarc"
		then
			grep -v -E "[[:space:]]$pkg_regexp_esc$" "${DBDIR}/deleted_conflicts" \
				> ${DBDIR}/deleted_conflicts.tmp || :
			mv "${DBDIR}/deleted_conflicts.tmp" "${DBDIR}/deleted_conflicts"
		else
			message_echo "WARNING: Failed to restore by the backed-up package." >&2
			touch "$tmp_isfailed"
			break
		fi
		
	done < $tmp_restore
	:
}

# ============= Back-up of the currently installed package =============
reinstall_pkg_backup ()
{
	local backup_pkg
	reinstall_chk_stage_complete PKG_BACKUP && return
	if [ -n "$REINSTALL_CURRENTPKG" ]
	then
		message_echo "-- (Creating temporary backup package for $REINSTALL_ORIGPKGTAG)"
		backup_pkg=`pkgsys_create_backup_pkg "$REINSTALL_CURRENTPKG" "${DBDIR}/backup_packages"` || \
			message_echo "WARNING: Failed to create the backup package, but ignored by hoping success." >&2
		cd "${PORTSDIR}/$REINSTALL_ORIGIN"
		echo "$REINSTALL_CURRENTPKG" > $REINSTALL_DBNODE_DIR/backedup_version
		echo "$backup_pkg" > $REINSTALL_DBNODE_DIR/backedup_pkgfile
	fi
	reinstall_register_stage_complete PKG_BACKUP
}

# ============= Deinstallation of the currently installed package =============
reinstall_deinstall ()
{
	[ -n "$REINSTALL_CURRENTPKG" ] || return 0
	reinstall_chk_stage_complete DEINSTALL && return
	pkg_delete_f "$REINSTALL_CURRENTPKG" || \
	{
		message_echo "WARNING: Failed to deinstall $REINSTALL_CURRENTPKG by $PKGSYS_CMD_PKG_DELETE." >&2
	}
	message_echo "-- (Trying to deinstall by ports to make sure. This usually ends up with warnings.)"
	reinstall_make deinstall || \
	{
		message_echo "WARNING: Failed to deinstall $REINSTALL_CURRENTPKG by make deinstall." >&2
	}
	reinstall_register_stage_complete DEINSTALL
}

# ============= Recovery after failure of installation of the new package =============
reinstall_failed_install_recover ()
{
	local backedup_version backup_pkg
	reinstall_chk_stage_complete FAILED_INSTALL.RECOVER && return
	message_echo "INFO: Trying to deinstall the failed/terminated installation (Ignore failures)."
	if [ -n "$REINSTALL_CURRENTPKG" ]
	then
		pkg_delete_f "$REINSTALL_CURRENTPKG" || :
	fi
	message_echo "INFO: Trying to deinstall by ports to make sure (This usually ends up with warnings)."
	reinstall_make deinstall || :
	backedup_version=`cat "$REINSTALL_DBNODE_DIR/backedup_version" 2> /dev/null || :`
	if [ -n "$backedup_version" ]
	then
		message_echo "INFO: Restoring the backup of $backedup_version."
		backup_pkg=`cat "$REINSTALL_DBNODE_DIR/backedup_pkgfile" 2> /dev/null || :`
		if [ -z "$backup_pkg" -o ! -e "$backup_pkg" ]
		then
			message_echo "WARNING: No backup exists, gave up." >&2
		elif ! pkg_add_fF "$backup_pkg"
		then
			message_echo "WARNING: Failed to restore $backedup_version. Note that your system may experience troubles by this error." >&2
		fi
	fi
	reinstall_register_stage_complete FAILED_INSTALL.RECOVER
}

# ============= Report an installation success to the all dependents =============
reinstall_tell_update_to_depandents ()
{
	local tag level dbsuffix
	pkgsys_is_pkgtool "$REINSTALL_ORIGIN" && return
	reinstall_chk_stage_complete TELL_UPDATE_TO_DEPANDENTSL && return
	for tag in all run build none
	do
		for level in full direct
		do
			dbsuffix=${tag}.${level}
			cat "$REINSTALL_DBNODE_DIR/dependents.$dbsuffix" 2> /dev/null \
				| while read origin_dependent
			do
				touch "${DBDIR}/requires/$origin_dependent/need_reinstall_due_to_upgraded_requirements.$dbsuffix"
				fileedit_rm_a_line "$origin_dependent" \
					"${DBDIR}/success.$dbsuffix.list"
				fileedit_rm_a_line "$origin_dependent" \
					"${DBDIR}/todo_after_requirements_succeed.$dbsuffix.list"
			done
		done
	done
	reinstall_register_stage_complete TELL_UPDATE_TO_DEPANDENTS
}

# ============= Closing operations after an installation success =============
reinstall_closing_operations_after_successful_install ()
{
	local tag level initial_orig 
	reinstall_chk_stage_complete CLOSING_OPERATIONS_AFTER_SUCCESSFUL_INSTALL && return
	database_build_update_pkgname "$REINSTALL_ORIGIN" > /dev/null
	for tag in all run build none
	do
		for level in full direct
		do
			rm -f "$REINSTALL_DBNODE_DIR/need_reinstall_due_to_upgraded_requirements.${tag}.${level}"
			[ -e "$REINSTALL_DBNODE_DIR/failed_requirements.${tag}.${level}" ] || continue
			cp "$REINSTALL_DBNODE_DIR/failed_requirements.${tag}.${level}" \
				"$REINSTALL_DBNODE_DIR/failed_requirements.${tag}.${level}.previous"
		done
	done
	pkgsys_delete_backup_pkg "$REINSTALL_ORIGIN"
	initial_orig=`cat "$REINSTALL_DBNODE_DIR/initial_orig" 2> /dev/null` || :
	[ -n "$initial_orig" ] && pkgsys_delete_backup_pkg "$initial_orig"
	rm -f "$REINSTALL_DBNODE_DIR/backedup_pkgfile"
	reinstall_deregister_stage in_install
	database_record_success "$REINSTALL_ORIGIN"
	temp_set_msg_current_stage "${_MSG_CURRENT_STAGE_general}"
	message_echo "===>  Done successfully"
	reinstall_register_stage_complete CLOSING_OPERATIONS_AFTER_SUCCESSFUL_INSTALL
}

# ============= Reinstallation for an origin =============
reinstall_exec ()
{
	( set -e
	REINSTALL_ORIGIN=$1
	REINSTALL_DBNODE_DIR=${DBDIR}/requires/$REINSTALL_ORIGIN
	REINSTALL_CURRENTPKG=`database_build_update_pkgname "$REINSTALL_ORIGIN"`
	REINSTALL_IS_CURRENTPKG_LATEST=no
	database_build_is_currentpkg_latest "$REINSTALL_ORIGIN" && REINSTALL_IS_CURRENTPKG_LATEST=yes
	database_build_update_pkgtag "$REINSTALL_ORIGIN"
	REINSTALL_PKGTAG=`cat "$REINSTALL_DBNODE_DIR/pkgtag"`
	REINSTALL_ORIGPKGTAG="$REINSTALL_ORIGIN ($REINSTALL_PKGTAG)"
	REINSTALL_DBSUFFIX=`options_get_dependency_type`.`options_get_dependency_level`
	REINSTALL_NEWPKGNAME=`database_build_get_new_pkgname "$REINSTALL_ORIGIN"`
	message_stage_title "$PROGRAM_STEP_COUNTER $REINSTALL_ORIGPKGTAG"
	database_query_get_target_attributes currentorigin "$REINSTALL_ORIGIN"
	if [ -z "${currentorigin_is_all}" -a -z "${currentorigin_is_relevant}" ]
	then
		reinstall_skip 'Skipped because being irrelevant'
		return
	fi
	reinstall_skip_if_in_a_list 'Skipped because being a leaf port' leaf_ports_to_delete && return
	if [ -e "${DBDIR}/target_all" ]
	then
		if [ $REINSTALL_IS_CURRENTPKG_LATEST = yes ]
		then
			reinstall_restore_if_temporarily_deinstalled
			reinstall_skip 'Skipped because being already latest' "$REINSTALL_ORIGIN"
			return
		fi
	elif [ ! -e "$REINSTALL_DBNODE_DIR/conf_updated" ]
	then
		if [ -e "$nodedir/installed_by_pkg" ] && database_query_is_default_conf "$REINSTALL_ORIGIN" quiet
		then
			reinstall_restore_if_temporarily_deinstalled
			reinstall_skip 'Skipped because already upgraded with a prebuilt package'
			return
		fi
		reinstall_skip_if_in_a_list 'Skipped because the reinstallation has been already completed' \
			"success.${REINSTALL_DBSUFFIX}.list" restore && return
		if [ $opt_skip_unchanged = yes ]
		then
			if [ ! -e "$REINSTALL_DBNODE_DIR/necessary_upgrade.${REINSTALL_DBSUFFIX}" ]
			then
				reinstall_restore_if_temporarily_deinstalled
				if [ -e "$REINSTALL_DBNODE_DIR/installed_version" ]
				then
					reinstall_skip 'Skipped because being already latest as well as the all requirements'
				else
					reinstall_skip 'Skipped because being an only-build-time dependency of already latest packages'
				fi
				return
			fi
		fi
		if [ \( $opt_skip_unchanged = no -a -e "$REINSTALL_DBNODE_DIR/succeeded_once" \) \
			-o \( $opt_skip_unchanged = yes -a $REINSTALL_IS_CURRENTPKG_LATEST = yes \) ]
		then
			if [ ! -e "$REINSTALL_DBNODE_DIR/need_reinstall_due_to_upgraded_requirements.${REINSTALL_DBSUFFIX}" ]
			then
				if ! database_query_is_necessary_upgrade "$REINSTALL_ORIGIN"
				then
					reinstall_restore_if_temporarily_deinstalled
					fileedit_add_a_line_if_new "$REINSTALL_ORIGIN" \
						"${DBDIR}/todo_after_requirements_succeed.${REINSTALL_DBSUFFIX}.list"
					reinstall_skip 'Skipped because being already latest or failed as well as the all requirements'
					return
				fi
			fi
		fi
	fi
	reinstall_skip_if_in_a_list 'Skipped because being a hold package' conf/HOLD:PORTS.parsed restore && return
	reinstall_skip_if_in_a_list 'Marked to be manually-done' manually_done.list restore && return
	if database_query_is_a_port_suppressed "$REINSTALL_ORIGIN"
	then
		reinstall_restore_if_temporarily_deinstalled
		reinstall_skip 'Skipped because being suppressed'
		return
	fi
	reinstall_skip_if_in_a_list 'Ignored because being taboo' taboo.all.list && return
	if ! reinstall_chk_and_restore_requirements
	then
		reinstall_restore_if_temporarily_deinstalled
		for tag in all run build none
		do
			for level in full direct
			do
				fileedit_add_a_line_if_new "$REINSTALL_ORIGIN" \
					"${DBDIR}/todo_after_requirements_succeed.${tag}.${level}.list"
			done
		done
		reinstall_skip 'Skipped because the requirements cannot be ready'
		return
	fi
	if [ -e "$REINSTALL_DBNODE_DIR/installed_version" ]
	then
		insttarget=reinstall
		instdesc='a reinstallation'
	else
		insttarget=install
		instdesc='an installation'
	fi
	temp_set_msg_current_stage "$instdesc process for $REINSTALL_ORIGPKGTAG $PROGRAM_STEP_COUNTER"
	message_target_relations "$REINSTALL_ORIGIN"
	message_echo "------- Starting $instdesc process --------"
	if [ $opt_dry_run = yes ]
	then
		message_dry_run
		temp_set_msg_current_stage "${_MSG_CURRENT_STAGE_general}"
		return
	fi
	if [ $opt_inst_by_pkg_if_can = yes ] && database_query_is_default_conf "$REINSTALL_ORIGIN"
	then
		message_echo "INFO: The configuration of this port is default, so use of a prebuilt package is attempted."
		if reinstall_chk_stage in_bypkg
		then
			message_echo "(Restarting the previously terminated (re)installation-by-package process...)"
		else
			reinstall_register_stage in_bypkg
		fi
		reinstall_pkg_backup
		reinstall_deinstall
		if ! reinstall_chk_stage_complete INSTALL_BY_PKG
		then
			if ! reinstall_chk_stage FAILED_INSTALL_BY_PKG
			then
				func_pkg_inst_remote=pkg_inst_remote
				[ $opt_use_legacy_pkg_for_missing_pkgng = yes ] && \
					func_pkg_inst_remote=pkg_inst_remote_wild
				if $func_pkg_inst_remote "$REINSTALL_NEWPKGNAME"
				then
					reinstall_register_stage_complete INSTALL_BY_PKG
				else
					reinstall_register_stage FAILED_INSTALL_BY_PKG
				fi
			fi
			if reinstall_chk_stage FAILED_INSTALL_BY_PKG
			then
				reinstall_failed_install_recover
			fi
		fi
		if reinstall_chk_stage_complete INSTALL_BY_PKG
		then
			touch "$REINSTALL_DBNODE_DIR/installed_timestamp"
			touch "$REINSTALL_DBNODE_DIR/installed_by_pkg"
			reinstall_deregister_stage in_bypkg
			reinstall_tell_update_to_depandents
			reinstall_closing_operations_after_successful_install
			message_echo
			return
		fi
		reinstall_deregister_stage in_bypkg
		message_echo "WARNING: (Re)installation-by-package is unsuccessful, so retrying by using port." >&2
	fi
	if reinstall_chk_stage in_build
	then
		message_echo "(Restarting the previously terminated build process...)"
		flag_restarted_build=yes
	else
		reinstall_restore_conflicts
		reinstall_register_stage in_build
		flag_restarted_build=no
	fi
	if [ -e "$REINSTALL_DBNODE_DIR/BEFOREBUILD.conf" ] && ! reinstall_chk_stage_complete BEFOREBUILD
	then
		message_echo "-- BEFOREBUILD operations (start)"
		sh -e "$REINSTALL_DBNODE_DIR/BEFOREBUILD.conf" || \
		{
			reinstall_restore_if_temporarily_deinstalled
			reinstall_errproc 'BEFOREBUILD operations'
			temp_set_msg_current_stage "${_MSG_CURRENT_STAGE_general}"
			return
		}
		message_echo "-- BEFOREBUILD operations (end)"
		reinstall_register_stage_complete BEFOREBUILD
	fi
	if ! reinstall_chk_stage_complete CLEAN_BEFORE_BUILD
	then
		reinstall_make_anymode clean || \
		{
			reinstall_restore_if_temporarily_deinstalled
			reinstall_errproc 'clean before build'
			temp_set_msg_current_stage "${_MSG_CURRENT_STAGE_general}"
			return
		}
		message_echo
		reinstall_register_stage_complete CLEAN_BEFORE_BUILD
	fi
	if ! reinstall_chk_stage_complete FETCH
	then
		if ! reinstall_chk_stage FAILED_FETCH
		then
			reinstall_make_anymode checksum || \
				reinstall_register_stage FAILED_FETCH
		fi
		if reinstall_chk_stage FAILED_FETCH
		then
			message_echo "INFO: Refetching distfiles for $REINSTALL_ORIGPKGTAG."
			if ! reinstall_chk_stage_complete FAILED_FETCH.REFETCH
			then
				if ! reinstall_make_anymode fetch FETCH_ARGS=-Ap
				then
					reinstall_restore_conflicts "$REINSTALL_ORIGIN"
					reinstall_restore_if_temporarily_deinstalled
					reinstall_errproc 'fetch'
					temp_set_msg_current_stage "${_MSG_CURRENT_STAGE_general}"
					return
				fi
				reinstall_make_anymode checksum || \
				{
					reinstall_restore_conflicts "$REINSTALL_ORIGIN"
					reinstall_restore_if_temporarily_deinstalled
					reinstall_errproc 'checksum'
					temp_set_msg_current_stage "${_MSG_CURRENT_STAGE_general}"
					return
				}
				reinstall_register_stage_complete FAILED_FETCH.REFETCH
			fi
		fi
		reinstall_register_stage_complete FETCH
	fi
	if [ $opt_batch_ports_only = yes -o $opt_interactive_ports_only = yes ]
	then
		to_skip=no
		case `database_query_get_makevar_val "$REINSTALL_ORIGIN" IS_INTERACTIVE` in
			yes)	msg_is_interactive='interactive'
				[ $opt_batch_ports_only = yes ] && to_skip=yes
				;;
			'')	msg_is_interactive='not interactive'
				[ $opt_interactive_ports_only = yes ] && to_skip=yes
				;;
		esac
		if [ $to_skip = yes ]
		then
			reinstall_restore_if_temporarily_deinstalled
			database_build_update_pkgname "$REINSTALL_ORIGIN" > /dev/null
			message_echo "INFO: Further processes for this port are skipped because it is $msg_is_interactive."
			message_echo
			fileedit_rm_a_line "$REINSTALL_ORIGIN" "${DBDIR}/stage.loop_list/reinst_todo.remain"
			temp_set_msg_current_stage "${_MSG_CURRENT_STAGE_general}"
			return
		fi
	fi
	if ! reinstall_chk_stage_complete BUILD
	then
		if ! reinstall_make build
		then
			if pkgsys_get_conflicting_pkgs build "$REINSTALL_ORIGIN" \
				> ${TMPDIR}/reinstall_exec::conflicts_build
			then
				reinstall_backup_and_delete_conflicts < ${TMPDIR}/reinstall_exec::conflicts_build
			elif [ $flag_restarted_build = no ]
			then
				reinstall_restore_conflicts "$REINSTALL_ORIGIN"
				reinstall_restore_if_temporarily_deinstalled
				reinstall_errproc 'build'
				temp_set_msg_current_stage "${_MSG_CURRENT_STAGE_general}"
				return
			else
				message_echo "INFO: This failure may be due to restarting from a terminated build."
			fi
			reinstall_deregister_stage_complete CLEAN_BEFORE_BUILD
			message_echo "INFO: Retrying the build process after cleaning for $REINSTALL_ORIGPKGTAG."
			reinstall_make clean || \
			{
				reinstall_restore_conflicts "$REINSTALL_ORIGIN"
				reinstall_restore_if_temporarily_deinstalled
				reinstall_errproc 'build and clean after build failure'
				temp_set_msg_current_stage "${_MSG_CURRENT_STAGE_general}"
				return
			}
			reinstall_make build || \
			{
				reinstall_restore_conflicts "$REINSTALL_ORIGIN"
				reinstall_restore_if_temporarily_deinstalled
				reinstall_errproc 'retrial of build after failure'
				temp_set_msg_current_stage "${_MSG_CURRENT_STAGE_general}"
				return
			}
		fi
		reinstall_register_stage_complete BUILD
	fi
	reinstall_deregister_stage in_build
	if reinstall_chk_stage in_instal
	then
		message_echo "(Restarting the previously terminated installation process...)"
	else
		reinstall_register_stage in_install
	fi
	reinstall_pkg_backup
	if [ -e "$REINSTALL_DBNODE_DIR/BEFOREDEINSTALL.conf" ] && ! reinstall_chk_stage_complete BEFOREDEINSTALL
	then
		if [ -n "$REINSTALL_CURRENTPKG" ]
		then
			message_echo "-- BEFOREDEINSTALL operations (start)"
			sh -e "$REINSTALL_DBNODE_DIR/BEFOREDEINSTALL.conf" || \
			{
				reinstall_restore_conflicts "$REINSTALL_ORIGIN"
				reinstall_restore_if_temporarily_deinstalled
				reinstall_errproc 'BEFOREDEINSTALL operations'
				temp_set_msg_current_stage "${_MSG_CURRENT_STAGE_general}"
				return
			}
			message_echo "-- BEFOREDEINSTALL operations (end)"
		fi
		reinstall_register_stage_complete BEFOREDEINSTALL
	fi
	reinstall_deinstall
	if ! reinstall_chk_stage_complete INSTALL
	then
		if ! reinstall_chk_stage FAILED_INSTALL
		then
			if reinstall_make $insttarget || \
				{
					message_echo "INFO: Cleaning up for retrial."
					reinstall_make deinstall \
						|| message_echo "WARNING: Continuing by hoping a success." >&2
					message_echo "INFO: Retrying the installation."
					reinstall_make $insttarget
				}
			then
				touch "$REINSTALL_DBNODE_DIR/installed_timestamp"
				reinstall_register_stage_complete INSTALL
			elif pkgsys_get_conflicting_pkgs install "$REINSTALL_ORIGIN" \
				> ${TMPDIR}/reinstall_exec::conflicts_install
			then
				reinstall_backup_and_delete_conflicts < ${TMPDIR}/reinstall_exec::conflicts_install
				message_echo "INFO: Retrying to the installation process for $REINSTALL_ORIGPKGTAG."
				if reinstall_make $insttarget
				then
					reinstall_register_stage_complete INSTALL
				else
					reinstall_register_stage FAILED_INSTALL
				fi
			else
				reinstall_register_stage FAILED_INSTALL
			fi
		fi
		if reinstall_chk_stage FAILED_INSTALL
		then
			reinstall_failed_install_recover
			if [ -e "$REINSTALL_DBNODE_DIR/AFTERINSTALL.conf" ] && \
				! reinstall_chk_stage_complete FAILED_INSTALL.AFTERINSTALL
			then
				if [ -n "$REINSTALL_CURRENTPKG" ]
				then
					message_echo "-- AFTERINSTALL operations (start)"
					sh -e "$REINSTALL_DBNODE_DIR/AFTERINSTALL.conf" || \
					{
						message_echo "WARNING: Failed in AFTERINSTALL operations." >&2
						message_echo "---- (The process is continued anyway)"
					}
					message_echo "-- AFTERINSTALL operations (end)"
				fi
				reinstall_register_stage_complete FAILED_INSTALL.AFTERINSTALL
			fi
			reinstall_restore_conflicts "$REINSTALL_ORIGIN"
			reinstall_restore_if_temporarily_deinstalled
			reinstall_errproc 'install'
			temp_set_msg_current_stage "${_MSG_CURRENT_STAGE_general}"
			return
		fi
	fi
	reinstall_restore_if_temporarily_deinstalled
	if [ -e "$REINSTALL_DBNODE_DIR/AFTERINSTALL.conf" ] && ! reinstall_chk_stage_complete AFTERINSTALL
	then
		message_echo "-- AFTERINSTALL operations (start)"
		sh -e "$REINSTALL_DBNODE_DIR/AFTERINSTALL.conf" || \
		{
			reinstall_restore_conflicts "$REINSTALL_ORIGIN"
			reinstall_errproc 'AFTERINSTALL operations'
			temp_set_msg_current_stage "${_MSG_CURRENT_STAGE_general}"
			return
		}
		message_echo "-- AFTERINSTALL operations (end)"
		reinstall_register_stage_complete AFTERINSTALL
	fi
	reinstall_restore_conflicts "$REINSTALL_ORIGIN"
	if ! reinstall_chk_stage_complete CLEAN_AFTER_INSTALL
	then
		reinstall_make clean || \
		{
			message_echo "WARNING: Failed to clean $REINSTALL_ORIGPKGTAG." >&2
		}
		reinstall_register_stage_complete CLEAN_AFTER_INSTALL
	fi
	reinstall_tell_update_to_depandents
	reinstall_closing_operations_after_successful_install
	message_echo
	)
}
