#!/bin/sh -e
# ==============================================================================
# portsreinstall library script
# - Wrappers for hiding version differences in the Ports/Packages system -
# Copyright (C) 2013 Mamoru Sakaue, MwGhennndo, All Rights Reserved.
# This software is distributed under the 2-Clause BSD License.
# ==============================================================================

# ============= Variables =============
PKGSYS_USE_PKGNG=no	# no: legacy pkg_* tools, yes: the new generation package (pkgng)
PKGSYS_CMD_PKG_INFO='pkg_info'	# Corresponding command for pkg_info
PKGSYS_CMD_PKG_CREATE='pkg_create'	# Corresponding command for pkg_create
PKGSYS_CMD_PKG_DELETE='pkg_delete'	# Corresponding command for pkg_delete
PKGSYS_CMD_PKG_ADD='pkg_add'	# Corresponding command for pkg_add
PKGSYS_AVR_REFETCH_TIMES_PER_SITE=1	# Average number (integer) of retrials for retrieving package or distfiles per mirror site
PKGSYS_AVR_REFETCH_TIMES_FOR_CHKSUMERR=2	#  Number (integer) of retrials for check sum error in retrieving a package

# ============= System defined value for the ports/packages =============
pkgsys_sysvar ()
{
	local var tmp_work
	var=$1
	tmp_work=${TMPDIR}/pkgsys_sysvar:work
	rm -rf "$tmp_work"
	mkdir "$tmp_work"
	make -C "$tmp_work" -f "${PORTSDIR}/Mk/bsd.port.mk" -V "$var"
}

# ============= Get the file name of package check sum file =============
pkgsys_pkgchksum_file ()
{
	echo CHECKSUM.MD5
}

# ============= Check whether a port is indispensable for the standard function of the ports/packages system =============
pkgsys_is_pkgtool ()
{
	case $1 in
	ports-mgmt/pkg|ports-mgmt/dialog4ports)
		;;
	*)	return 1;;
	esac
}

# ============= Check whether the dialog for selecting port options is dialog4ports =============
pkgsys_is_dialog4ports_used ()
{
	[ -n "`pkgsys_sysvar DIALOG4PORTS`" ]
}

# ============= Get the number of mirror sites for legacy packages =============
pkgsys_num_mirrorsites ()
{
	local siteroots
	siteroots=$1
	echo "$siteroots" | tr '|' '\n' | wc -l
}

# ============= Get a URL one of mirror sites =============
pkgsys_url_from_mirrors ()
{
	local siteroots subdir nsites id_site site platform version
	siteroots=$1
	subdir=$2
	nsites=`pkgsys_num_mirrorsites "$siteroots"`
	id_site=`(set +e; random -e $nsites; echo $?)`
	site=`echo "$siteroots" | tr '|' '\n' | sed -n $((${id_site}+1))p`
	platform=`uname -p`
	version=`uname -r | cut -d - -f 1,2 | tr [:upper:] [:lower:]`
	echo -n "$site"
	printf "$subdir\n" "$platform" "$version"
}

# ============= Fetch a file from one of mirror sites =============
pkgsys_fetch_from_mirrors ()
{
	local siteroots subdir filename dst tmp_work fetch itrial origdir url
	siteroots=$1
	subdir=$2
	filename=$3
	dst=$4
	tmp_work=${TMPDIR}/pkgsys_fetch_from_mirrors:work
	rm -rf "$tmp_work"
	mkdir "$tmp_work"
	fetch=`pkgsys_sysvar FETCH_CMD`
	itrial=$((`pkgsys_num_mirrorsites "$siteroots"`*$PKGSYS_AVR_REFETCH_TIMES_PER_SITE))
	origdir=`pwd`
	cd "$tmp_work"
	while [ $itrial -ge 1 ]
	do
		url=`pkgsys_url_from_mirrors "$siteroots" "$subdir"`$filename
		message_echo "INFO: Fetching from $url:"
		$fetch "$url"&& break
		rm -f "$filename"
		itrial=$(($itrial-1))
	done
	cd "$origdir"
	[ -e "$tmp_work/$filename" ] || return
	mv "$tmp_work/$filename" "$dst"
}

# ============= Get the package check sums file ready =============
pkgsys_ready_checksum_file ()
{
	local chksumfile
	tmp_savedpath=${TMPDIR}/pkgsys_ready_checksum_file:savedpath
	rm -f "$tmp_savedpath"
	chksumfile=`pkgsys_pkgchksum_file`
	if [ ! -e "${DBDIR}/checksum/$chksumfile" ]
	then
		[ -d "${DBDIR}/checksum" ] || mkdir "${DBDIR}/checksum"
		if ! pkgsys_fetch_from_mirrors "${PACKAGECHECKSUMROOTS}" "${PACKAGECHECKSUMDIR}" \
			"$chksumfile" "${DBDIR}/checksum"
		then
			message_echo "WARNING: No check sum file is available." >&2
			return 1
		fi
	fi
	echo "${DBDIR}/checksum/$chksumfile" > $tmp_savedpath
}

# ============= Get the location of a check sums file fetched by pkgsys_ready_checksum_file =============
pkgsys_ready_checksum_file__fetched_file ()
{
	cat "${TMPDIR}/pkgsys_ready_checksum_file:savedpath"
}

# ============= Fetch a legacy package from one of remote sites =============
pkgsys_fetch_legacy_remote ()
{
	local pkg tmp_work tmp_pkgpath pkg_regexp checksumpath validMD5 fetchedMD5 needs_fetch itrial
	pkg=$1
	tmp_work=${TMPDIR}/pkgsys_fetch_legacy_remote:work
	tmp_pkgpath=${TMPDIR}/pkgsys_fetch_legacy_remote:pkgpath
	rm -rf "$tmp_work"
	mkdir "$tmp_work"
	pkg_regexp=`str_escape_regexp "$pkg"`
	pkgsys_ready_checksum_file || return
	checksumpath=`pkgsys_ready_checksum_file__fetched_file`
	validMD5=`grep -m 1 -E -e "^MD5[[:space:]]*\($pkg_regexp\.tbz\)[[:space:]]*=" "$checksumpath" | sed -E "s/^[^=]*=[[:space:]]*(.*)/\1/"`
	if [ -z "$validMD5" ]
	then
		message_echo "WARNING: No check sum for $pkg.tbz." >&2
		return 1
	fi
	needs_fetch=yes
	[ -d "${PKGREPOSITORY}" ] || mkdir -p "${PKGREPOSITORY}"
	if [ -e "${PKGREPOSITORY}/$pkg.tbz" ]
	then
		fetchedMD5=`md5 "${PKGREPOSITORY}/$pkg.tbz" | sed -E "s/^[^=]*=[[:space:]]*(.*)/\1/"`
		if [ "x$fetchedMD5" = "x$validMD5" ]
		then
			needs_fetch=no
		else
			mv "${PKGREPOSITORY}/$pkg.tbz" "${PKGREPOSITORY}/$pkg.md5=$fetchedMD5.tbz"
		fi
	fi
	if [ $needs_fetch = yes ]
	then
		itrial=$PKGSYS_AVR_REFETCH_TIMES_FOR_CHKSUMERR
		while [ $itrial -ge 1 ]
		do
			if pkgsys_fetch_from_mirrors "${PACKAGEROOTS}" "${PACKAGEDIR}" \
				"$pkg.tbz" "$tmp_work"
			then
				fetchedMD5=`md5 "$tmp_work/$pkg.tbz" | sed -E "s/^[^=]*=[[:space:]]*(.*)/\1/"`
				[ "x$fetchedMD5" = "x$validMD5" ] && break
				message_echo "WARNING: Check sum mismatches for $pkg.tbz." >&2
			fi
			itrial=$(($itrial-1))
		done
		[ $itrial -ge 1 ] || return
		mv "$tmp_work/$pkg.tbz" "${PKGREPOSITORY}"
	fi
	echo "${PKGREPOSITORY}/$pkg.tbz" > $tmp_pkgpath
}

# ============= Get the location of a package fetched by pkgsys_fetch_legacy_remote =============
pkgsys_fetch_legacy_remote__fetched_pkg ()
{
	cat "${TMPDIR}/pkgsys_fetch_legacy_remote:pkgpath"
}

# ============= Check whether the dependency of a legacy package is the latest =============
pkg_is_dependency_of_a_legacypkg_latest ()
{
	local pkgarc tmp_extract tmp_contents tmp_origin tmp_pkg pkg nlines iline origin_req pkg_req pkg_new
	pkgarc=$1
	tmp_extract=${TMPDIR}/pkgng:pkg_is_dependency_of_a_legacypkg_latest:extract
	tmp_contents=${TMPDIR}/pkgng:pkg_is_dependency_of_a_legacypkg_latest:contents
	tmp_origin=${TMPDIR}/pkgng:pkg_is_dependency_of_a_legacypkg_latest:origin
	tmp_pkg=${TMPDIR}/pkgng:pkg_is_dependency_of_a_legacypkg_latest:pkg
	pkg=`pkgsys_pkgarc_to_pkgname "$pkgarc"`
	[ -e "$pkgarc" ] || return
	rm -rf "$tmp_extract"
	mkdir "$tmp_extract"
	tar xf "$pkgarc" -C "$tmp_extract" +CONTENTS
	grep -e '^@pkgdep[[:space:]]' -e '^@comment[[:space:]]*DEPORIGIN:' "$tmp_extract/+CONTENTS" \
		| sed 's/^@pkgdep[[:space:]]*//;s/^@comment[[:space:]]*DEPORIGIN://' > $tmp_contents
	nlines=`wc -l < $tmp_contents`
	iline=1
	while [ $iline -le $nlines ]
	do
		origin_req=`sed -n ${iline}p "$tmp_contents"`
		pkg_req=`sed -n $(($iline+1))p "$tmp_contents"`
		iline=$(($iline+2))
		pkg_new=`cat "${DBDIR}/requires/$origin_req/new_version" 2> /dev/null` || :
		if [ -z "$pkg_new" -o "$pkg_new" != "$pkg_req" ]
		then
			message_echo "WARNING: Requirements of remote package $pkg are not the latest." >&2
			return 1
		fi
	done
	:
}

# ============= Define wrapper commands for hiding the differences between pkg_* tools and pkgng =============
pkgsys_def_pkgtools ()
{
	if [ "${DBDIR}/WITH_PKGNG" -nt /etc/make.conf ]
	then
		PKGSYS_USE_PKGNG=`cat "${DBDIR}/WITH_PKGNG"`
	else
		PKGSYS_USE_PKGNG=`pkgsys_sysvar WITH_PKGNG | tr '[:upper:]' '[:lower:]'`
		if [ -d "${DBDIR}" ]
		then
			echo "$PKGSYS_USE_PKGNG" > ${DBDIR}/WITH_PKGNG.tmp
			mv "${DBDIR}/WITH_PKGNG.tmp" "${DBDIR}/WITH_PKGNG"
		fi
	fi
	if [ "x$PKGSYS_USE_PKGNG" = xyes ]
	then
		export WITH_PKGNG=yes
		PKGSYS_CMD_PKG_INFO='pkg info'
		PKGSYS_CMD_PKG_CREATE='pkg create'
		PKGSYS_CMD_PKG_DELETE='pkg delete'
		PKGSYS_CMD_PKG_ADD='pkg add'
		pkg_is_tool_available ()
		{
			which -a pkg | grep -v '^/usr/sbin/pkg$' | grep -m 1 '/sbin/pkg$' > /dev/null 2>&1
		}
		pkg_info_qoa ()
		{
			pkg info -qoa 2> /dev/null
		}
# 		pkg_info_qox ()
# 		{
# 			pkg info -qox "$@" 2> /dev/null
# 		}
		pkg_info_qoX ()
		{
			pkg info -qoX "$@" 2> /dev/null
		}
		pkg_info_qO ()
		{
			pkg info -qO "$@" 2> /dev/null
		}
		pkg_info_qo ()
		{
			pkg info -qo "$@" 2> /dev/null
		}
		pkg_info_qr ()
		{
			pkg info -qd "$@" 2> /dev/null
		}
		pkg_info_e ()
		{
			pkg info -e "$@" 2> /dev/null
		}
		pkg_info_eO ()
		{
			pkg info -eO "$1" 2> /dev/null
		}
		pkg_info_Eg ()
		{
			pkg info -Eg "$@" 2> /dev/null
		}
		pkg_info_qR ()
		{
			pkg info -qr "$@" 2> /dev/null
		}
		pkg_info_Ex ()
		{
			pkg info -Ex "$@" 2> /dev/null
		}
		pkg_info_gen_pkg_origin_table ()
		{
			pkg query -g '%n-%v\t%o' \* 2> /dev/null > ${DBDIR}/installed_ports:pkg_vs_origin.tbl
		}
		pkg_create_b ()
		{
			pkg create "$@"
		}
		pkg_delete_f ()
		{
			pkg delete -fqy "$@"
		}
		pkg_add_tools ()
		{
			local pkgarc tmp_extract prefix prefix_parent pkg
			pkgarc=$1
			tmp_extract=${TMPDIR}/pkgng:pkg_add_tools:extract
			rm -rf "$tmp_extract"
			mkdir "$tmp_extract"
			tar xf "$pkgarc" -C "$tmp_extract"
			prefix=`grep -m 1 '^prefix: ' "$tmp_extract/+MANIFEST" | sed 's/^prefix: *//'`
			prefix_parent=`dirname "$prefix"`
			cp -Rp "$tmp_extract/$prefix" "$prefix_parent"/
			pkg=`pkgsys_pkgarc_to_pkgname "$pkgarc"`
			message_echo "INFO: Contents of $pkg are temporarily installed by simple copy."
			if env ASSUME_ALWAYS_YES=YES pkg add "$pkgarc"
			then
				message_echo "INFO: $pkg is successfully registered."
			else
				message_echo "WARNING: Failed to register $pkg, but the process is continued." >&2
			fi
		}
		pkg_add_f ()
		{
			local pkgarc pkg pkg_tool pkg_gen
			pkg_tool=
			pkg_gen=
			for pkgarc in "$@"
			do
				pkg=`basename "$pkgarc"`
				if expr "$pkg" : '^pkg-[0-9][0-9]*\..*' > /dev/null
				then
					pkg_tool=$pkgarc
				else
					pkg_gen="$pkg_gen $pkgarc"
				fi
			done
			[ -n "$pkg_tool" ] && pkg_add_tools "$pkg_tool"
			[ -n "$pkg_gen" ] && env ASSUME_ALWAYS_YES=YES pkg add $pkg_gen
		}
		pkg_add_fF ()
		{
			pkg_add_f "$@"
		}
		pkg_inst_remote ()
		{
			local pkg pkgarc
			pkg=$1
			tmp_extract=${TMPDIR}/pkgng:pkg_inst_remote:extract
			pkg fetch -yU "$pkg" || return
			pkgarc=`pkgsys_pkgname_to_pkgarc "${PKGNG_PKG_CACHEDIR}/All" "$pkg"` || return
			rm -rf "$tmp_extract"
			mkdir "$tmp_extract"
			tar xf "$pkgarc" -C "$tmp_extract" +MANIFEST
			sed -E '1,/^deps:/d;/^[^[:space:]]/,$d;s/^[[:space:]]*([^:]+):[[:space:]]*\{origin:[[:space:]]*([^,]+),[[:space:]]*version:[[:space:]]*([^}]+)\}/\2\\\1-\3/' "$tmp_extract/+MANIFEST" \
				| tr '\\' '\t' | while read origin_req pkg_req
			do
				pkg_new=`cat "${DBDIR}/requires/$origin_req/new_version" 2> /dev/null` || :
				if [ -z "$pkg_new" -o "$pkg_new" != "$pkg_req" ]
				then
					message_echo "WARNING: Requirements of remote package $pkg are not latest." >&2
					return 1
				fi
			done
			env ASSUME_ALWAYS_YES=YES pkg add "$pkgarc"
		}
		pkg_inst_remote_wild ()
		{
			local pkg mode pkgarc
			pkg=$1
			mode=$2
			if pkg_is_tool_available
			then
				pkg_inst_remote "$pkg" && return
			fi
			message_echo "INFO: Trying to use a legacy package and convert it to pkgng."
			pkgsys_fetch_legacy_remote "$pkg" || return
			pkgarc=`pkgsys_fetch_legacy_remote__fetched_pkg`
			if [ "x$mode" != xnodepschk ]
			then
				pkg_is_dependency_of_a_legacypkg_latest "$pkgarc" || return
			fi
			pkg_add -ifF "$pkgarc" || return
			message_echo "INFO: Trying to convert the installed legacy package to pkgng."
			pkg2ng || :
			message_echo "INFO: Checking whether the conversion is successful."
			pkg info -e "$pkg"
		}
		pkg_loadconf ()
		{
			local pkg_conf
			# Deafult configuration for pkg(1)
			PKGNG_PACKAGESITE='http://pkg.freebsd.org/${ABI}/latest'
			PKGNG_SRV_MIRRORS=YES
			PKGNG_PKG_DBDIR=/var/db/pkg
			PKGNG_PKG_CACHEDIR=/var/cache/pkg
			PKGNG_PORTSDIR=/usr/ports
			PKGNG_PUBKEY=/etc/ssl/pkg.conf
			PKGNG_HANDLE_RC_SCRIPTS=NO
			PKGNG_PKG_MULTIREPOS=NO
			PKGNG_ASSUME_ALWAYS_YES=NO
			PKGNG_SYSLOG=YES
			PKGNG_SHLIBS=NO
			PKGNG_AUTODEPS=NO
			PKGNG_PORTAUDIT_SITE='http=//portaudit.FreeBSD.org/auditfile.tbz'
			# Load configuration for pkg(1)
			pkg_conf=`pkg query %Fp pkg | grep '/etc/pkg\.conf\.sample$' | sed 's/\.sample$//'`
			grep -v -e '^[[:space:]]*#' -e '^[[:space:]]*$' "$pkg_conf" 2> /dev/null \
				| grep -e '^[[:space:]]*[A-Z0-9_]*[[:space:]]*:[[:space:]]*.*' \
				| while read srcline
			do
				var=`expr "$srcline" : '^[[:space:]]*\([A-Z0-9_]*\)[[:space:]]*:.*'` || :
				val=`expr "$srcline" : '^[[:space:]]*[A-Z0-9_]*[[:space:]]*:[[:space:]]*\(.*\)'` || :
				eval PKGNG_$var=\$val
				misc_get_all_vardefs | grep -E "^PKGNG_$var="
			done > ${TMPDIR}/pkgsys_def_pkgtools:pkg.conf.sh
			. "${TMPDIR}/pkgsys_def_pkgtools:pkg.conf.sh"
		}
		pkg_rescue_tools ()
		{
			local packagepath checksumpath pkgname
			packagepath=`pkgsys_get_backup_pkg ports-mgmt/pkg` && \
				pkg_add_tools "$packagepath" && return
			[ -x /usr/local/sbin/pkg ] && return
			message_echo "WARNING: WITH_PKGNG is set, but pkgng is missing. It is installed now." >&2
			[ -x /usr/sbin/pkg ] && /usr/sbin/pkg && return
			pkgsys_ready_checksum_file || return
			checksumpath=`pkgsys_ready_checksum_file__fetched_file`
			pkgname=`sed 's/^MD5[[:space:]]*(//;s/\.tbz)[[:space:]]*=[^=]*$//' "$checksumpath" \
				| grep -m 1 -E -e "^pkg-[0-9]"` || :
			[ -z "$pkgname" ] && pkg_inst_remote_wild "$pkgname" nodepschk && return
			make -C "${PORTSDIR}/ports-mgmt/pkg" install clean && return
			message_echo "ERROR: WITH_PKGNG is set, but pkgng is missing. It is installed now." >&2
			:
		}
		pkg_rescue_tools
		pkg_loadconf
	else
		unset WITH_PKGNG
		PKGSYS_USE_PKGNG=no
		PKGSYS_CMD_PKG_INFO='pkg_info'
		PKGSYS_CMD_PKG_CREATE='pkg_create'
		PKGSYS_CMD_PKG_DELETE='pkg_delete'
		PKGSYS_CMD_PKG_ADD='pkg_add'
		pkg_is_tool_available ()
		{
			:
		}
		pkg_info_qoa ()
		{
			pkg_info -qoa 2> /dev/null
		}
# 		pkg_info_qox ()
# 		{
# 			pkg_info -qox "$@" 2> /dev/null
# 		}
		pkg_info_qoX ()
		{
			pkg_info -qoX "$@" 2> /dev/null
		}
		pkg_info_qO ()
		{
			pkg_info -qO "$@" 2> /dev/null
		}
		pkg_info_qo ()
		{
			pkg_info -qo "$@" 2> /dev/null
		}
		pkg_info_qr ()
		{
			pkg_info -qr "$@" | grep '^@pkgdep ' | sed 's/^@pkgdep[[:space:]]*//' 2> /dev/null
		}
		pkg_info_e ()
		{
			pkg_info -e "$@" 2> /dev/null
		}
		pkg_info_eO ()
		{
			[ `pkg_info -qO "$1" 2> /dev/null | wc -l` -gt 0 ]
		}
		pkg_info_Eg ()
		{
			pkg_info -E "$@" 2> /dev/null
		}
		pkg_info_qR ()
		{
			pkg_info -qR "$@" | grep -v '^$' 2> /dev/null
		}
		pkg_info_Ex ()
		{
			pkg_info -Ex "$@" 2> /dev/null
		}
		pkg_info_gen_pkg_origin_table ()
		{
			pkg_info -aE 2> /dev/null | while read pkg
			do
				origin=`pkg_info -qo "$pkg" 2> /dev/null`
				printf '%s\t%s\n' "$pkg" "$origin"
			done > ${DBDIR}/installed_ports:pkg_vs_origin.tbl
		}
		pkg_create_b ()
		{
			pkg_create -b "$@"
		}
		pkg_delete_f ()
		{
			pkg_delete -f "$@"
		}
		pkg_add_f ()
		{
			pkg_add -if "$@"
		}
		pkg_add_fF ()
		{
			pkg_add -ifF "$@"
		}
		pkg_inst_remote ()
		{
			local pkg pkgarc
			pkg=$1
			pkgsys_fetch_legacy_remote "$pkg" || return
			pkgarc=`pkgsys_fetch_legacy_remote__fetched_pkg`
			pkg_is_dependency_of_a_legacypkg_latest "$pkgarc" || return
			pkg_add "$pkgarc" || return
		}
		pkg_inst_remote_wild ()
		{
			pkg_inst_remote "$1"
		}
		pkg_loadconf () { :; }
		pkg_rescue_tools () { :; }
	fi
}

# ============= Check existence of initially or currently installed package for an origin =============
pkgsys_pkg_info_eO ()
{
	local origin origin_regexp
	origin=$1
	origin_regexp=`str_escape_regexp "$origin"`
	cut -f 2 "${DBDIR}/installed_ports:pkg_vs_origin.tbl" 2> /dev/null \
		| grep -m 1 -E "^$origin_regexp$" > /dev/null && return
	pkg_info_eO "$origin"
}

# ============= Get the name of an initially installed package for an origin =============
pkgsys_pkg_info_qO_init ()
{
	local origin tmppkg origin_regexp npkgs
	origin=$1
	tmppkg=${TMPDIR}/pkgsys_pkg_info_qO_init::pkg
	origin_regexp=`str_escape_regexp "$origin"`
	{ sed -n -E "/[[:space:]]$origin_regexp$/p" "${DBDIR}/installed_ports:pkg_vs_origin.tbl" 2> /dev/null || :; } \
		| cut -f 1 > $tmppkg
	npkgs=`wc -l < $tmppkg`
	if [ $npkgs -gt 0 ]
	then
		cat "$tmppkg"
	else
		pkg_info_qO "$origin"
	fi
}

# ============= Get the package name of this utility =============
pkgsys_get_my_current_pkg ()
{
	pkg_info_Ex "${APPNAME}-[0-9].*"
}

# ============= Get the origin of this utility =============
pkgsys_get_my_origin ()
{
	pkg_info_qo "`pkgsys_get_my_current_pkg`"
}

# ============= Get the origin of an initially installed package by ambiguous matching =============
pkgsys_init_pkg_orig_by_ambiguous_matching ()
{
	local pkg origin tmporigin ambsuffix len_pkg pkg_regexp norigins
	pkg=$1
	origin=`pkg_info_qo "$pkg" || :`
	[ -n "$origin" ] && { echo "$origin"; return; }
	tmporigin=${TMPDIR}/pkgsys_init_pkg_orig_by_ambiguous_matching::origin
	ambsuffix=
	norigins=0
	len_pkg=`echo -n "$pkg" | wc -c`
	while [ $len_pkg -gt 0 ]
	do
		pkg_regexp=`str_escape_regexp "$pkg"`$ambsuffix
		grep -E "^${pkg_regexp}[[:space:]]" "${DBDIR}/installed_ports:pkg_vs_origin.tbl" 2> /dev/null \
			| cut -f 2 > $tmporigin
		norigins=`wc -l < $tmporigin`
		[ $norigins -gt 0 ] && break
		len_pkg=$(($len_pkg-1))
		pkg=`echo -n "$pkg" | head -c $len_pkg`
		ambsuffix='[a-zA-Z0-9.,_+-]*'
	done
	[ $norigins -eq 1 ] || return
	cat "$tmporigin"
}

# ============= A part of message indicating tools for showing concerned issues in UPDATING =============
pkgsys_show_pkg_updating_commands ()
{
	if [ "x$PKGSYS_USE_PKGNG" = xyes ]
	then
		if which -s pkg_updating
		then
			echo 'pkg-updating(8) or pkg_updating(1)'
		else
			echo 'pkg-updating(8)'
		fi
	elif which -s pkg_updating
	then
		echo 'pkg_updating(1)'
	fi
}

# ============= Evaluation of ports globs =============
pkgsys_eval_ports_glob ()
{
	local pkglist origlist
	pkglist=${DBDIR}/pkgsys_eval_ports_glob:pkg.lst
	origlist=${DBDIR}/pkgsys_eval_ports_glob:origin.lst
	if [ ! -r "$pkglist" ]
	then
		if touch "$pkglist" 2>/dev/null
		then
			rm "$pkglist"
		else
			pkglist=${TMPDIR}/pkgsys_eval_ports_glob:pkg.lst
		fi
	fi
	if [ ! -r "$origlist" ]
	then
		if touch "$origlist" 2>/dev/null
		then
			rm "$origlist"
		else
			origlist=${TMPDIR}/pkgsys_eval_ports_glob:origin.lst
		fi
	fi
	[ -f "$pkglist" ] \
		|| cut -d \| -f 1 "${PORTS_INDEX_DB}" > $pkglist
	[ -f "$origlist" ] \
		|| cut -d \| -f 2 "${PORTS_INDEX_DB}" \
		| sed -E "s/^`str_escape_regexp "${PORTSDIR}"`\///" > $origlist
	while [ $# -gt 0 ]
	do
		glob=$1
		shift
		expr "x$glob" : '^x-' > /dev/null 2>&1 && continue
		glob_regexp=`str_convert_portsglob_to_regexp_pattern "$glob"`
		if expr "$glob" : '[^/][^/]*\/[^/][^/]*$' > /dev/null 2>&1
		then
			grep -E "$glob_regexp" "$origlist" 2>&1 || :
			{
				pkg_info_qoa
				cut -f 2 "${DBDIR}/installed_ports:pkg_vs_origin.tbl" 2> /dev/null
			} | grep -E "$glob" 2>&1 || :
		else
			if expr "$glob" : '^[a-z][a-zA-Z0-9+-]*[a-zA-Z0-9+]$' > /dev/null 2>&1 && \
				[ `expr "$glob" : '.*-[0-9]' 2>&1` -eq 0 ]
			then
				glob_regexp2=`expr "$glob_regexp" : '\(.*\)\$$' 2>&1`'-[0-9]'
			else
				glob_regexp2=$glob_regexp
			fi
			grep -n -E "$glob_regexp2" "$pkglist" 2>&1 | cut -d : -f 1 \
				| while read index
			do
				sed -n ${index}p "$origlist"
			done || :
			glob_regexp2=`expr "$glob_regexp" : '\(.*\)\$$' 2>&1`'[[:space:]]'
			{ sed -n -E "/$glob_regexp2/p" "${DBDIR}/installed_ports:pkg_vs_origin.tbl" 2> /dev/null || :; } \
				| cut -f 2
			pkg_info_qoX "$glob_regexp" || :
		fi
	done | sort -u
}

# ============= Create a back-up package archive =============
pkgsys_create_backup_pkg ()
{
	local pkgname dstdir origin backup_pkg_old origin_regexp pkgname_ptn backup_pkg pkgpath
	pkgname=$1
	dstdir=$2
	rm -rf "${TMPDIR}"/package.tmp
	mkdir "${TMPDIR}"/package.tmp
	origin=`pkg_info_qo "$pkgname"`
	if backup_pkg_old=`pkgsys_get_backup_pkg "$origin"` \
		[ "$backup_pkg_old" -nt "${DBDIR}/requires/$origin/installed_timestamp" ]
	then
		echo "$backup_pkg_old"
		return
	fi
	if ( cd "${TMPDIR}"/package.tmp && pkg_create_b "$pkgname" > /dev/null )
	then
		pkgname_ptn=`str_escape_regexp "$pkgname"`
		backup_pkg=`ls "${TMPDIR}"/package.tmp | \
			grep -m 1 -E "^${pkgname_ptn}\.(txz|tbz|tgz|tar)$"` || :
	fi
	if [ -z "$backup_pkg" ]
	then
		message_echo "WARNING: Failed to create backup package for $pkgname." >&2
		return 1
	fi
	[ -d "$dstdir" ] || mkdir -p "$dstdir"
	mv "${TMPDIR}/package.tmp/$backup_pkg" "$dstdir"
	pkgpath=$dstdir/$backup_pkg
	origin_regexp=`str_escape_regexp "$origin"`
	cat "${DBDIR}/backup_pkgarcs.lst" 2> /dev/null | \
		while read origin_cur pkgpath_cur
		do
			[ "$pkgpath_cur" = "$pkgpath" ] && continue
			if [ "$origin_cur" = "$origin" ]
			then
				rm -f "$pkgpath_cur"
			else
				printf '%s\t%s\n' "$origin_cur" "$pkgpath_cur"
			fi
		done > ${DBDIR}/backup_pkgarcs.lst.tmp
	printf '%s\t%s\n' "$origin" "$pkgpath" >> ${DBDIR}/backup_pkgarcs.lst.tmp
	mv "${DBDIR}/backup_pkgarcs.lst.tmp" "${DBDIR}/backup_pkgarcs.lst"
	echo "$pkgpath"
}

# ============= Delete a back-up package archive for a port origin =============
pkgsys_delete_backup_pkg ()
{
	local origin origin_regexp
	origin=$1
	origin_regexp=`str_escape_regexp "$origin"`
	grep -E "^${origin_regexp}[[:space:]]" "${DBDIR}/backup_pkgarcs.lst" 2> /dev/null \
		| cut -f 2 | while read pkgpath_cur
		do
			rm -f "$pkgpath_cur"
		done
	grep -v -E "^${origin_regexp}[[:space:]]" "${DBDIR}/backup_pkgarcs.lst" \
		2> /dev/null > ${DBDIR}/backup_pkgarcs.lst.tmp || :
	mv "${DBDIR}/backup_pkgarcs.lst.tmp" "${DBDIR}/backup_pkgarcs.lst"
}

# ============= Get an existing package archive path for a port origin =============
pkgsys_get_backup_pkg ()
{
	local origin origin_regexp
	origin=$1
	tmpnewest=${TMPDIR}/pkgsys_get_backup_pkg::newest
	origin_regexp=`str_escape_regexp "$origin"`
	rm -f "$tmpnewest"
	grep -E "^${origin_regexp}[[:space:]]" "${DBDIR}/backup_pkgarcs.lst" 2> /dev/null \
		| cut -f 2 | while read pkgpath
	do
		pkgpath_newest=`cat "$tmpnewest" 2> /dev/null` || :
		[ -e "$pkgpath" ] || continue
		[ -z "$pkgpath_newest" -o "$pkgpath" -nt "$pkgpath_newest" ] || continue
		echo "$pkgpath" > $tmpnewest
	done
	cat "$tmpnewest" 2> /dev/null
}

# ============= Get a package name from a package archive file name =============
pkgsys_pkgarc_to_pkgname ()
{
	local pkgfile_path
	pkgfile_path=$1
	basename "$pkgfile_path" | sed -E 's/\.(txz|tbz|tgz|tar)$//'
}

# ============= Get the file name of an existing package archive for a package name =============
pkgsys_pkgname_to_pkgarc ()
{
	local pkgdir pkgname pkgname_ptn pkgnode
	pkgdir=$1
	pkgname=$2
	[ -n "$pkgname" ] || return 1
	[ -d "$pkgdir" ] || return 1
	if [ "x$PKGSYS_USE_PKGNG" = xyes ]
	then
		pkgname_ptn=`str_escape_regexp "$pkgname"`
		pkgnode=`ls "$pkgdir" 2> /dev/null | grep -m 1 -E "^${pkgname_ptn}\.(txz|tbz|tgz|tar)$"` :
	elif [ -e "$pkgdir/$pkgname.tbz" ]
	then
		pkgnode=$pkgname.tbz
	fi
	[ -n "$pkgnode" ] || return 1
	echo "$pkgdir/$pkgnode"
}

# ============= Get port origins matching a glob pattern even if nonexistent =============
pkgsys_eval_ports_glob_even_if_nonexistent ()
{
	local glob_pattern
	glob_pattern=$1
	{
		pkgsys_eval_ports_glob "$glob_pattern" 2> /dev/null || :
		echo "$glob_pattern" | grep '^[a-z][a-z]*/[a-zA-Z0-9_+-][a-zA-Z0-9_+-]*$' || :
	} | grep -v -e '^$' | sort -u
}

# ============= Evaluate glob patterns and add/remove non-existing/existing ones of them to/from a file =============
pkgsys_register_evaluated_globs ()
{
	local mode listpath dirpath tmp_evaluated
	mode=$1
	listpath=$2
	shift 2
	dirpath=`dirname "$listpath"`
	tmp_evaluated=${TMPDIR}/pkgsys_register_evaluated_globs:pkgsys_eval_ports_glob
	echo "$@" | sed -E 's/[ :]+/\
/g' | grep -v '^$' | sort -u | while read glob
	do
		pkgsys_eval_ports_glob "$glob" > $tmp_evaluated
		[ `wc -l < $tmp_evaluated` -ge 1 ] || \
		{
			message_echo "WARNING: No matching ports/package glob [$glob]." >&2
			continue
		}
		cat "$tmp_evaluated"
	done | while read origin
	do
		[ -d "$dirpath" ] || mkdir -p "$dirpath"
		case $mode in
		remove)	fileedit_rm_a_line "$origin" "$listpath";;
		add)	fileedit_add_a_line_if_new "$origin" "$listpath";;
		esac
	done
}

# ============= Evaluate glob patterns for installed packages =============
pkgsys_eval_installed_pkgs_globs ()
{
	local tmp_evaluated
	tmp_evaluated=${TMPDIR}/pkgsys_eval_installed_pkgs_globs:origins
	rm -f "$tmp_evaluated"
	pkgsys_register_evaluated_globs add "$tmp_evaluated" "$@"
	[ -e "$tmp_evaluated" ] || return 0
	while read origin
	do
		pkgsys_pkg_info_eO "$origin" || echo "$origin"
	done < $tmp_evaluated
}

# ============= Get conflicting packages of a port =============
pkgsys_get_conflicting_pkgs ()
{
	local origin mode conflicts
	mode=$1
	origin=$2
	conflicts=`database_query_get_makevar_val "$origin" CONFLICTS`
	case $mode in
	build)
		conflicts=$conflicts' '`database_query_get_makevar_val "$origin" CONFLICTS_BUILD`
		;;
	install)
		conflicts=$conflicts' '`database_query_get_makevar_val "$origin" CONFLICTS_INSTALL`
		;;
	esac
	pkg_info_Eg $conflicts > ${TMPDIR}/pkgsys_get_conflicting_pkgs::conflicts || :
	cat "${TMPDIR}/pkgsys_get_conflicting_pkgs::conflicts"
	[ `wc -l < ${TMPDIR}/pkgsys_get_conflicting_pkgs::conflicts` -gt 0 ]
}	
