<?php
/**
 * phpoot - template engine for php
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * @author Haruki Setoyama <haruki@planewave.org>
 * @copyright Copyright &copy; 2003-2004, Haruki SETOYAMA <haruki@planewave.org>
 * @license http://www.opensource.org/licenses/lgpl-license.php LGPL
 * @version 0.5 $Id: phpoot.php,v 1.33 2004/04/27 02:33:41 haruki Exp $
 * @link http://phpoot.sourceforge.jp/
 * @package phpoot
 */
/**
 * Requires PEAR
 */
require_once 'PEAR.php';

// Constants
/**
 * mark for compiled template that mean PHPOOT library has been included
 */
define('PHPOOT', true);
/**
 * directory where PHPOOT library is in
 */
@define('PHPOOT_DIR', dirname(__FILE__));
/**
 * extention added to compiled template
 */
@define('PHPOOT_EXT', '.php');
/**
 * error code
 */
define('PHPOOT_ERR_INTERNAL', 1);
define('PHPOOT_ERR_TEMPLATE', 2);
define('PHPOOT_ERR_COMPILE_DIR', 4);
/**
 * PHPOOT Main Class.
 *
 * @access public
 * @package phpoot
 * @subpackage main_interface
 */
class phpoot extends PEAR {

    /**
     * options
     * <pre>
     * Key                  Meaning. defalult.
     * 'force_compile'      when true, template will be compiled everytime. false.
     * 'check_timestamp'    when true, modified time of template will be checked. true.
     * 'debug'              when true, display debug information in escape. false.
     * 'mode'               mode for compiled files and directories. 0777.
     * 'tag''raw'           extention of data to be displayed as raw strings without htmlspecialchars()
     * </pre>
     * @var array
     */
    var $option;

    /**
     * path to directory where compiled files will be stored
     * @var string
     * @access private
     */
    var $_compiled_dir;

    /**
     * @access private
     */
    var $_compiled_dir_use = false;

    /**
     * path to template file
     * @var string
     * @access private
     */
    var $_template_file;

    /**
     * XML_HTMLSax parser object
     * @access private
     */
    var $_parser;

    /**
     * handler object for XML_HTMLSax parser
     * @access private
     */
    var $_handler;

    /**
     * constructor
     */
    function phpoot()
    {
        $this->option
            = array(
                'force_compile'   => false,
                'check_timestamp' => true,
                'debug'           => false,
                'mode'            => 0777,
                'flock'           => true,
                'tag'             => array(
                    'raw'           => array('_raw', '_html')
                )
            );
    }

    /**
     * set a path for directory where compiled files will be stored
     *
     * @param stirng $dirname
     * @return void
     */
    function setCompileDir($dirname)
    {
        if ($dirname === null || $dirname === false) {
            $this->_compiled_dir_use = false;
        } else {
            $this->_compiled_dir = $dirname;
            $this->_compiled_dir_use = true;
        }
    }

    /**
     * set a path for template file
     *
     * @param stirng $template
     * @return void
     */
    function setTemplateFile($template)
    {
        $this->_template_file = $template;
        if (! $this->_compiled_dir_use) {
            $this->_compiled_dir = dirname($template);
        }
    }

    /**
     * set option
     * for option for phpoot_handler, see phpoot_handler class in phpoot.parser.php
     * @param mixed $key    string of key for option, or array of keys for option
     * @param mixed $val    value of option
     * @return void
     */
    function setOption($key, $val)
    {
        if (is_array($key)) {
            $temp =& $this->option;
            foreach ($key as $onekey) {
                $temp =& $temp[$onekey];
            }
            $temp = $val;
        } else {
            $this->option[$key] = $val;
        }
    }

    /**
     * displays the template results
     *
     * @param array $data   model data
     * @return bool|PEARError
     */
    function display($data)
    {
        $compiled_file = $this->_compiled_file();
        if ($this->option['force_compile'] != false
        || file_exists($compiled_file) != true) {
            $err = $this->compile();
            if ($err !== true) {
                return $err;
            }
        }
        elseif ($this->option['check_timestamp'] != false
        && (int)$this->_template_update_time() > filemtime($compiled_file)) {
            $err = $this->compile();
            if ($err !== true) {
                return $err;
            }
        }

        $err = phpoot_display_compiled($compiled_file, $data, $this->option['debug']);
        if ($err !== true) {
            return $this->raiseError(
                    "Failed to run the compiled file '$compiled_file'. ",
                    PHPOOT_ERR_INTERNAL);
        }
        return true;
    }

    /**
     * gets the template results
     *
     * @param sting $output     variable for output
     * @param array $data       model data
     * @return bool|PEARError
     */
    function expand(&$output, $data)
    {
        ob_start();
        $ret = $this->display($data);
        $output = ob_get_contents();
        ob_end_clean();
        return $ret;
    }

    /**
     * compile template file
     *
     * @return bool
     */
    function compile()
    {
        $compiled_file = $this->_compiled_file();
        if (file_exists($compiled_file)) {
            if (! is_writable($compiled_file)) {
                return $this->raiseError(
                    "Compiled file '$compiled_file' is not rewritable. ",
                    PHPOOT_ERR_COMPILE_DIR);
            }
        }
        elseif(is_dir($this->_compiled_dir)) {
            if (! is_writable($this->_compiled_dir)) {
                $dir = $this->_compiled_dir;
                return $this->raiseError(
                    "Directory for compiled files '$dir' is not writable. ",
                    PHPOOT_ERR_COMPILE_DIR);
            }
        }
        else {
            if (! $this->_mkdirs($this->_compiled_dir)) {
                $dir = $this->_compiled_dir;
                return $this->raiseError(
                    "Failed to create the directory for compiled files '$dir'. ",
                    PHPOOT_ERR_COMPILE_DIR);
            }
        }

        // flock.
        $lock = $this->_lock();
        if ($lock !== true) { return $err; }

        //
        $template = $this->_get_template();
        if (PEAR::isError($template)) return $template;

        $parsed = $this->_parse($template);
        $parsed = "<?php if(! defined('PHPOOT')) exit(); ?>\n"
                    . $parsed . "<?php \$_check = true; ?>\n"; // <?

        if ($this->option['flock'] && $lock === true) {
            $tempf = $compiled_file.'.tmp';
        } else {
            $tempf = tempnam(dirname($compiled_file), 'ppt');
        }
        $fp_w = fopen($tempf, 'wb');
        if (! $fp_w) {
            $this->_unlock();
            return $this->raiseError(
                "Failed to open a temporary file '$tempf'. ",
                PHPOOT_ERR_INTERNAL);
        }
        $err = fwrite($fp_w, $parsed);
        if (! $err) {
            fclose($fp_w);
            unlink($tempf);
            $this->_unlock();
            return $this->raiseError(
                "Fail to write compiled file to a temporary file '$tempf'. ",
                PHPOOT_ERR_INTERNAL);
        }
        fclose($fp_w);
        if (file_exists($compiled_file)) {
            unlink($compiled_file);
        }
        $err = rename($tempf, $compiled_file);
        if (! $err) {
            if (file_exists($tempf)) {
                unlink($tempf);
            }
            $this->_unlock();
            return $this->raiseError(
                "Fail to rename compiled file to '$compiled_file'. ",
                PHPOOT_ERR_INTERNAL);
        }
        $this->_unlock();
        return true;
    }

    /**
     * file pointer for lock file
     * @access private
     */
    var $_lfp;

    /**
     * lock compiled file
     * @access private
     */
    function _lock()
    {
        if ($this->option['flock']) {
            $lock = $this->_compiled_file().'.lock';
            $this->_lfp = fopen($lock, 'wb');
            if (! $this->_lfp) {
                return $this->raiseError(
                    "Failed to open lockfile '$lock'. ",
                    PHPOOT_ERR_INTERNAL);
            }
            if (! flock($this->_lfp, LOCK_EX)) {
                fclose($this->_lfp);
                unset($this->_lfp);
                return $this->raiseError(
                    "Failed to flock '$lock'. It may be too busy to compile. ",
                    PHPOOT_ERR_INTERNAL);
            }
        }
        return true;
    }

    /**
     * unlock compiled file
     * @access private
     */
    function _unlock()
    {
        if ($this->option['flock'] && isset($this->_lfp)) {
            $lock = $this->_compiled_file().'.lock';
            fclose($this->_lfp);
            unlink($lock);
        }
        return true;
    }

    /**
     * remove compiled template file
     *
     * @return bool
     */
    function removeCompiled()
    {
        if (! isset($this->_template_file)) {
            return $this->raiseError(
                    "Template file is not set. ",
                    PHPOOT_ERR_TEMPLATE);
        }

        $compiled_file = $this->_compiled_file();
        if (! file_exists($compiled_file)) return false;

        if (! unlink($compiled_file)) {
            return $this->raiseError(
                    "Failed to remove the compiled file '$compiled_file'. ",
                    PHPOOT_ERR_COMPILE_DIR);
        }
        return true;
    }

    /**
     * get file name for compiled
     *
     * @access private
     * @return string
     */
    function _compiled_file()
    {
        return $this->_compiled_dir . '/'
            . basename($this->_template_file) . PHPOOT_EXT;
    }

    /**
     * parse template file
     *
     * @param  string $template
     * @access private
     * @return string parsed PHP code
     */
    function _parse($template)
    {
        if (!isset($this->_parser) || !isset($this->_handler)) {
            include_once PHPOOT_DIR . '/phpoot.parser.php';
            $this->_parser = &new XML_HTMLSax();
            $this->_handler = new phpoot_handler($this->option);
            $this->_parser->set_object($this->_handler);
            $this->_handler->configParser($this->_parser);
        }

        $this->_handler->init($template);
        $this->_parser->parse($template);
        return $this->_handler->getParsed();
    }

    /**
     * make directory
     *
     * @param stirng $path
     * @access private
     * @return bool
     */
    function _mkdirs($path)
    {
        if (is_dir($path)) return true;
        if (is_file($path)) return false;
        if (! $this->_mkdirs(dirname($path))) return false;
        return mkdir($path, $this->option['mode']);
    }

    /**
     * get update time of the template file
     *
     * @access private
     * @return int
     */
    function _template_update_time()
    {
        return filemtime($this->_template_file);
    }

    /**
     * check whether the template file is readable
     *
     * @access private
     * @return bool
     */
    function _template_readable()
    {
        if (is_readable($this->_template_file)) {
            return true;
        } else {
            $file = $this->_template_file;
            return $this->raiseError(
                    "Template file '$file' does not exists or is not readable. ",
                    PHPOOT_ERR_TEMPLATE);
        }
    }

    /**
     * get text of the template file
     *
     * @access private
     * @return string|false
     */
    function _get_template()
    {
        $fd = fopen($this->_template_file, 'rb');
        if (! $fd) {
            $file = $this->_template_file;
            return $this->raiseError("Fail to open the template file '$file'. ", PHPOOT_ERR_TEMPLATE);
        }
        $contents = fread ($fd, filesize($this->_template_file));
        fclose ($fd);
        return $contents;
    }
}


/**
 * displays the template results with compiled template
 *
 * @param stirng    $compiled_file  path of compiled template file
 * @param array     $_val0          model data
 * @param bool      $_error         if this displayes error or not
 * @access private
 * @return bool
 */
function phpoot_display_compiled($compiled_file, $_val0, $_error =false)
{
    $_check = false; // flag for parse error check
    include $compiled_file;
    return $_check;
}

// functions below used in compiled template file.

/**
 * get array of data for nested variabe tag
 *
 * @return array
 * @access private
 */
function phpoot_each($val, $var)
{
    $target = phpoot_val($val, $var);
    if (is_array($target) && isset($target[0])) {
        return $target;
    }
    return array($target);
}

/**
 * make scalar data to array
 * @access private
 */
function phpoot_to_array(&$scalar, $var)
{
    if (is_scalar($scalar)) {
        $arr[$var] = $scalar;
        $scalar = $arr;
    }
}

/**
 * returns string data for display
 * @return string
 * @access private
 */
function phpoot_string($val, $var, $error)
{
    if (is_string($val) || is_numeric($val)) {
        echo phpoot_htmlspecialchars($val);
    }
    elseif (is_array($val)
    && (is_string($val[$var]) || is_numeric($val[$var]))) {
        echo phpoot_htmlspecialchars($val[$var]);
    }
    else {
        echo ($error) ? "<!-- NON STRING DATA '$var' -->" : '';
    }
}

/**
 * htmlspecialchars() for phpoot
 * @return string
 * @access private
 */
function phpoot_htmlspecialchars($val)
{
    return str_replace('  ', '&nbsp;&nbsp;', htmlspecialchars($val));
}

/**
 * returns string data for display
 * @return string
 * @access private
 */
function phpoot_raw($val, $var, $error)
{
    if (is_string($val) || is_numeric($val)) {
        echo $val;
    }
    elseif (is_array($val)
    && (is_string($val[$var]) || is_numeric($val[$var]))) {
        echo $val[$var];
    }
    else {
        echo ($error) ? "<!-- NON STRING DATA '$var' -->" : '';
    }
}

/**
 * makes string from data for attribute
 * @return string
 * @access private
 */
function phpoot_attr($attr, $val, $var)
{
    $target = phpoot_val($val, $var);

    if ($target === true) {
        return " $attr=\"$attr\"";
    } elseif (is_string($target) || is_numeric($target)) {
        return ' ' . $attr . '="' . htmlspecialchars($target) . '"';
    } else {
        return '';
    }
}

/**
 * get value from $val with variable name
 * @return mixed
 * @access private
 */
function phpoot_val($val, $var)
{
    if (is_array($val)) {
        if (isset($val[$var])) {
            return $val[$var];
        }
    }
    if (is_object($val)) {
        if (is_callable(array($val, $var))) {
            return $val->$var();
        } else {
            return $val->$var;
        }
    }
    return false;
}
?>