<?php
/**
 * phpoot - template engine for php
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * @author Haruki Setoyama <haruki@planewave.org>
 * @copyright Copyright &copy; 2003-2004, Haruki SETOYAMA <haruki@planewave.org>
 * @license http://www.opensource.org/licenses/lgpl-license.php LGPL
 * @version 0.5- $Id: phpoot.parser.php,v 1.18 2004/04/27 02:34:00 haruki Exp $
 * @link http://phpoot.sourceforge.jp/
 * @package phpoot
 * @subpackage _parser
 */
/**
* Requires PEAR
*/
require_once 'PEAR.php';
/**
* XML_HTMLSax Package is Required
*/
require_once 'XML/XML_HTMLSax.php';

/**
* attribute name for Variable Tag
*/
@define('PHPOOT_VAR', 'var');

/**
* handler for HTMLSax
* parse of template is done with phpoot_parser classes actually
* @package phpoot
* @subpackage _parser
*/
class phpoot_handler {

    /**
     * option
     * <pre>
     * 'escape''remove'     when true, remove all the <! ... >
     * 'pi''remove'         when true, remove all the <? ... ?>
     * 'jasp''remove'       when true, remove all the <% ... %>
     * 'tag''empty'         names of empty tag which should be < /> style
     * 'tag''raw'           extention of model data with HTML.
     * 'tag''allow_no_brace'    variable name is normaly "{foo}" style. when true, allow just "foo"
     * </pre>
     * @var array
     */
    var $option;

    /**
     * template text
     * @var string
     * @access private
     */
    var $_template = '';

    /**
     * blanck phpoot_parser_tag object
     * @var array of phpoot_parser_tag
     * @access private
     */
    var $_tag_parsers = array();
    /**
     * @access private
     */
    var $_level = 0;
    /**
     * @access private
     */
    var $_stack = array();

    /**
     * constructor
     * @access public
     */
    function phpoot_handler(&$option, $parsers = array('num', 'variable'))
    {
        // option
        $default = array(
            'tag' => array(
                'empty' => array(
                    'br' => 1,
                    'param' => 1,
                    'hr' => 1,
                    'input' => 1,
                    'col' => 1,
                    'img' => 1,
                    'area' => 1,
                    'frame' => 1,
                    'meta' => 1,
                    'link' => 1,
                    'base' => 1,
                    'basefont' => 1,
                    'isindex' => 1
                ),
                'raw' => array(),
                'allow_no_brace' => false
            ),
            'data' => array(),
            'escape' => array(
                'remove' => false
            ),
            'pi' => array(
                'remove' => true
            ),
            'jasp' => array(
                'remove' => true
            ),
            '_merged' => 1
        );
        $this->option =& $option;
        if (! isset($this->option['_merged'])) {
            $this->option = array_merge_recursive($default, $this->option);
        }
    
        $this->_tag_parsers = array();
        foreach ($parsers as $parser) {
            $class = 'phpoot_parser_tag_' . $parser;
            $this->_tag_parsers[] = new $class();
        }
        $this->_tag_parsers[] = new phpoot_parser_tag_static();
    }

    /**
     * initialize before HTMLSax parse
     * @access public
     */
    function init(&$template)
    {
        // template
        $template = str_replace(array("\r\n", "\r"), array("\n", "\n"), $template);
        $this->_template =& $template;
        
        // init private prop.
        $this->_level = 0;
        $this->_error = array();
        $this->_stack = array();
        $this->_stack[0] = new phpoot_parser_root;
        
        // option
        $num = count($this->_tag_parsers);
        for ($i=0; $i<$num; $i++) {
            $this->_tag_parsers[$i]->setOption($this->option['tag']);
        } 
    }

    /**
     * set handler on HTMLSax parser
     * @access public
     */
    function configParser(&$parser)
    {
        $parser->set_element_handler('openHandler', 'closeHandler');
        $parser->set_data_handler('dataHandler');
        $parser->set_escape_handler('escapeHandler');
        $parser->set_pi_handler('piHandler');
        $parser->set_jasp_handler('jaspHandler');
        $parser->set_option('XML_OPTION_FULL_ESCAPES');
    }

    /**
     * get parsed template, that is PHP script
     * @access public
     */
    function getParsed()
    {
        while($this->_level > 0) {
            $this->_stack[$this->_level]->close(strlen($this->_template));
            $this->_level--;
        }

        return $this->_stack[$this->_level]->parse();
    }

    /**
     * handler method for tag open
     * @access public
     */
    function openHandler(&$parser, $name, $attrs)
    {
        $name = strtolower($name);
        $attrs = array_change_key_case($attrs, CASE_LOWER);
        $this->open_position = $parser->get_current_position();

        $i=0;
        while($this->_tag_parsers[$i]->accept($name, $attrs) === false){
            $i++;
        }

        $this->_level++;
        unset($this->_stack[$this->_level]);
        $this->_stack[$this->_level] = $this->_tag_parsers[$i]; // copy

        $this->_stack[$this->_level]->open(
            $parser->get_current_position()
            , $name
            , $attrs
            , $this->_stack[$this->_level -1]->info
        );

        $this->_stack[$this->_level -1]->data[] =& $this->_stack[$this->_level];
    }

    /**
     * handler method for tag close
     * @access public
     */
    function closeHandler(&$parser, $close_tag)
    {
        $close_tag = strtolower($close_tag);

        $this->_stack[$this->_level]->close($parser->get_current_position());

        if ($this->_level > 0) {
            $this->_level--;
            if ($this->_stack[$this->_level +1]->open_tag() != $close_tag) {
                $this->closeHandler($parser, $close_tag);
            }
        }
    }

    /**
     * handler method for data
     * @access public
     */
    function dataHandler(&$parser, $data)
    {
        $this->_stack[$this->_level]->data[]
            = new phpoot_parser_data($data, $this->option['data']);
    }

    /**
     * handler method for escape
     * @access public
     */
    function escapeHandler(&$parser, $data)
    {
        $this->_stack[$this->_level]->data[]
            = new phpoot_parser_escape($data, $this->option['escape']);
    }

    /**
     * handler method for pi
     * @access public
     */
    function piHandler(&$parser, $target, $data)
    {
        $this->_stack[$this->_level]->data[]
            = new phpoot_parser_pi($target, $data, $this->option['pi']);
    }

    /**
     * handler method for jasp
     * @access public
     */
    function jaspHandler(&$parser, $data)
    {
        $this->_stack[$this->_level]->data[]
            = new phpoot_parser_jasp($data, $this->option['jasp']);
    }
}

/**
* base class of template parser classes
* This class and its sub classes are accessed only by phpoot_handler class
* This class is also used as null object.
* @see phpoot_handler
* @package phpoot
* @subpackage _parser
*/
class phpoot_parser {
    /**
     * indent string at the beginning
     * @var string
     */
    var $indent = '';

    /**
     * line feed at the end
     * @var string
     */
    var $linefeed = '';

    /**
     * @access private
     */
    var $_data = '';

    /**
     * parse into PHP script
     * @return string
     */
    function parse()
    {
        return '';
    }

    /**
     * returns if this content needs indent and linefeed
     * @return string
     */
    function need_indent()
    {
        return false;
    }

    /**
     * returns indent for next contents
     * @return string
     */
    function indent_for_next()
    {
        return '';
    }

    /**
     * returns linefeed "\n" for previous contents
     * @return string
     */
    function linefeed_for_prev()
    {
        return '';
    }

    /**
     * returns if this content is empty ''
     * @return string
     */
    function is_empty()
    {
        return false;
    }

    /**
     * returns if this class can have alternative content(s)
     * @return string
     */
    function is_alternative()
    {
        return false;
    }
}

/**
* parser for data
* @package phpoot
*/
class phpoot_parser_data extends phpoot_parser {

    function phpoot_parser_data($data, $option) {
        $this->_data = $data;
    }

    function parse() {
        return $this->_data;
    }

    function indent_for_next() {
        if (preg_match('/^(.*\n)([\040\011]*)$/sD', $this->_data, $match)) {
            $this->_data = $match[1];
            return $match[2];
        } else {
            return '';
        }
    }

    function linefeed_for_prev() {
        if (substr($this->_data, 0, 1) == "\n") {
            $this->_data = substr($this->_data, 1);
            return "\n";
        } else {
            return '';
        }
    }

    function is_empty() {
        return ($this->_data == '');
    }

}

/**
* parser for escape
* @package phpoot
*/
class phpoot_parser_escape extends phpoot_parser {
    /**
     * @access private
     */
    var $_remove = false;

    function phpoot_parser_escape($data, $option) {
        $this->_data = $data;
        $this->_remove = (bool)$option['remove'];
    }

    function parse() {
        if (! $this->_remove) {
            return '<!' . $this->_data . '>';
        } else {
            return
                  "<?php if (\$_error) { ?>\n"
                . $this->indent
                . '<!-- escape removed -->'
                . $this->linefeed
                . "<?php } ?>\n";;
        }
    }

    function need_indent() {
        return $this->_remove;
    }
}

/**
* parser for pi
* @package phpoot
*/
class phpoot_parser_pi extends phpoot_parser {
    /**
     * @access private
     */
    var $_remove = false;
    /**
     * @access private
     */
    var $_target;

    function phpoot_parser_pi($target, $data, $option) {
        $this->_target = $target;
        $this->_data = $data;
        $this->_remove = (bool)$option['remove'];
    }

    function parse() {
        if (! $this->_remove) {
            return '<?' . $this->_target . ' ' . $this->_data . '?>';
        } else {
            return
                 "<?php if (\$_error) { ?>\n"
                . $this->indent
                . '<!-- pi removed -->'
                . $this->linefeed
                . "<?php } ?>\n";
        }
    }

    function need_indent() {
        return $this->_remove;
    }
}

/**
* parser for jasp
* @package phpoot
*/
class phpoot_parser_jasp extends phpoot_parser {
    /**
     * @access private
     */
    var $_remove = false;

    function phpoot_parser_jasp($data, $option) {
        $this->_data = $data;
        $this->_remove = (bool)$option['remove'];
    }

    function parse() {
        if (! $this->_remove) {
            return '<%' . $this->_data . '%>';
        } else {
            return
                  "<?php if (\$_error) { ?>\n"
                . $this->indent
                . '<!-- jasp removed -->'
                . $this->linefeed
                . "<?php } ?>\n";
        }
    }

    function need_indent() {
        return $this->_remove;
    }
}

/**
 * Base of parser for tag
 * @package phpoot
 * @abstract
 */
class phpoot_parser_tag extends phpoot_parser {
    /**
     * data in the tag
     * @var phpoot_parser
     */
    var $data = '';

    /**
     * information about variable tag
     * @var array
     */
    var $info;

    /**
     * name of open tag
     * @access private
     */
    var $_open_tag;

    /**
     * attribute in the open tag
     * @var array
     * @access private
     */
    var $_attrs;

    /**
     * options
     * @var array
     * @access private
     */
    var $_option;

    /**
     * if this tag can have data or not
     * @var bool
     * @access private
     */
    var $_empty_tag = false;

    /**
     * position when tag is opend
     * @access private
     */
    var $_position;

    /**
     * tags that will be deleted when no attribute
     * @var array
     */
    var $_delete = array('span' => 1);

    /**
     * set option
     * @access public
     */
    function setOption($option)
    {
        $this->_option = $option;
    }

    /**
     * judge if this class accept the tag
     * @abstract
     * @access public
     * @return bool
     */
    function accept($open_tag, $attrs)
    {
        return false;
    }

    /**
     * initialize when the tag is opened
     * @abstract
     */
    function open($position, $name, $attrs, &$info) {}

    /**
     * initialize
     * @access private
     */
    function _open($position, $name, $attrs) {
        $this->_position = $position;
        $this->_open_tag = $name;
        $this->_attrs = $attrs;

        if (isset($option['empty'][$name])) {
            $this->_empty_tag = true;
        }

        $this->_init();
    }

    /**
     * initialize data stack
     * @access private
     */
    function _init() {
        $this->data = array();
        $this->data[] = new phpoot_parser;
    }

    /**
     * parse all the data in $this->data stack and add the results on $this->_data
     * @access public
     */
    function close($position) {
        if ($this->_position == $position) {
            $this->_empty_tag = true;
        }

        $this->data[] = new phpoot_parser;
        $num = count($this->data) -1;

        // get indent and linefeed for pritty print
        for ($i = $num-1; $i > 0; $i--) {
            if ($this->data[$i]->need_indent()) {
                $this->data[$i]->indent = $this->data[$i -1]->indent_for_next();
                $this->data[$i]->linefeed = $this->data[$i +1]->linefeed_for_prev();
            }
        }

        // altarnative & parse
        for ($i=1; $i < $num; $i++) {
            if ($this->data[$i]->is_alternative() == true
            && $this->data[$i+1]->is_empty()) {
                $alts = $this->_get_alternative($i, $num, $this->data[$i]);
                if ($alts > 0) {
                    $this->_data .= $this->_parse_alternative($i, $alts);
                    $i += $alts*2;
                    continue;
                }
            }
            $this->_data .= $this->data[$i]->parse();
        }

        $this->_init();
    }

    /**
     * get alternative tags
     * @access private
     * @return int  number of the alternatives
     */
    function _get_alternative($i, $num, &$phpoot_parser) {
        $open_tag = $phpoot_parser->open_tag();
        $variable = $phpoot_parser->info['name'];
        $alts = 0;
        $i += 2;
        while($i < $num){
            if ($this->data[$i]->is_alternative() == true
            && $this->data[$i]->open_tag() == $open_tag
            && $this->data[$i]->info['name'] == $variable) {
                $alts += 1;
                if (! $this->data[$i+1]->is_empty()) {
                    return $alts;
                }
                $i += 2;
            } else {
                return $alts;
            }
        } // while
        return $alts;
    }

    /**
     * parse the tag with alternative
     * @access private
     * @return string
     */
    function _parse_alternative($i, $alts)
    {
        $alts += 1;

        $level      = $this->data[$i]->info['level'];
        $_var       = $this->data[$i]->info['name'];
        $_each      = '$_each' . $level;
        $_valprev   = '$_val'  . ($level -1);
        $_valcurr   = '$_val'  . $level;
        $_num       = '$_num'  . $level;

        $parsed
            = '<?php '
            . "$_each = phpoot_each($_valprev, '$_var'); "
            . "$_num = 0; "
            . "foreach ($_each as $_valcurr) { ";

        for ($j=0; $j < $alts; $j++) {
            $parsed
                .= "if ($_num%$alts == $j) { "
                . $this->data[$i+$j*2]->parse_in_loop()
                . "continue; } ";
        }

        $parsed
            .= "} ?>\n";

        return $parsed;
    }

    /**
     * returns name of open tag
     * @return string
     */
    function open_tag()
    {
        return $this->_open_tag;
    }

    /**
     * parse tag and set $this->_parsed_close_tag and $this->_parsed_open_tag
     * @access private
     */
    function _parse_tag()
    {
        // open tag
        $open_tag = '<' . $this->_open_tag;
        $attr_variables = array();
        $attr_statics = 0;
        $_val = '$_val' . ($this->info['level'] -1);

        foreach ($this->_attrs as $key => $value) {
            if ($key == PHPOOT_VAR && !is_string($value)) {
                $_val = '$_val' . $this->info['level'];
                continue;
            }

            if (preg_match('/^\{([a-zA-Z_\x7f-\xff][a-zA-Z0-9_\x7f-\xff]*)\}$/D', $value, $match)) {
            // variable attribute, bar="{foo}" style
                $_varibale = $match[1];
                $open_tag .= "<?php echo phpoot_attr('$key', $_val, '$_varibale'); ?>\n";
                $attr_variables[] = "phpoot_attr('$key', $_val, '$_varibale')";
            }
            else {
            // static attribute
                $open_tag .= ' ' . $key . '=' . $this->_double_quote_html($value);
                $attr_statics++;
            }
        }

        if ($this->_empty_tag == true) {
            $open_tag  .= ' />';
        } else {
            $open_tag  .= '>';
        }

        // close tag
        $close_tag = $this->_indent_close_tag();
        if ($this->_empty_tag == true) {
            $close_tag  = '';
        } else {
            $close_tag .= '</' . $this->_open_tag . '>';
        }

        // deletable tag?
        if ($attr_statics == 0 && isset($this->_delete[$this->_open_tag])) {
            if (empty($attr_variables)) {
                $open_tag  = '';
                $close_tag = '';
            } else {
                $_condition = implode(" != '' || ", $attr_variables) . " != ''";
                $open_tag = "<?php if ($_condition) { ?>\n" . $open_tag . "<?php } ?>\n";
                if ($close_tag != '') {
                    $close_tag = "<?php if ($_condition) { ?>\n" . $close_tag . "<?php } ?>\n";
                }
            }
        }

        return array($this->indent . $open_tag, $close_tag . $this->linefeed);
    }

    /**
     * get indent for close tag
     * @access private
     * @return string
     */
    function _indent_close_tag() {
        // for static tag
        return '';
    }

    /**
     * double quote for html attribute
     * @access private
     * @return string
     */
    function _double_quote_html($str)
    {
        return '"' . str_replace(array('"', '<', '>'), array('&quot', '&lt;', '&gt;'), $str) . '"';
    }

}

/**
 * Parser for root level of the stack
 *
 * @package phpoot
 */
class phpoot_parser_root extends phpoot_parser_tag {

    var $info = array('level' => 0, 'nests' => false);

    function phpoot_parser_root() {
        $this->_init();
    }

    function close($position)
    {
        // todo: error
        return;
    }

    function parse()
    {
        parent::close(0);
        return $this->_data;
    }
}


/**
 * Parser for tags without variable attribute
 *
 * @package phpoot
 */
class phpoot_parser_tag_static extends phpoot_parser_tag {

    function accept($open_tag, $attrs)
    {
        return true;
    }

    function open($position, $name, $attrs, &$info)
    {
        $this->_open($position, $name, $attrs);
        $this->info =& $info;
    }

    function parse()
    {
        list($opentag, $closetag) = $this->_parse_tag();
        return $opentag . $this->_data . $closetag;
    }
}

/**
 * Parser for variable tag
 * variable tag means the one with var attribute.
 *
 * @package phpoot
 */
class phpoot_parser_tag_variable extends phpoot_parser_tag {

    function accept($open_tag, $attrs)
    {
        if (! isset($attrs[PHPOOT_VAR])) {
            return false;
        }
        if (preg_match('/^\{[a-zA-Z_\x7f-\xff][a-zA-Z0-9_\x7f-\xff]*\}$/D', $attrs[PHPOOT_VAR])) {
            return true;
        }
        if ($this->_option['allow_no_brace'] == true
        && preg_match('/^[a-zA-Z_\x7f-\xff][a-zA-Z0-9_\x7f-\xff]*$/D', $attrs[PHPOOT_VAR])) {
            return true;
        }
        return false;
    }

    function open($position, $name, $attrs, &$info)
    {
        $this->info['level'] = (int)$info['level'] +1;
        $this->info['nests'] = false;
        $info['nests'] = true;
        if (preg_match('/^\{([a-zA-Z_\x7f-\xff][a-zA-Z0-9_\x7f-\xff]*)\}$/D', $attrs[PHPOOT_VAR], $match)) {
            $this->info['name'] = $match[1];
        } else {
            $this->info['name'] = $attrs[PHPOOT_VAR];
        }

        $this->_open($position, $name, $attrs);

        $this->_attrs[PHPOOT_VAR] = true;
    }

    function parse()
    {
        $_var       = $this->info['name'];
        $_each      = '$_each' . $this->info['level'];
        $_valprev   = '$_val'  . ($this->info['level'] -1);
        $_valcurr   = '$_val'  . $this->info['level'];
        $_num       = '$_num'  . $this->info['level'];

        $parsed
            = '<?php '
            . "$_each = phpoot_each($_valprev, '$_var'); "
            . "$_num = 0; "
            . "foreach ($_each as $_valcurr) { "
            . $this->parse_in_loop()
            . "} ?>\n";

        return $parsed;
    }

    function parse_in_loop() {
        list($open_tag, $close_tag) = $this->_parse_tag();

        $_var       = $this->info['name'];
        $_each      = '$_each' . $this->info['level'];
        $_valcurr   = '$_val'  . $this->info['level'];
        $_num       = '$_num'  . $this->info['level'];

        $parsed
            = "if ($_valcurr === null || $_valcurr === false) { "
            .  "if (\$_error) { ?>\n"
            .  $this->indent
            .  "<!-- NO DATA for '$_var' -->"
            .  $this->linefeed
            .  "<?php } "
            .  "} else { $_num++; ";

        if (! $this->info['nests']) {
            $parsed
                .= "?>\n"
                .  $open_tag;

            $_func = 'phpoot_string';
            foreach ($this->_option['raw'] as $ext) {
                if (substr($_var, -strlen($ext)) == $ext) {
                    $_func = 'phpoot_raw';
                }
            }

            $parsed
                .= "<?php if ($_valcurr === true) { ?>\n"
                .  $this->_data
                .  "<?php } else { $_func($_valcurr, '$_var', \$_error); } ?>\n"
                .  $close_tag;
        } else {
            $parsed
                .= "phpoot_to_array($_valcurr, '$_var'); ?>\n"
                 . $open_tag
                 . $this->_data
                 . $close_tag;
        }

        $parsed
            .= '<?php } ';

        return $parsed;
    }

    function _indent_close_tag()
    {
        // indent before closetag
        if (preg_match('/^(.*\n)([\040\011]*)$/sD', $this->_data, $match)
        && substr($match[1], -3) != ' ?>') {
            $this->_data = $match[1];
            return $match[2];
        } else {
            return '';
        }
    }

    function need_indent()
    {
        return true;
    }

    function is_alternative()
    {
        return true;
    }
}

/**
 * Parser for tags with '#num' id attribute
 *
 * @package phpoot
 */
class phpoot_parser_tag_num extends phpoot_parser_tag_static {
    function accept($open_tag, $attrs)
    {
        if (@$attrs[PHPOOT_VAR] == '{#num}') {
                return true;
        }
        if ($this->_option['allow_no_brace'] == true
        && @$attrs[PHPOOT_VAR] == '#num') {
                return true;
        }
        return false;
    }

    function open($position, $name, $attrs, &$info)
    {
        parent::open($position, $name, $attrs, $info);
        unset($this->_attrs[PHPOOT_VAR]);
    }

    function parse()
    {
        list($open_tag, $close_tag) = $this->_parse_tag();

        $_num = '$_num' . $this->info['level'];
        return $open_tag . "<?php echo $_num; ?>\n" . $close_tag;
    }
}

?>