/*******************************************************************************
 * Copyright (C) 2018 OTK Software
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************/
package com.otk.application.util.draw;

import java.awt.Dimension;
import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.geom.AffineTransform;
import java.awt.image.BufferedImage;

import com.otk.application.util.ImageUtils;
import com.otk.application.util.MathUtils;

public class Mosaic {

	public static BufferedImage generate(IMosaicDescriptor descriptor) {
		Image image = descriptor.getInputImage();
		int imageWidth = image.getWidth(null);
		int imageHeight = image.getHeight(null);
		Dimension cellSize = getCellSize(descriptor);
		int cellSpacing = getCellSpacing(descriptor);
		Dimension mosaicSize = getMosaicSize(descriptor);
		double angleRadians = descriptor.getCellRotationDegrees() * Math.PI / 180;

		BufferedImage mosaic = new BufferedImage(mosaicSize.width, mosaicSize.height,
				ImageUtils.getAdaptedBufferedImageType());
		Graphics2D canvas = mosaic.createGraphics();
		AffineTransform initialTransform = canvas.getTransform();
		for (int col = 0; col < descriptor.getColumnCount(); col++) {
			for (int row = 0; row < descriptor.getRowCount(); row++) {
				if (!descriptor.isCellVisible(col, row)) {
					continue;
				}
				int x = cellSpacing + col * (cellSize.width + cellSpacing);
				int y = cellSpacing + row * (cellSize.height + cellSpacing);
				if (angleRadians != 0.0) {
					canvas.setTransform(initialTransform);
					canvas.translate(x, y);
					canvas.translate((cellSize.width - imageWidth) / 2, (cellSize.height - imageHeight) / 2);
					canvas.rotate(-angleRadians, imageWidth / 2, imageHeight / 2);
					canvas.translate(-x, -y);
				}
				canvas.drawImage(image, x, y, null);
			}
		}
		return mosaic;
	}

	public static int getCellSpacing(IMosaicDescriptor descriptor) {
		Image image = descriptor.getInputImage();
		int imageWidth = image.getWidth(null);
		int imageHeight = image.getHeight(null);
		return MathUtils.round(descriptor.getCellSpacingRatio() * (imageWidth + imageHeight) / 2.0);
	}

	public static Dimension getMosaicSize(IMosaicDescriptor descriptor) {
		Dimension cellSize = getCellSize(descriptor);
		int cellSpacing = getCellSpacing(descriptor);
		int mosaicWidth = descriptor.getColumnCount() * cellSize.width
				+ (descriptor.getColumnCount() + 1) * cellSpacing;
		int mosaicHeight = descriptor.getRowCount() * cellSize.height + (descriptor.getRowCount() + 1) * cellSpacing;
		return new Dimension(mosaicWidth, mosaicHeight);
	}

	public static Dimension getCellSize(IMosaicDescriptor descriptor) {
		Image image = descriptor.getInputImage();
		int imageWidth = image.getWidth(null);
		int imageHeight = image.getHeight(null);
		int cellWidth;
		int cellHeight;
		double angle = descriptor.getCellRotationDegrees() * Math.PI / 180;
		if (angle != 0.0) {
			double sin = Math.abs(Math.sin(angle)), cos = Math.abs(Math.cos(angle));
			int w = imageWidth;
			int h = imageHeight;
			cellWidth = (int) Math.floor(w * cos + h * sin);
			cellHeight = (int) Math.floor(h * cos + w * sin);
		} else {
			cellWidth = imageWidth;
			cellHeight = imageHeight;
		}
		return new Dimension(cellWidth, cellHeight);
	}

	public static interface IMosaicDescriptor {

		Image getInputImage();

		boolean isCellVisible(int col, int row);

		int getCellRotationDegrees();

		double getCellSpacingRatio();

		int getColumnCount();

		int getRowCount();

	}

}
