/*******************************************************************************
 * Copyright (C) 2018 OTK Software
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************/
package com.otk.application.util;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.charset.Charset;
import java.util.Arrays;
import java.util.Date;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;

import com.otk.application.error.AbstractApplicationError;
import com.otk.application.error.StandardError;
import com.otk.application.error.UnexpectedError;

public abstract class CommandLineInterface<T> {

	private String commandOptions;
	private File executableFile;
	private File workingDirectory;
	private Map<String, String> env;
	private int safeReadingTimeoutMilliseconds = 5000;

	private final Object STARTUP_MUTEX = new Object();
	private CommandExecutor commandExecutor;
	private Thread orchestrationThread;
	private boolean starting = false;
	private ByteArrayOutputStream outputBuffer = createOutputBuffer();
	private ByteArrayOutputStream errorBuffer = createErrorBuffer();
	private ExecutorService safeReadingExecutor;
	private T executionResult;
	private Throwable executionError;

	protected abstract T communicate(Process process);

	public CommandLineInterface(File workingDirectory, File executableFile, String commandOptions) {
		this.commandOptions = commandOptions;
		this.executableFile = executableFile;
		this.workingDirectory = workingDirectory;
	}

	public String getCommandOptions() {
		return commandOptions;
	}

	public void setCommandOptions(String commandOptions) {
		this.commandOptions = commandOptions;
	}

	public File getExecutableFile() {
		return executableFile;
	}

	public void setExecutableFile(File executableFile) {
		this.executableFile = executableFile;
	}

	public File getWorkingDirectory() {
		return workingDirectory;
	}

	public void setWorkingDirectory(File workingDirectory) {
		this.workingDirectory = workingDirectory;
	}

	public Map<String, String> getEnv() {
		return env;
	}

	public void setEnv(Map<String, String> env) {
		this.env = env;
	}

	public int getSafeReadingTimeoutMilliseconds() {
		return safeReadingTimeoutMilliseconds;
	}

	public void setSafeReadingTimeoutMilliseconds(int safeReadingTimeoutMilliseconds) {
		this.safeReadingTimeoutMilliseconds = safeReadingTimeoutMilliseconds;
	}

	protected ByteArrayOutputStream createOutputBuffer() {
		return new ByteArrayOutputStream();
	}

	protected ByteArrayOutputStream createErrorBuffer() {
		return new ByteArrayOutputStream();
	}

	protected ByteArrayOutputStream getOutputBuffer() {
		return outputBuffer;
	}

	protected ByteArrayOutputStream getErrorBuffer() {
		return errorBuffer;
	}

	public CommandExecutor getCommandExecutor() {
		return commandExecutor;
	}

	public ExecutorService getSafeReadingExecutor() {
		return safeReadingExecutor;
	}

	protected void handleOrchestrationError(final Throwable t) {
		executionError = t;
	}

	public boolean isExecuting() {
		if (starting) {
			return true;
		}
		if (orchestrationThread != null) {
			if (orchestrationThread.isAlive()) {
				return true;
			}
		}
		if (commandExecutor != null) {
			if (commandExecutor.isProcessAlive()) {
				return true;
			}
		}
		return false;
	}

	public T execute() {
		executeAsynchronously();
		waitForExecutionEnd();
		Throwable error = getExecutionError();
		if (error != null) {
			AbstractApplicationError.rethrow(error);
		}
		return executionResult;
	}

	public void executeAsynchronously() {
		synchronized (STARTUP_MUTEX) {
			if (isExecuting()) {
				throw new UnexpectedError("The command already executing");
			}
			starting = true;
			try {
				orchestrationThread = createOrchestrationThread();
				orchestrationThread.start();
			} finally {
				starting = false;
			}
		}
	}

	public void waitForExecutionEnd() {
		while (isExecuting()) {
			MiscUtils.relieveCPU();
		}
	}

	public void requestExecutionInterruption() {
		if (orchestrationThread != null) {
			orchestrationThread.interrupt();
		}
	}

	public T getExecutionResult() {
		return executionResult;
	}

	public Throwable getExecutionError() {
		return executionError;
	}

	protected CommandExecutor createCommandExecutor() {
		CommandExecutor result = new CommandExecutor();
		result.setCommandLine(CommandExecutor.quoteArgument(getExecutableFile().getPath()) + " " + getCommandOptions());
		result.setWorkingDir(getWorkingDirectory());
		result.setEnv(getEnv());
		return result;
	}

	protected Thread createOrchestrationThread() {
		return new Thread(
				MiscUtils.formatThreadName(CommandLineInterface.class, "Executor of " + CommandLineInterface.this)) {
			@Override
			public void run() {
				try {
					orchestrate();
				} catch (Throwable t) {
					handleOrchestrationError(t);
				}
			}
		};
	}

	protected void orchestrate() {
		commandExecutor = createCommandExecutor();
		commandExecutor.startProcess();
		safeReadingExecutor = createSafeReadingExecutor();
		try {
			executionResult = communicate(commandExecutor.getLaunchedProcess());
		} finally {
			safeReadingExecutor.shutdownNow();
			commandExecutor.disconnectProcess();
			if (commandExecutor.isProcessAlive()) {
				commandExecutor.killProcess();
			}
		}
	}

	protected ExecutorService createSafeReadingExecutor() {
		return Executors.newSingleThreadExecutor(new ThreadFactory() {
			@Override
			public Thread newThread(Runnable runnable) {
				return createSafeReadingThread(runnable);
			}
		});
	}

	protected Thread createSafeReadingThread(Runnable runnable) {
		Thread result = new Thread(runnable, MiscUtils.formatThreadName(CommandLineInterface.class, "Safe Reader"));
		return result;
	}

	protected int acquireSafelyOutputDataUntil(Process process, byte[]... endings) {
		try {
			@SuppressWarnings("rawtypes")
			Listener<Future> onTimeout = null;
			InputStream inputStream = process.getInputStream();
			OutputStream outputStream = outputBuffer;
			return MiscUtils.readUntil(safeReadingExecutor, inputStream, outputStream, safeReadingTimeoutMilliseconds,
					onTimeout, endings);
		} catch (IOException e) {
			throw new StandardError("Output stream reading error: " + e.toString(), e);
		}
	}

	protected void provideSafelyInput(Process process, String s) {
		try {
			process.getOutputStream().write(s.getBytes(getIOCharset()));
			process.getOutputStream().flush();
		} catch (IOException e) {
			throw new StandardError("Input stream writing error: " + e.toString(), e);
		}
	}

	protected void waitSafelyFor(final Process process, int timeoutMilliSeconds) {
		try {
			if (!process.waitFor(timeoutMilliSeconds, TimeUnit.MILLISECONDS)) {
				throw new StandardError("Process interruption timeout");
			}
		} catch (InterruptedException e) {
			throw new UnexpectedError(e);
		}
	}

	protected int acquireSafelyErrorDataUntil(Process process, byte[]... endings) {
		try {
			@SuppressWarnings("rawtypes")
			Listener<Future> onTimeout = null;
			InputStream inputStream = process.getErrorStream();
			OutputStream outputStream = errorBuffer;
			return MiscUtils.readUntil(safeReadingExecutor, inputStream, outputStream, safeReadingTimeoutMilliseconds,
					onTimeout, endings);
		} catch (IOException e) {
			throw new StandardError("Error stream reading error: " + e.toString(), e);
		}
	}

	protected void acquireSafelyAvailableOutputData() {
		int initialTimeout = getSafeReadingTimeoutMilliseconds();
		setSafeReadingTimeoutMilliseconds(1000);
		try {
			acquireSafelyOutputDataUntil(getCommandExecutor().getLaunchedProcess());
		} catch (Throwable ignore) {
		} finally {
			setSafeReadingTimeoutMilliseconds(initialTimeout);
		}
	}

	protected void acquireSafelyAvailableErrorData() {
		int initialTimeout = getSafeReadingTimeoutMilliseconds();
		setSafeReadingTimeoutMilliseconds(1000);
		try {
			acquireSafelyErrorDataUntil(getCommandExecutor().getLaunchedProcess());
		} catch (Throwable ignore) {
		} finally {
			setSafeReadingTimeoutMilliseconds(initialTimeout);
		}
	}

	protected String collectAvailableDiagnosticData() {
		try {
			String result = "";
			Process process = commandExecutor.getLaunchedProcess();
			result += "- Options: " + Arrays.toString(CommandExecutor.splitArguments(commandOptions));
			if (process == null) {
				result += "\n- Process status: failed to start";
			} else {
				try {
					result += "\n- Process status: terminated with exit value: "
							+ commandExecutor.getLaunchedProcess().exitValue();
				} catch (IllegalThreadStateException e) {
					result += "\n- Process status: not terminated";
				}
				acquireSafelyAvailableErrorData();
				String outputLogs = new String(getOutputBuffer().toByteArray(), getIOCharset());
				String errorLogs = new String(getErrorBuffer().toByteArray(), getIOCharset());
				if (outputLogs.length() > 0) {
					outputLogs = cleanLogs(outputLogs);
					result += "\n- Diagnostic output logs: " + outputLogs;
				}
				if (errorLogs.length() > 0) {
					errorLogs = cleanLogs(outputLogs);
					result += "\n- Diagnostic error logs: " + errorLogs;
				}
			}
			return result;
		} catch (Throwable t) {
			return "Diagnostic data collection failed (" + t.toString() + ")";
		}
	}

	protected String cleanLogs(String outputLogs) {
		return outputLogs;
	}

	protected Charset getIOCharset() {
		return Charset.forName("UTF-8");
	}

	@Override
	public String toString() {
		return "CommandLineInterface [commandOptions=" + commandOptions + "]";
	}

	public static void main(String[] args) {
		CommandLineInterface<Date> dateInterface = new CommandLineInterface<Date>(null, new File("cmd"), "/C datee") {

			@Override
			protected CommandExecutor createCommandExecutor() {
				CommandExecutor result = super.createCommandExecutor();
				return result;
			}

			@SuppressWarnings("deprecation")
			@Override
			protected Date communicate(Process process) {
				acquireSafelyOutputDataUntil(process, "The current date is: ".getBytes());

				getOutputBuffer().reset();
				acquireSafelyOutputDataUntil(process, "/".getBytes());
				Integer date = Integer.valueOf(getOutputBuffer().toString().replace("/", "").trim());

				getOutputBuffer().reset();
				acquireSafelyOutputDataUntil(process, "/".getBytes());
				Integer month = Integer.valueOf(getOutputBuffer().toString().replace("/", "").trim());

				getOutputBuffer().reset();
				acquireSafelyOutputDataUntil(process, "\n".getBytes());
				Integer year = Integer.valueOf(getOutputBuffer().toString().trim());

				return new Date(year, month, date);
			}
		};
		Date currentDate = dateInterface.execute();
		System.out.println("date according to the command line: " + currentDate);
	}

}
