/*******************************************************************************
 * Copyright (C) 2018 OTK Software
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************/
package com.otk.application.image.printer;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.Rectangle;
import java.awt.RenderingHints;
import java.awt.geom.Rectangle2D;
import java.awt.geom.Rectangle2D.Double;
import java.awt.image.BufferedImage;
import java.awt.print.Book;
import java.awt.print.PageFormat;
import java.awt.print.Paper;
import java.awt.print.Printable;
import java.awt.print.PrinterException;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Locale;

import javax.print.Doc;
import javax.print.DocFlavor;
import javax.print.DocPrintJob;
import javax.print.PrintService;
import javax.print.PrintServiceLookup;
import javax.print.SimpleDoc;
import javax.print.attribute.HashPrintRequestAttributeSet;
import javax.print.attribute.PrintRequestAttributeSet;
import javax.print.attribute.standard.JobName;
import javax.print.attribute.standard.PrintQuality;
import javax.print.event.PrintJobEvent;
import javax.print.event.PrintJobListener;

import org.apache.pdfbox.exceptions.COSVisitorException;
import org.apache.pdfbox.pdmodel.PDDocument;
import org.apache.pdfbox.pdmodel.PDPage;
import org.apache.pdfbox.pdmodel.common.PDRectangle;
import org.apache.pdfbox.pdmodel.edit.PDPageContentStream;
import org.apache.pdfbox.pdmodel.graphics.xobject.PDJpeg;

import com.otk.application.TheConstants;
import com.otk.application.error.StandardError;
import com.otk.application.error.UnexpectedError;
import com.otk.application.util.ImageUtils;
import com.otk.application.util.Log;
import com.otk.application.util.MathUtils;
import com.otk.application.util.MiscUtils;

/**
 * Allows to print images to actual printing devices, PDF files or just memory.
 * 
 * @author olitank
 *
 */
public class PrintingManager {

	public static void main(String[] args) throws Exception {
		get().setPrinterName(get().getAvailablePrinterNames().get(0));
		if (args.length == 1) {
			String filePath = args[0];
			BufferedImage image = ImageUtils.loadAdaptedImage(filePath);
			get().print(image);
		} else if (args.length == 0) {
			get().print(TheConstants.notFoundImage);
		} else {
			throw new RuntimeException("Invalid command line arguments. Expected: [<imagge-file-path-to-print>");
		}
	}

	public static final int BAD_QUALITY_MEMORY_PRINT_DPI = 72;
	public static final int GOOD_QUALITY_MEMORY_PRINT_DPI = 300;
	public static final int PRINT_DPI = 72;

	private static PrintingManager INSTANCE;

	private int freezeTimeoutMilliSeconds = 300000;
	private PrintService printService;
	private boolean printing = false;
	private String printerName = "";
	private boolean verbose = false;
	private double paperWidthMillimeters = 105;
	private double paperHeightMillimeters = 148;
	private double leftMarginMillimiters = 5;
	private double rightMarginMillimiters = 5;
	private double topMarginMillimiters = 5;
	private double bottomMarginMillimiters = 5;
	private int deviceDpi = 300;
	private boolean antialiasingEnabled = true;

	private PrintingManager() {
	}

	public static PrintingManager get() {
		if (INSTANCE == null) {
			INSTANCE = new PrintingManager();
		}
		return INSTANCE;
	}

	public int getDeviceDpi() {
		return deviceDpi;
	}

	public boolean isAntialiasingEnabled() {
		return antialiasingEnabled;
	}

	public void setAntialiasingEnabled(boolean antialiasingEnabled) {
		this.antialiasingEnabled = antialiasingEnabled;
	}

	public void setDeviceDpi(int deviceDpi) {
		this.deviceDpi = deviceDpi;
	}

	public int getFreezeTimeoutMilliSeconds() {
		return freezeTimeoutMilliSeconds;
	}

	public void setFreezeTimeoutMilliSeconds(int freezeTimeoutMilliSeconds) {
		this.freezeTimeoutMilliSeconds = freezeTimeoutMilliSeconds;
	}

	public String getPrinterName() {
		return printerName;
	}

	public void setPrinterName(String printerName) {
		this.printerName = printerName;
	}

	public boolean isVerbose() {
		return verbose;
	}

	public void setVerbose(boolean verbose) {
		this.verbose = verbose;
	}

	public double getPaperWidthMillimeters() {
		return paperWidthMillimeters;
	}

	public void setPaperWidthMillimeters(double paperWidthMillimeters) {
		this.paperWidthMillimeters = paperWidthMillimeters;
	}

	public double getPaperHeightMillimeters() {
		return paperHeightMillimeters;
	}

	public void setPaperHeightMillimeters(double paperHeightMillimeters) {
		this.paperHeightMillimeters = paperHeightMillimeters;
	}

	public double getLeftMarginMillimiters() {
		return leftMarginMillimiters;
	}

	public void setLeftMarginMillimiters(double leftMarginMillimiters) {
		this.leftMarginMillimiters = leftMarginMillimiters;
	}

	public double getRightMarginMillimiters() {
		return rightMarginMillimiters;
	}

	public void setRightMarginMillimiters(double rightMarginMillimiters) {
		this.rightMarginMillimiters = rightMarginMillimiters;
	}

	public double getTopMarginMillimiters() {
		return topMarginMillimiters;
	}

	public void setTopMarginMillimiters(double topMarginMillimiters) {
		this.topMarginMillimiters = topMarginMillimiters;
	}

	public double getBottomMarginMillimiters() {
		return bottomMarginMillimiters;
	}

	public void setBottomMarginMillimiters(double bottomMarginMillimiters) {
		this.bottomMarginMillimiters = bottomMarginMillimiters;
	}

	public boolean isPrinting() {
		return printing;
	}

	public BufferedImage printToMemory(Image image, int dpi) {
		PageFormat pageFormat = createPageFormat(dpi);
		Paper paper = pageFormat.getPaper();
		BufferedImage result = new BufferedImage(MathUtils.round(paper.getWidth()), MathUtils.round(paper.getHeight()),
				ImageUtils.getAdaptedBufferedImageType());
		Graphics2D g2d = result.createGraphics();
		g2d.setColor(Color.WHITE);
		g2d.fillRect(0, 0, result.getWidth(), result.getHeight());
		try {
			createPrintable(image).print(g2d, pageFormat, 0);
		} catch (PrinterException e) {
			throw new UnexpectedError(e);
		}
		g2d.dispose();
		return result;
	}

	public void printToPDF(Image image, File file) throws IOException {
		PageFormat pageFormat = createPageFormat(deviceDpi);
		Paper paper = pageFormat.getPaper();
		PDDocument doc = new PDDocument();
		try {
			PDPage page = new PDPage(new PDRectangle((float) paper.getWidth(), (float) paper.getHeight()));
			doc.addPage(page);
			PDPageContentStream contents = new PDPageContentStream(doc, page);
			createPrintable(image).print(contents, pageFormat, doc);
			contents.close();
			doc.save(file);
		} catch (COSVisitorException e) {
			throw new UnexpectedError(e);
		} finally {
			doc.close();
		}
	}

	public synchronized void print(final Image image) {
		print(image, true);
	}

	public synchronized void print(final Image image, final boolean waitForPrintingJobEnd) {
		printing = true;
		try {
			withoutFreezing(new Runnable() {

				@Override
				public void run() {
					initialize();
					Log.info("Printing...");
					PageFormat pageFormat = createPageFormat(PRINT_DPI);
					Book book = new Book();
					book.append(createPrintable(fixPrintPoorQualityOnSomeDevices(image)), pageFormat);
					DocPrintJob printJob = printService.createPrintJob();
					Doc doc = new SimpleDoc(book, DocFlavor.SERVICE_FORMATTED.PAGEABLE, null);
					ThePrintJobListener listener = createPrintJobListener();
					printJob.addPrintJobListener(listener);
					PrintRequestAttributeSet printAttributes = new HashPrintRequestAttributeSet();
					printAttributes.add(PrintQuality.HIGH);
					printAttributes.add(new JobName(printService.getName(), Locale.getDefault()));
					try {
						printJob.print(doc, printAttributes);
						if (waitForPrintingJobEnd) {
							while (!listener.isDead()) {
								MiscUtils.sleepSafely(1000);
							}
						}
					} catch (Exception e) {
						if (e instanceof UnexpectedError) {
							throw (UnexpectedError) e;
						}
					}
				}

				private Image fixPrintPoorQualityOnSomeDevices(Image image) {
					Double scaledBounds = getImageScaledBoundsInsidePrintableArea(image.getWidth(null),
							image.getHeight(null), deviceDpi);
					Dimension scaledSize = MathUtils.toDimension(MathUtils.getSize(scaledBounds));
					if (antialiasingEnabled) {
						return ImageUtils.scalePreservingRatio(image, scaledSize.width, scaledSize.height, true, true);
					} else {
						return image.getScaledInstance(scaledSize.width, scaledSize.height, Image.SCALE_SMOOTH);
					}
				}
			});
		} finally {
			printing = false;
		}
	}

	protected ThePrintable createPrintable(Image image) {
		return new ThePrintable(image);
	}

	protected void withoutFreezing(final Runnable runnable) {
		if (!MiscUtils.runWithTimeout(new Runnable() {
			@Override
			public void run() {
				runnable.run();
			}
		}, freezeTimeoutMilliSeconds, null)) {
			throw new UnexpectedError("Time out error. The software execution may become unstable");
		}
	}

	protected ThePrintJobListener createPrintJobListener() {
		return new ThePrintJobListener();
	}

	public void initialize() {
		Log.info("Initializing printing");
		if (printerName.length() == 0) {
			throw new StandardError("The printing device name to use was not specified");
		}
		List<PrintService> availablePrinters = getAvailablePrinters();
		printService = null;
		for (PrintService candidatePrinter : availablePrinters) {
			if (verbose) {
				Log.info("Found printer service: '" + candidatePrinter.getName() + "'");
			}
			if (printerName.equals(candidatePrinter.getName())) {
				printService = candidatePrinter;
			}
		}
		Log.info(availablePrinters.size() + " printer service(s) found");
		if (printService == null) {
			throw new StandardError("'" + printerName + "' printer service not found");
		}
		Log.info("Selected printer service: '" + printerName + "'");
	}

	public List<String> getAvailablePrinterNames() {
		List<String> result = new ArrayList<String>();
		for (PrintService serv : getAvailablePrinters()) {
			result.add(serv.getName());
		}
		return result;
	}

	private List<PrintService> getAvailablePrinters() {
		List<PrintService> result = new ArrayList<PrintService>();
		result.addAll(Arrays.asList(PrintServiceLookup.lookupPrintServices(DocFlavor.INPUT_STREAM.GIF, null)));
		return result;
	}

	private PageFormat createPageFormat(int dpi) {
		PageFormat pageFormat = new PageFormat();
		Paper paper = pageFormat.getPaper();
		double widthDots = MathUtils.millimetersToDots(paperWidthMillimeters, dpi);
		double heightDots = MathUtils.millimetersToDots(paperHeightMillimeters, dpi);
		paper.setSize(widthDots, heightDots);
		double leftMarginDots = MathUtils.millimetersToDots(leftMarginMillimiters, dpi);
		double rightMarginDots = MathUtils.millimetersToDots(rightMarginMillimiters, dpi);
		double topMarginDots = MathUtils.millimetersToDots(topMarginMillimiters, dpi);
		double bottomMarginDots = MathUtils.millimetersToDots(bottomMarginMillimiters, dpi);
		double insideMarginsWidth = paper.getWidth() - (rightMarginDots + leftMarginDots);
		double insideMarginsHeight = paper.getHeight() - (bottomMarginDots + topMarginDots);
		paper.setImageableArea(leftMarginDots, topMarginDots, insideMarginsWidth, insideMarginsHeight);
		pageFormat.setPaper(paper);
		return pageFormat;
	}

	public Double getImageScaledBoundsInsidePrintableArea(int imageWidth, int imageHeight, int dpi) {
		return MathUtils.getImageScaledBoundsInsidePrintableArea(imageWidth, imageHeight, createPageFormat(dpi));
	}

	public double getPrintableAreaWidthOverHeightRatio() {
		PageFormat pageFormat = createPageFormat(PRINT_DPI);
		return pageFormat.getImageableWidth() / pageFormat.getImageableHeight();
	}

	public Rectangle getPrintableAreaDots(int dpi) {
		PageFormat pageFormat = createPageFormat(dpi);
		int x = MathUtils.round(pageFormat.getImageableX());
		int y = MathUtils.round(pageFormat.getImageableY());
		int w = MathUtils.round(pageFormat.getImageableWidth());
		int h = MathUtils.round(pageFormat.getImageableHeight());
		return new Rectangle(x, y, w, h);
	}

	public Dimension getPaperSizeDots(int dpi) {
		Paper paper = createPageFormat(dpi).getPaper();
		return new Dimension(MathUtils.round(paper.getWidth()), MathUtils.round(paper.getHeight()));
	}

	private class ThePrintJobListener implements PrintJobListener {

		private boolean dead = false;

		public boolean isDead() {
			return dead;
		}

		private void throwError(String generalErrorMsg) {
			throw new StandardError(generalErrorMsg);
		}

		@Override
		public void printJobFailed(PrintJobEvent paramPrintJobEvent) {
			dead = true;
			throwError("Print job failed");
		}

		@Override
		public void printJobCanceled(PrintJobEvent paramPrintJobEvent) {
			dead = true;
			throwError("Cancelled print job");
		}

		@Override
		public void printJobNoMoreEvents(PrintJobEvent paramPrintJobEvent) {
			dead = true;
		}

		@Override
		public void printJobRequiresAttention(PrintJobEvent paramPrintJobEvent) {
			throwError("Blocked print job");
		}

		@Override
		public void printDataTransferCompleted(PrintJobEvent pje) {
		}

		@Override
		public void printJobCompleted(PrintJobEvent pje) {
			dead = true;
			Log.info("Print job completed");
		}
	}

	protected class ThePrintable implements Printable {
		private Image image;

		public ThePrintable(Image image) {
			this.image = image;
		}

		@Override
		public int print(Graphics graphics, PageFormat pageFormat, int pageIndex) throws PrinterException {
			Graphics2D graphics2d = (Graphics2D) graphics;
			final Rectangle2D.Double bounds = MathUtils.getImageScaledBoundsInsidePrintableArea(image.getWidth(null),
					image.getHeight(null), pageFormat);
			ImageUtils.setSmoothScalingRenderingHints(graphics2d);
			graphics2d.setRenderingHint(RenderingHints.KEY_RENDERING, RenderingHints.VALUE_RENDER_QUALITY);
			graphics2d.drawImage(image, MathUtils.round(bounds.x), MathUtils.round(bounds.y),
					MathUtils.round(bounds.width), MathUtils.round(bounds.height), null);
			return PAGE_EXISTS;
		}

		public void print(PDPageContentStream contents, PageFormat pageFormat, PDDocument doc) throws IOException {
			final Rectangle2D.Double drawBounds = MathUtils
					.getImageScaledBoundsInsidePrintableArea(image.getWidth(null), image.getHeight(null), pageFormat);
			drawBounds.x += pageFormat.getImageableX();
			drawBounds.y += pageFormat.getImageableY();

			PDJpeg pdImage = new PDJpeg(doc,
					ImageUtils.getBufferedImage(image.getScaledInstance((int) Math.round(drawBounds.width),
							(int) Math.round(drawBounds.height), Image.SCALE_SMOOTH)));

			contents.drawImage(pdImage, (float) drawBounds.x, (float) drawBounds.y);
		}
	}
}
