/*******************************************************************************
 * Copyright (C) 2018 OTK Software
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************/
package com.otk.application.image.filter;

import java.awt.image.BufferedImage;
import com.otk.application.util.ImageUtils;

public abstract class AbstractBoxFilter extends AbstractFilter {

	public int radius = 1;

	private int[] precomputedOutputValues;
	private int precomputedRadius = -1;

	protected abstract int getComponentCount();

	protected abstract int getComponentValueFromPixel(int pixel, int c);

	protected abstract int getPixelWithNewComponentValue(int pixel, int c, int value);

	protected abstract int getPrecomputedOutputValue(int sum, int count);

	public static int scaleRadiusAccordingImageSize(int radius, BufferedImage image) {
		int result = Math.min(image.getWidth(), image.getHeight());
		result = (result / 2) - 1;
		result = Math.round(result * radius / 100f);
		return result;
	}

	public static int unscaleRadiusAccordingImageSize(int radius, BufferedImage image) {
		int result = Math.min(image.getWidth(), image.getHeight());
		result = (result / 2) - 1;
		result = Math.round(100f * radius / result);
		return result;
	}

	@Override
	public FilteringContext doExecute(FilteringContext context) {
		BufferedImage image = context.getImage();

		int scaledRadius = scaleRadiusAccordingImageSize(radius, image);
		if (isPrecomputingRequired(image, scaledRadius)) {
			precompute(image, scaledRadius);
		}

		int[] pixels = getPixelsFromImage(image);
		int w = image.getWidth();
		int h = image.getHeight();

		int xMax = w - 1;
		int yMax = h - 1;
		int pixelCount = w * h;
		int componentCount = getComponentCount();
		int components[][] = new int[componentCount][pixelCount];
		int x, y, i, pixel, firstRangePixel, lastRangePixel, yp, yi, yw;
		int componentSums[] = new int[componentCount];
		int vmin[] = new int[Math.max(w, h)];
		int vmax[] = new int[Math.max(w, h)];

		yw = yi = 0;

		for (y = 0; y < h; y++) {
			for (int c = 0; c < componentCount; c++) {
				componentSums[c] = 0;
			}
			for (i = -scaledRadius; i <= scaledRadius; i++) {
				pixel = pixels[yi + Math.min(xMax, Math.max(i, 0))];
				for (int c = 0; c < componentCount; c++) {
					componentSums[c] += getComponentValueFromPixel(pixel, c);
				}
			}
			for (x = 0; x < w; x++) {

				for (int c = 0; c < componentCount; c++) {
					components[c][yi] += precomputedOutputValues[componentSums[c]];
				}

				if (y == 0) {
					vmin[x] = Math.min(x + scaledRadius + 1, xMax);
					vmax[x] = Math.max(x - scaledRadius, 0);
				}
				firstRangePixel = pixels[yw + vmin[x]];
				lastRangePixel = pixels[yw + vmax[x]];

				for (int c = 0; c < componentCount; c++) {
					componentSums[c] += getComponentValueFromPixel(firstRangePixel, c);
					componentSums[c] -= getComponentValueFromPixel(lastRangePixel, c);
				}
				yi++;
			}
			yw += w;
		}

		for (x = 0; x < w; x++) {
			for (int c = 0; c < componentCount; c++) {
				componentSums[c] = 0;
			}
			yp = -scaledRadius * w;
			for (i = -scaledRadius; i <= scaledRadius; i++) {
				yi = Math.max(0, yp) + x;
				for (int c = 0; c < componentCount; c++) {
					componentSums[c] += components[c][yi];
				}
				yp += w;
			}
			yi = x;
			for (y = 0; y < h; y++) {
				for (int c = 0; c < componentCount; c++) {
					pixels[yi] = getPixelWithNewComponentValue(pixels[yi], c,
							precomputedOutputValues[componentSums[c]]);
				}
				if (x == 0) {
					vmin[y] = Math.min(y + scaledRadius + 1, yMax) * w;
					vmax[y] = Math.max(y - scaledRadius, 0) * w;
				}
				firstRangePixel = x + vmin[y];
				lastRangePixel = x + vmax[y];

				for (int c = 0; c < componentCount; c++) {
					componentSums[c] += components[c][firstRangePixel];
					componentSums[c] -= components[c][lastRangePixel];
				}

				yi += w;
			}
		}

		BufferedImage result = createImageFromPixels(pixels, w, h);
		if (result != null) {
			context = context.withImage(result);
		}
		return context;

	}

	protected int[] getPixelsFromImage(BufferedImage image) {
		return FastestRGBAccess.get(image);
	}

	protected BufferedImage createImageFromPixels(int[] pixels, int w, int h) {
		BufferedImage result = new BufferedImage(w, h, ImageUtils.getAdaptedBufferedImageType());
		FastestRGBAccess.set(pixels, result);
		return result;
	}

	protected boolean isPrecomputingRequired(BufferedImage image, int radius) {
		return precomputedRadius != radius;
	}

	protected void precompute(BufferedImage image, int radius) {
		int slidingRangeLenth = radius + radius + 1;
		precomputedOutputValues = new int[256 * slidingRangeLenth];
		for (int sum = 0; sum < 256 * slidingRangeLenth; sum++) {
			precomputedOutputValues[sum] = getPrecomputedOutputValue(sum, slidingRangeLenth);
		}
		precomputedRadius = radius;
	}

}
