/*******************************************************************************
 * Copyright (C) 2018 OTK Software
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************/
package com.otk.application.image.camera;

import java.awt.Graphics2D;
import java.awt.image.BufferedImage;
import java.util.Iterator;
import java.util.List;

import com.otk.application.error.AbstractApplicationError;
import com.otk.application.util.ImageUtils;

/**
 * Base class for {@link Camera} drivers that use webcam-oriented APIs.
 * 
 * @author olitank
 *
 */
public abstract class WebcamDriver extends AbstractCameraDriver {

	protected abstract void startCapture(FrameFormat format);

	protected WebcamDriver(Camera camera) {
		super(camera);
	}

	@Override
	public String getName() {
		return "Webcam";
	}

	@Override
	public void startCapture() {
		startCapture(camera.getVideoFormat());
	}

	protected void preventSuccessiveActivationCrashWhile(Runnable runnable) {
		try {
			doCleanUp();
			doSetUp();
		} catch (Exception e) {
			AbstractApplicationError.rethrow(e);
		}
		try {
			runnable.run();
		} finally {
			try {
				doCleanUp();
				doSetUp();
			} catch (Exception e) {
				AbstractApplicationError.rethrow(e);
			}
		}
	}

	protected static void arrangeFormatList(List<FrameFormat> list) {
		for (Iterator<FrameFormat> it = list.iterator(); it.hasNext();) {
			FrameFormat format = it.next();
			if (format.width == 0) {
				it.remove();
			}
			if (format.height == 0) {
				it.remove();
			}
		}
	}

	public BufferedImage adaptDeviceImage(BufferedImage inputImage) {
		int width = inputImage.getWidth(null);
		int height = inputImage.getHeight(null);
		BufferedImage result = new BufferedImage(width, height, ImageUtils.getAdaptedBufferedImageType());
		Graphics2D g2d = result.createGraphics();
		g2d.translate(width, 0);
		g2d.scale(-1, 1);
		g2d.drawImage(inputImage, getDeviceImagePrecorrectionOp(inputImage), 0, 0);
		g2d.dispose();
		return result;
	}

}
