/*******************************************************************************
 * Copyright (C) 2018 OTK Software
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************/
package com.otk.application.image.camera;

import java.awt.Image;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import com.otk.application.util.Listener;

import xy.reflect.ui.util.ReflectionUIError;

/**
 * This class is a facade class allowing to easily get a picture from a webcam
 * device.
 * 
 * @author olitank
 *
 */
public class CameraImageSource {

	/**
	 * Object managing the actual access to the webcam device.
	 */
	private static Camera camera;

	/**
	 * Available webcam device names.
	 */
	private static List<String> deviceNames;

	/**
	 * The last captured image.
	 */
	private Image capturedImage;

	/**
	 * Construct an instance with a default captured image.
	 * 
	 * @param capturedImage
	 *            The value of {@link #capturedImage}.
	 */
	public CameraImageSource(Image capturedImage) {
		this.capturedImage = capturedImage;
	}

	/**
	 * @return The reference to the {@link #camera} object. The object is lazily
	 *         created when this method is called for the 1st time.
	 */
	public Camera getCamera() {
		if (camera == null) {
			camera = new Camera();
			camera.setFreezeTimeoutMilliSeconds(20000);
			setDeviceName(getDeviceNames().get(0));
			selectBestResolution();
		}
		return camera;
	}

	/**
	 * Selects a decent resolution among those available.
	 */
	public void selectBestResolution() {
		FrameFormat bestFormat = new FrameFormat(640, 480);
		List<FrameFormat> formats = getResolutions();
		if (formats.size() > 0) {
			if (!formats.contains(bestFormat)) {
				formats = new ArrayList<FrameFormat>(formats);
				Collections.sort(formats);
				bestFormat = formats.get(formats.size() - 1);
			}
		}
		if (bestFormat != null) {
			setResolution(bestFormat);
		}
	}

	/**
	 * @return The current webcam device resolution.
	 */
	public FrameFormat getResolution() {
		return getCamera().getVideoFormat();
	}

	/**
	 * Requests the current webcam device to provide images of the given resolution.
	 * 
	 * @param resolution
	 *            The new resolution to set.
	 */
	public void setResolution(FrameFormat resolution) {
		getCamera().withTemporaryInterruptionIfActive(new Runnable() {
			@Override
			public void run() {
				getCamera().setVideoFormat(resolution);
			}
		});
		capturedImage = null;
	}

	/**
	 * @return The resolutions available from the current webcam device.
	 */
	public List<FrameFormat> getResolutions() {
		return getCamera().getVideoFormats();
	}

	/**
	 * @return The current device name.
	 */
	public String getDeviceName() {
		return getCamera().getDeviceFullName();
	}

	/**
	 * Sets the current webcam device name.
	 * 
	 * @param deviceName
	 *            The device name to set.
	 */
	public void setDeviceName(String deviceName) {
		getCamera().withTemporaryInterruptionIfActive(new Runnable() {
			@Override
			public void run() {
				getCamera().setDeviceFullName(deviceName);
				selectBestResolution();
			}
		});
		capturedImage = null;
	}

	/**
	 * @return The names of available webcam devices. Note that those names are
	 *         cached during the first call of this method.
	 */
	public List<String> getDeviceNames() {
		if (deviceNames == null) {
			deviceNames = getCamera().listAvailableDevices(new Listener<Throwable>() {
				@Override
				public void handle(Throwable t) {
					throw new ReflectionUIError(t);
				}
			});
		}
		return deviceNames;
	}

	/**
	 * Clears the cache of used by {@link #getDeviceNames()}.
	 */
	public void clearDeviceNameCache() {
		deviceNames = null;
	}

	/**
	 * Connects to the current webcam device and start a video stream. The current
	 * image can be obtained through {@link #getCurrentImage()}.
	 */
	public void start() {
		getCamera().start();
	}

	/**
	 * Disconnects to the current webcam device.
	 */
	public void stop() {
		getCamera().stop();
	}

	/**
	 * @return The last captured image.
	 */
	public Image getCapturedImage() {
		return capturedImage;
	}

	/**
	 * @return The last captured image updated with the current webcam device image
	 *         if available.
	 */
	public Image getCurrentImage() {
		Image image = getCamera().getCurrentImage();
		if (image != null) {
			capturedImage = image;
		}
		return capturedImage;
	}
	

	

}
