/*******************************************************************************
 * Copyright (C) 2018 OTK Software
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************/
package com.otk.application.image.camera;

import java.awt.image.BufferedImage;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.swing.JOptionPane;

import com.otk.application.error.AbstractApplicationError;
import com.otk.application.error.StandardError;
import com.otk.application.error.UnexpectedError;
import com.otk.application.service.AbstractService;
import com.otk.application.util.Accessor;
import com.otk.application.util.Listener;

import xy.reflect.ui.util.component.ImagePanel;

/**
 * This class allows to use camera devices with video streaming capabilities.
 * 
 * @author olitank
 *
 */
public class Camera extends AbstractService {

	public static void main(String[] args) throws Throwable {
		Camera camera = new Camera();
		List<String> deviceNames = camera.listAvailableDevices(null);
		camera.setDeviceFullName(deviceNames.get(deviceNames.size() - 1));
		camera.start();
		for (int i = 0; i < 3; i++) {
			JOptionPane.showMessageDialog(null, new ImagePanel(camera.getCurrentImage()));
		}
		camera.stop();
	}

	private static final String DRIVER_AND_DEVICE_NAME_PATTERN = "\\[([^\\]]+)\\] (.+)";
	private static final String DRIVER_AND_DEVICE_NAME_FORMAT = "[{0}] {1}";

	private AbstractCameraDriver activeDriver;
	private BufferedImage lastImageFromDevice;
	private BufferedImage currentImage;
	private Map<String, List<FrameFormat>> videoFormatsByDevice = new HashMap<String, List<FrameFormat>>();
	private int frameCountSinceStarted = 0;
	private FrameProcessor frameProcessor;
	private List<AbstractCameraDriver> drivers;

	private String deviceFullName = "";
	private FrameFormat videoFormat = new FrameFormat(320, 240);
	private String threadPriority = "NORM_PRIORITY";
	private int freezeTimeoutMilliSeconds = 30000;
	private int brightnessCompensation = 0;
	private int contrastCompensation = 0;

	public Camera() {
		drivers = createDrivers();
	}

	@Override
	protected String getServiceName() {
		return "Camera";
	}

	public int getBrightnessCompensation() {
		return brightnessCompensation;
	}

	public void setBrightnessCompensation(int brightnessCompensation) {
		this.brightnessCompensation = brightnessCompensation;
		driverSettingsUpdated();
	}

	protected void driverSettingsUpdated() {
		if (activeDriver != null) {
			activeDriver.settingsUpdated();
		}
	}

	public int getContrastCompensation() {
		return contrastCompensation;
	}

	public void setContrastCompensation(int contrastCompensation) {
		this.contrastCompensation = contrastCompensation;
		driverSettingsUpdated();
	}

	public int getFreezeTimeoutMilliSeconds() {
		return freezeTimeoutMilliSeconds;
	}

	public void setFreezeTimeoutMilliSeconds(int freezeTimeoutMilliSeconds) {
		this.freezeTimeoutMilliSeconds = freezeTimeoutMilliSeconds;
		driverSettingsUpdated();
	}

	public String getDeviceFullName() {
		return this.deviceFullName;
	}

	public void setDeviceFullName(String deviceFullName) {
		this.deviceFullName = deviceFullName;
		driverSettingsUpdated();
	}

	public FrameFormat getVideoFormat() {
		return this.videoFormat;
	}

	public void setVideoFormat(FrameFormat videoFormat) {
		this.videoFormat = videoFormat;
		driverSettingsUpdated();
	}

	public String getThreadPriority() {
		return this.threadPriority;
	}

	public void setThreadPriority(String threadPriority) {
		this.threadPriority = threadPriority;
		driverSettingsUpdated();
	}

	public void logInfo(String s) {
		System.out.println(s);
	}

	public void logError(String s) {
		System.err.println(s);
	}

	public int getFrameCountSinceStarted() {
		return frameCountSinceStarted;
	}

	public BufferedImage getCurrentImage() {
		return currentImage;
	}

	public void setCurrentImage(BufferedImage image) {
		currentImage = image;
	}

	@Override
	protected void handleInconsistentState() {
		super.handleInconsistentState();
	}

	public BufferedImage getCurrentRawImage() {
		return lastImageFromDevice;
	}

	protected List<FrameFormat> getFormats(Accessor<List<FrameFormat>> activeDriverFormatsAccessor,
			Map<String, List<FrameFormat>> formatCache) {
		String deviceName = isSetup() ? formatDeviceFullName(activeDriver, extractLocalDeviceName(getDeviceFullName()))
				: getDeviceFullName();
		List<FrameFormat> result = formatCache.get(deviceName);
		if (result != null) {
			return result;
		}
		result = new ArrayList<FrameFormat>();
		synchronized (getLifeCycleMutex()) {
			if (isSetup()) {
				result.addAll(activeDriverFormatsAccessor.get());
			} else {
				try {
					setUp();
				} catch (Exception e) {
					return Collections.emptyList();
				}
				result.addAll(activeDriverFormatsAccessor.get());
				try {
					cleanUp();
				} catch (Exception ignore) {
					handleInconsistentState();
				}
			}
		}
		formatCache.put(deviceName, result);
		return result;
	}

	public List<FrameFormat> getVideoFormats() {
		Accessor<List<FrameFormat>> activeDriverFormatsAccessor = new Accessor<List<FrameFormat>>() {

			@Override
			public List<FrameFormat> get() {
				logInfo("Enumerating Video Formats");
				return activeDriver.getVideoFormatsWhileCameraInitializedAndNotActive();
			}
		};
		return getFormats(activeDriverFormatsAccessor, videoFormatsByDevice);
	}

	public List<String> listAvailableDevices(Listener<Throwable> errorListener) {
		logInfo("Enumerating camera devices");
		List<String> result = new ArrayList<String>();
		synchronized (getLifeCycleMutex()) {
			withTemporaryInterruptionIfActive(new Runnable() {
				@Override
				public void run() {
					for (AbstractCameraDriver driver : drivers) {
						logInfo("Enumerating  '" + driver.getName() + "' driver devices");
						try {
							for (String deviceLocalName : driver.listDeviceLocalNamesWhileCameraNotInitialized()) {
								result.add(formatDeviceFullName(driver, deviceLocalName));
							}
						} catch (Throwable t) {
							if (errorListener != null) {
								errorListener.handle(AbstractApplicationError.getToRethrow(
										MessageFormat.format("{0} device list enumeration failure", driver.getName()),
										t));
							}
						}
					}
				}
			});
		}
		return result;
	}

	protected void checkFormat(FrameFormat format, List<FrameFormat> formats) {
		if (!formats.contains(format)) {
			throw new StandardError("Format not supported: " + format + ".\n"
					+ "Restart the software if there was a hardware modification.");
		}
	}

	public void handleNewImageFromDevice(final BufferedImage newImage) {
		lastImageFromDevice = newImage;
		frameProcessor.handleNewImageFromDevice();
	}

	@Override
	protected void triggerActivation() throws Exception {
		checkFormat(getVideoFormat(), getVideoFormats());
		activeDriver.startCapture();
	}

	@Override
	protected void triggerInterruption() throws Exception {
		activeDriver.stopCapture();
	}

	@Override
	protected void waitForCompleteInterruption() {
		activeDriver.doWaitForCompleteInterruption();
	}

	@Override
	protected void setUp() throws Exception {
		logInfo("Connecting to camera device '" + getDeviceFullName() + "'");
		activeDriver = getDriver();
		activeDriver.settingsUpdated();
		activeDriver.doSetUp();
		frameCountSinceStarted = 0;
		frameProcessor = new SynchronousFrameProcessor();
		frameProcessor.begin();
	}

	@Override
	protected void cleanUp() throws Exception {
		logInfo("Disonnecting from camera device '" + getDeviceFullName() + "'");
		frameProcessor.end();
		setCurrentImage(null);
		activeDriver.doCleanUp();
		activeDriver = null;
		lastImageFromDevice = null;
		frameProcessor = null;
	}

	@Override
	public void handleRunErrorAndStop(Throwable t) {
		super.handleRunErrorAndStop(t);
	}

	public List<AbstractCameraDriver> createDrivers() {
		List<AbstractCameraDriver> result = new ArrayList<AbstractCameraDriver>();
		result.add(new NOCamDriver(this));
		result.add(getWebcamDriver());
		return result;
	}

	@SuppressWarnings("unchecked")
	protected AbstractCameraDriver getWebcamDriver() {
		try {
			String webcamDriverClassName = System.getProperty(Camera.class.getName() + ".driver",
					Camera.class.getPackage().getName() + ".OpenIMAJDriver");
			Class<? extends AbstractCameraDriver> webcamDriverClass = (Class<? extends AbstractCameraDriver>) Class
					.forName(webcamDriverClassName);
			return webcamDriverClass.getConstructor(Camera.class).newInstance(Camera.this);
		} catch (Throwable t) {
			throw new UnexpectedError(t);
		}
	}

	public String getDriverName() {
		return extractDriverName(getDeviceFullName());
	}

	public String getLocalDeviceName() {
		return extractLocalDeviceName(getDeviceFullName());
	}

	public List<AbstractCameraDriver> getDrivers() {
		return drivers;
	}

	public AbstractCameraDriver getDriver() {
		if ("".equals(getDeviceFullName())) {
			throw new StandardError("Camera device not selected!");
		}
		String driverName = getDriverName();
		if (driverName == null) {
			throw new StandardError("Invalid camera device name: '" + getDeviceFullName() + "'");
		}
		for (AbstractCameraDriver driver : drivers) {
			if (driver.getName().equals(driverName)) {
				return driver;
			}
		}
		throw new StandardError("Invalid camera driver name: '" + driverName + "'");
	}

	public <D extends AbstractCameraDriver> D findDriver(Class<D> driverClass) {
		for (AbstractCameraDriver driver : drivers) {
			if (driverClass.isInstance(driver)) {
				return driverClass.cast(driver);
			}
		}
		return null;
	}

	public static String extractDriverName(String deviceFullName) {
		Matcher matcher = Pattern.compile(DRIVER_AND_DEVICE_NAME_PATTERN).matcher(deviceFullName);
		if (matcher.matches()) {
			String result = matcher.group(1);
			if (result.length() > 0) {
				return result;
			}
		}
		return null;
	}

	public static String formatDeviceFullName(AbstractCameraDriver driver, String deviceLocalName) {
		return MessageFormat.format(DRIVER_AND_DEVICE_NAME_FORMAT, driver.getName(), deviceLocalName);
	}

	protected static String extractLocalDeviceName(String deviceFullName) {
		Matcher matcher = Pattern.compile(DRIVER_AND_DEVICE_NAME_PATTERN).matcher(deviceFullName);
		if (!matcher.matches()) {
			return "";
		}
		String result = matcher.group(2);
		return result;
	}

	protected interface FrameProcessor {

		void begin();

		void handleNewImageFromDevice();

		void end();

	}

	protected class SynchronousFrameProcessor implements FrameProcessor {

		@Override
		public void begin() {
		}

		@Override
		public void handleNewImageFromDevice() {
			BufferedImage imageFromDevice = lastImageFromDevice;
			if (imageFromDevice == null) {
				return;
			}
			displayNewImageFromDevice(imageFromDevice);
		}

		private void displayNewImageFromDevice(BufferedImage imageFromDevice) {
			setCurrentImage(imageFromDevice);
			frameCountSinceStarted++;
		}

		@Override
		public void end() {
		}

	}

}
