/*******************************************************************************
 * Copyright (C) 2018 OTK Software
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************/
package com.otk.application.error;

import java.util.Arrays;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

import javax.swing.SwingUtilities;

import com.otk.application.IdPhotoUI;
import com.otk.application.util.Analytics;
import com.otk.application.util.Log;
import com.otk.application.util.MiscUtils;

import xy.reflect.ui.util.ReflectionUIError;

/**
 * This class handles the display of normal and uncaught errors.
 * 
 * @author olitank
 *
 */
public class ErrorManager {

	protected static Lock exceptionHandlerLock = new ReentrantLock();

	public static void initialize() {
		Thread.setDefaultUncaughtExceptionHandler(new TheUncaughtExceptionHandler());
		System.setProperty("sun.awt.exception.handler", TheUncaughtExceptionHandler.class.getName());
	}

	public static void handleUncaughtException(final Throwable t) {
		if (!exceptionHandlerLock.tryLock()) {
			t.printStackTrace();
			return;
		}
		try {
			SwingUtilities.invokeLater(new Runnable() {
				@Override
				public void run() {
					showErrorDialog(t);
				}
			});
		} finally {
			exceptionHandlerLock.unlock();
		}
	}

	public static void showErrorDialog(Throwable t) {
		try {
			IdPhotoUI.RENDERER.handleExceptionsFromDisplayedUI(MiscUtils.getFocusedWindow(),
					new ReflectionUIError(formatError(t), t));
		} catch (Throwable t1) {
			try {
				Log.error(t);
			} catch (Throwable t2) {
				t.printStackTrace();
			}
		}
	}

	public static String formatError(Throwable t) {
		if (t instanceof OutOfMemoryError) {
			return "The memory is not enough to perform the requested action(s).";
		}
		if (t instanceof NullPointerException) {
			return "An internal error occurred. The system may become unstable.";
		}
		String result = t.getMessage();
		if ((result == null) || (result.length() == 0)) {
			result = t.getClass().getName();
		}
		String className = t.getClass().getName();
		if (result.contains(className)) {
			String classCaption = MiscUtils.identifierToCaption(t.getClass().getSimpleName());
			if (!classCaption.contains("Error") && classCaption.contains("Exception")) {
				classCaption = classCaption.replace("Exception", "Error");
			}
			result = result.replace(className, classCaption);
		}
		if (t.getCause() != null) {
			result = result.replace(t.getCause().toString(), formatError(t.getCause()));
		}
		return result;
	}

	public static class TheUncaughtExceptionHandler implements Thread.UncaughtExceptionHandler {

		@Override
		public void uncaughtException(Thread thread, Throwable t) {
			if (isBenign(t)) {
				try {
					Log.info("Benign exception skipped: " + MiscUtils.getPrintedStackTrace(t));
					Analytics.track("Benign Exception", t.toString() + "\n" + Arrays.toString(t.getStackTrace()));
				} catch (Throwable ignore) {
				}
				return;
			}
			ErrorManager.handleUncaughtException(t);
		}

		protected boolean isBenign(Throwable t) {
			String printedStackTrace = MiscUtils.getPrintedStackTrace(t);
			if (printedStackTrace.contains("java.lang.IncompatibleClassChangeError")) {
				if (printedStackTrace.contains("com.sun.glass.ui.gtk.GtkApplication._runLoop")) {
					return true;
				}
			}
			return false;
		}

		public void handle(Throwable thrown) {
			uncaughtException(null, thrown);
		}

	}

	public static void rethrow(Throwable thrown) {
		rethrow(thrown, true);
	}

	public static void rethrow(Throwable e, boolean unknownErrorAsUnexpected) {
		if (e instanceof StandardError) {
			throw (StandardError) e;
		}
		if (e instanceof UnexpectedError) {
			throw (UnexpectedError) e;
		}
		if (unknownErrorAsUnexpected) {
			throw new UnexpectedError(e);
		} else {
			throw new StandardError(e.toString(), e);
		}
	}
}
