/*******************************************************************************
 * Copyright (C) 2018 OTK Software
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************/
package com.otk.application;

import java.awt.Component;
import java.awt.Image;
import java.awt.Window;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Arrays;
import java.util.List;

import com.otk.application.error.UnexpectedError;
import com.otk.application.util.Analytics;

import xy.reflect.ui.CustomizedUI;
import xy.reflect.ui.control.swing.renderer.CustomizedSwingRenderer;
import xy.reflect.ui.control.swing.renderer.Form;
import xy.reflect.ui.control.swing.renderer.SwingRenderer;
import xy.reflect.ui.control.swing.renderer.WindowManager;
import xy.reflect.ui.info.field.IFieldInfo;
import xy.reflect.ui.info.menu.IMenuElement;
import xy.reflect.ui.info.menu.MenuModel;
import xy.reflect.ui.info.menu.builtin.swing.SaveMenuItem;
import xy.reflect.ui.info.method.IMethodInfo;
import xy.reflect.ui.info.method.InvocationData;
import xy.reflect.ui.info.parameter.IParameterInfo;
import xy.reflect.ui.info.type.ITypeInfo;
import xy.reflect.ui.info.type.factory.EncapsulatedObjectFactory;
import xy.reflect.ui.info.type.factory.InfoProxyFactory;
import xy.reflect.ui.util.Accessor;
import xy.reflect.ui.util.ReflectionUIError;
import xy.reflect.ui.util.ReflectionUIUtils;
import xy.reflect.ui.util.SwingRendererUtils;
import xy.reflect.ui.util.Visitor;

/**
 * This class generates the GUI. It uses the ReflectionUI library.
 * 
 * @author olitank
 *
 */
public class IdPhotoUI {

	/**
	 * Whether the ReflectionUI design mode is active or not.
	 */
	private static final boolean DESIGN_MODE = Boolean
			.valueOf(System.getProperty(IdPhotoUI.class.getName() + ".designMode", "false"));

	/**
	 * Used to complete (allow to undo) and improve (add usage tracking) some
	 * ReflectionUI type informations.
	 */
	public static final InfoProxyFactory TYPEFACTORY = new InfoProxyFactory() {

		@Override
		protected List<IParameterInfo> getParameters(IMethodInfo method, ITypeInfo containingType) {
			if ((containingType.getName().equals(IdPhotoProject.class.getName()))
					&& method.getName().equals("setPhotograph")) {
				RENDERER.getLastInvocationDataByMethodSignature().remove(method.getSignature());
			}
			return super.getParameters(method, containingType);
		}

		@Override
		protected Runnable getNextInvocationUndoJob(final IMethodInfo method, ITypeInfo containingType, Object object,
				InvocationData invocationData) {
			if ((object instanceof IdPhotoProject) && method.getName().startsWith("setPhotograph")) {
				final IdPhotoProject idPhotoProject = (IdPhotoProject) object;
				final Image oldPhotograph = idPhotoProject.getRawPhotograph();
				return new Runnable() {
					@Override
					public void run() {
						IMethodInfo setPhotographMethod = ReflectionUIUtils.findInfoByName(containingType.getMethods(),
								"setPhotograph");
						invoke(object, new InvocationData(object, setPhotographMethod, oldPhotograph),
								setPhotographMethod, containingType);
					}
				};
			} else if ((object instanceof IdPhotoProject) && method.getName().startsWith("addFaceFrame")
					&& method.getName().endsWith("Ratio")) {
				final double ratioOffset = (double) invocationData.getParameterValue(0);
				return new Runnable() {
					@Override
					public void run() {
						InvocationData oppositeInvocationData = new InvocationData(object, method, -ratioOffset);
						method.invoke(object, oppositeInvocationData);
					}
				};
			} else {
				return super.getNextInvocationUndoJob(method, containingType, object, invocationData);
			}
		}

		@Override
		protected void save(ITypeInfo type, Object object, OutputStream out) {
			if (object instanceof IdPhotoProject) {
				Analytics.track("Saving(" + object + ")");
				((IdPhotoProject) object).save(out);
			} else {
				super.save(type, object, out);
			}
		}

		@Override
		protected void load(ITypeInfo type, Object object, InputStream in) {
			if (object instanceof IdPhotoProject) {
				Analytics.track("Loading(" + object + ")");
				((IdPhotoProject) object).load(in);
			} else {
				super.load(type, object, in);
			}
		}

		@Override
		protected void setValue(Object object, Object value, IFieldInfo field, ITypeInfo containingType) {
			if (object instanceof IdPhotoProject) {
				Analytics.track("Setting(" + object + " - " + field.getName() + ")", String.valueOf(value));
			}
			super.setValue(object, value, field, containingType);
		}

		@Override
		protected Object getValue(Object object, IFieldInfo field, ITypeInfo containingType) {
			if (object instanceof IdPhotoProject) {
				if (field.getName().equals("detailedPrintPreview")) {
					Analytics.track("Getting(" + object + " - " + field.getName() + ")");
				}
			}
			return super.getValue(object, field, containingType);
		}

		@Override
		protected Object invoke(Object object, InvocationData invocationData, IMethodInfo method,
				ITypeInfo containingType) {
			if (object instanceof IdPhotoProject) {
				Analytics.track("Invoking " + method.getName() + "(", invocationData.toString() + ") on " + object);
			}
			return super.invoke(object, invocationData, method, containingType);
		}

	};

	/**
	 * Project-specific {@link CustomizedUI} implementation.
	 */
	public static final CustomizedUI REFLECTIONUI = new CustomizedUI() {
		@Override
		protected ITypeInfo getTypeInfoBeforeCustomizations(ITypeInfo type) {
			type = TYPEFACTORY.wrapTypeInfo(type);
			return type;
		}
	};

	/**
	 * Project-specific {@link SwingRenderer} implementation.
	 */
	public static final SwingRenderer RENDERER = DESIGN_MODE ? createDesignModeRenderer() : createClassicRenderer();

	/**
	 * @return A project-specific {@link SwingRenderer} implementation allowing to
	 *         generate and edit the GUI.
	 */
	private static CustomizedSwingRenderer createDesignModeRenderer() {
		String cutomizationsFilePath = "src/main/resources/" + IdPhotoUI.class.getPackage().getName().replace('.', '/')
				+ "/default.icu";
		try {
			return (CustomizedSwingRenderer) Class.forName("xy.reflect.ui.control.swing.customizer.SwingCustomizer")
					.getConstructor(CustomizedUI.class, String.class).newInstance(REFLECTIONUI, cutomizationsFilePath);
		} catch (Exception e) {
			throw new UnexpectedError(e);
		}
	}

	/**
	 * @return A project-specific {@link SwingRenderer} implementation allowing to
	 *         just generate the GUI.
	 */
	private static CustomizedSwingRenderer createClassicRenderer() {
		return new CustomizedSwingRenderer(REFLECTIONUI) {
			{
				try {
					getInfoCustomizations().loadFromStream(IdPhotoUI.class.getResourceAsStream("default.icu"), null);
				} catch (Throwable t) {
					throw new AssertionError(t);
				}
			}

			@Override
			public WindowManager createWindowManager(Window window) {
				return new WindowManager(this, window) {

					@Override
					public void set(Component content, Accessor<List<Component>> toolbarControlsAccessor, String title,
							Image iconImage) {
						super.set(content, toolbarControlsAccessor, title, iconImage);
						if (content != null) {
							if (SwingRendererUtils.isForm(content, swingRenderer)) {
								Form form = (Form) content;
								Object object = form.getObject();
								if (object instanceof EncapsulatedObjectFactory.Instance) {
									object = ((EncapsulatedObjectFactory.Instance) object).getValue();
									form = SwingRendererUtils.findFirstObjectDescendantForm(object, form, RENDERER);
									if (object instanceof IdPhotoProject) {
										manageIdPhotoWindowLifeCycle(window, form, object);

									}
								}
							}
						}
					}

					void manageIdPhotoWindowLifeCycle(final Window window, final Form form, final Object object) {
						window.addWindowListener(new WindowAdapter() {

							@Override
							public void windowOpened(WindowEvent e) {
								Analytics.track("windowOpened(" + object + ")");
							}

							@Override
							public void windowClosing(WindowEvent e) {
								Analytics.track("windowClosing(" + object + ")");
								ITypeInfo type = REFLECTIONUI.getTypeInfo(REFLECTIONUI.getTypeInfoSource(object));
								MenuModel menuModel = type.getMenuModel();
								final SaveMenuItem[] saveMenu = new SaveMenuItem[1];
								menuModel.visit(new Visitor<IMenuElement>() {
									@Override
									public boolean visit(IMenuElement e) {
										if (e instanceof SaveMenuItem) {
											saveMenu[0] = (SaveMenuItem) e;
											return false;
										}
										return true;
									}
								});
								if (!saveMenu[0].isFileSynchronized(form, RENDERER)) {
									String objectTitle = RENDERER.getObjectTitle(object);
									if (RENDERER.openQuestionDialog(form,
											"The '" + objectTitle
													+ "' has been modified.\nSave changes before closing?",
											RENDERER.getObjectTitle(object))) {
										saveMenu[0].execute(form, RENDERER);
									}
								}
							}
						});
					}

				};
			}

			@Override
			public void handleExceptionsFromDisplayedUI(Component activatorComponent, Throwable t) {
				Analytics.track("Uncaught Exception", t.toString() + " " + Arrays.toString(t.getStackTrace()));
				if (new ReflectionUIError(t).toString()
						.contains(new ReflectionUIError(new OutOfMemoryError()).toString())) {
					String errorMessage = "The maximum memory is not enough to perform the requested action.";
					if (System.getProperty(Main.LAUNCHER_PROPERTY_KEY) == null) {
						errorMessage += "\nRestart the program with more memory by using this command (for example):"
								+ "\njava -Xmx1024M phoyo-id.jar";
					}
					t = new ReflectionUIError(errorMessage);
				}
				super.handleExceptionsFromDisplayedUI(activatorComponent, t);
			}

		};
	}

}
