/*
 * This file is part of the petitwork package.
 * (c) 2007-2008 Exbridge,inc. <info@exbridge.jp>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

//===================================================================================================
//require library
// http://www.json.org/json2.js
// http://www.mts.net/~tfriesen/dhtml/ie_methods.js (for firefox)
//===================================================================================================

/**
 * Project: petitwork: the PHP lightweight web framework
 * File:    xbpwContext.php
 *
 * @link http://exbridge.jp/
 * @author S.Tajima <tajima@exbridge.jp>
 * @version svn:$Id:$
 * @copyright 2007-2008 Exbridge,Inc.
 */

//----------------------------------------------------------------------------------------------------
// <input type="text" 
//        id="user_nm" 
//        name="user_nm" 
//        validate="len,0,255;iif{type:'not-null',target:['user_cd',..],check:['null',..]};" 
//        column="ユーザー名">
//----------------------------------------------------------------------------------------------------

//===================================================================================================
// エラーハンドル方法
// true ：エラー発生時点で終了
// false：全てのエラーをチェック
//===================================================================================================
var ERROR_RETURN = true;

//===================================================================================================
// 警告方法
// ALERT：alertでの警告
// OTHER：ラベル表示（スタイル「error」を定義してください）
//===================================================================================================
var ERROR_DISPLAY_TYPE = 'ALERT';

//===================================================================================================
// 使用不可の文字・外字
//===================================================================================================
var EXTERNAL_FONTS = /[<>\\'"①②③④⑤⑥⑦⑧⑨⑩⑪⑫⑬⑭⑮⑯⑰⑱⑲⑳ⅠⅡⅢⅣⅤⅥⅦⅧⅨⅩ㍉㌔㌢㍍㌘㌧㌃㌶㍑㍗㌍㌦㌣㌫㍊㌻㎜㎝㎞㎎㎏㏄㎡㍻〝〟№㏍℡㊤㊥㊦㊧㊨㈱㈲㈹㍾㍽㍼≒≡∫∮∑√⊥∠∟⊿∵∩∪]/;

//===================================================================================================
//エラーメッセージ
//===================================================================================================
var VALIDATE_ERR_01='必須項目です。入力してください[%s]';
var VALIDATE_ERR_02='半角数字で入力してください[%s]';
var VALIDATE_ERR_03='半角英数字で入力してください[%s]';
var VALIDATE_ERR_04='半角文字で入力してください[%s]';
var VALIDATE_ERR_05='全角文字で入力してください[%s]';
var VALIDATE_ERR_07='半角文字%s文字以内または全角文字%s文字以内で入力してください[%s]';
var VALIDATE_ERR_08='%s文字以内で入力してください[%s]';
var VALIDATE_ERR_09='%s文字以上%s文字以内で入力してください[%s]';
var VALIDATE_ERR_11='使用できない文字が含まれています[%s]';
var VALIDATE_ERR_14='日付形式ではありません[%s]';
var VALIDATE_ERR_16='整数部%s桁、小数部%s桁以内で入力してください[%s]';
var VALIDATE_ERR_18='%s～%sの範囲で入力してください[%s]';

//===================================================================================================
//Check type
//===================================================================================================
//'sql'
//'null'
//'external'
//'range'
//'len'
//'len-return'
//'byte'
//'decimal'
//'number'
//'number-sign'
//'han'
//'han_eisu'
//'zen'
//'zen-rtn'
//'date'

/**
 * エラーメッセージ保持用配列
 */
var _error_list=[];

/**
 * 検証
 */
function validate(target_form_id)
{
    auto_check_flg = true;
    user_check_flg = true;

    _clsChecker.init(target_form_id);

    //一般的なチェック
    auto_check_flg = _clsChecker.check();

    //ユーザー拡張チェック
    try {
        user_check_flg = user_validate();
    }
    catch(e) {
    }

    //エラー判定
    if (auto_check_flg==false || user_check_flg==false) {
        print_stack_error();
        return false;
    }
    return true;
}

/**
 * エラースタック
 */
function push_error(message)
{
    _error_list[_error_list.length] = message;
}

/**
 * エラー表示
 */
function print_stack_error()
{
    var errmessage = "";
    var errcount = _error_list.length;
    for(var i = 0; i < errcount; i++){
        errmessage += _error_list[i] + "\n";
    }
    if (errcount > 0) {
        if (ERROR_DISPLAY_TYPE=='ALERT') {
            alert(errmessage)
        }
    }
    _error_list=[];
}

/**
 * SQLチェック
 */
function check_sql(v, c)
{
    if (c==undefined) {
        c = '';
    }
    if (v.match(/[<>&'%_"\\]/g)) {
        var message = VALIDATE_ERR_11;
        push_error(message.replace(/%s/, c));
        return false;
    }
    return true;
}

/**
 * NULLチェック
 */
function check_null(v, c)
{
    if (c==undefined) {
        c = '';
    }
    if ((v == '') || (v == null) || (v == undefined) || (v.length == 0)) {
        var message = VALIDATE_ERR_01;
        push_error(message.replace(/%s/, c));
        return false;
    }
    return true;
}

/**
 * 特殊文字（外字）チェック
 */
function check_external(v, c)
{
    if (c==undefined) {
        c = '';
    }
    if (v.match(EXTERNAL_FONTS)) {
        var message = VALIDATE_ERR_11;
        push_error(message.replace(/%s/, c));
        return false;
    }
    return true;
}

/**
 * 数字(小数)チェック
 * @param allLen  桁数制限値(整数部＋小数部) 0の時、桁数チェックなし
 * @param decLen  桁数制限値(小数部)
 */
function check_decimal(v, allLen, decLen, c) {
    if (c==undefined) {
        c = ''
    }
    if (!is_null(v)) {
        return true;
    }
    if (v == 0) {
        return true;
    }
    if (isNaN(v) || !v.match(/^[0-9.]+$/)) {
        var message = VALIDATE_ERR_02;
        push_error(message.replace(/%s/, c));
        return false;
    }
    if (allLen > 0) {
        intLen = allLen - decLen;
        var r = /^([-+]?[0-9]+)\.?([0-9]*)$/;
        var a = r.exec(v);
        if (a[1].length > intLen || a[2].length > decLen) {
            var message = VALIDATE_ERR_16;
            push_error(message.replace(/%s/, intLen).replace(/%s/, decLen).replace(/%s/, c));
            return false;
        }
    }
    return true;
}

/**
 * 数字チェック１
 */
function check_number(v, c)
{
    if (c==undefined) {
        c = '';
    }
    if (is_null(v)) {
        return true;
    }
    if (v == 0) {
        return true;
    }
    v = v.replace(/^0+/,"");
    if (!v.match(/^[1-9]+[0-9]*$/)) {
        var message = VALIDATE_ERR_02;
        push_error(message.replace(/%s/, c));
        return false;
    }
    return true;
}

/**
 * 数字チェック２
 */
function check_number_with_sign(v, c)
{
    if (c==undefined) {
        c = '';
    }
    if (is_null(v)) {
        return true;
    }
    if (v == 0) {
        return true;
    }
    v = v.replace(/^0+/,"");
    if (!v.match(/^[-]?[1-9]+[0-9]*$/)) {
        var message = E_10002;
        push_error(message.replace(/%s/, c));
        return false;
    }
    return true;
}

/**
 * 半角チェック
 */
function check_hankaku(v, c)
{
    if (c==undefined) {
        c = '';
    }
    if (is_null(v)) {
        return true;
    }
    if (!v.match(/^[-+"\\\[\]0-9a-zA-Zｱ-ﾝﾞﾟｦｧ-ｯｰ!*/,.;:@&$%#`^~_(){}?=| ]+$/g)){
        var message = VALIDATE_ERR_04;
        push_error(message.replace(/%s/, c));
        return false;
    }
    return true;
}

/**
 * 半角英数字チェック
 */
function check_hankaku_eisu(v, c)
{
    if (c==undefined) {
        c = '';
    }
    if (is_null(v)) {
        return true;
    }
    if (!v.match(/^[ a-zA-Z0-9.()]+$/)) {
        var message = VALIDATE_ERR_03;
        push_error(message.replace(/%s/, c));
        return false;
    }
    return true;
}

/**
 * 全角チェック
 */
function check_zenkaku(v, c)
{
    return _check_zenkaku(v, c, 0);
}

/**
 * 改行を含む全角チェック
 */
function check_zenkaku_return(v, c)
{
    return _check_zenkaku(v, c, 1);
}

/**
 * 全角チェック（メイン）
 */
function _check_zenkaku(v, c, flg)
{
    if (c==undefined) {
        c = '';
    }
    for (var i = 0; i < v.length; ++i) {
        var cc = v.charCodeAt(i);
        if (flg == 1) {
            if (cc == 0x0d || cc == 0x0a) {
                continue;
            }
        }
        if (cc < 256 || (cc >= 0xff61 && cc <= 0xff9f)) {
            var message = VALIDATE_ERR_05;
            push_error(message.replace(/%s/, c));
            return false;
        }
    }
    return true;
}

/**
 * 値の範囲チェック
 */
function check_range(v, lmin, lmax, c)
{
    if (c==undefined) {
        c = '';
    }
    if(v < lmin || v > lmax){
        var message = VALIDATE_ERR_18;
        push_error(message.replace(/%s/, lmin).replace(/%s/, lmax).replace(/%s/, c));
        return false;
    }
    return true;
}

/**
 * lengthチェック
 */
function check_length(v, lmin, lmax, c)
{
    if (c==undefined) {
        c = '';
    }
    if (is_null(v)) {
        return true;
    }
    var l = v.length;
    if (l < lmin || l > lmax) {
        if (lmin==0) {
            var message = VALIDATE_ERR_08;
            message = message.replace(/%s/, lmax);
        }
        else {
            var message = VALIDATE_ERR_09;
            message = message.replace(/%s/, lmin).replace(/%s/, lmax);
        }
        push_error(message.replace(/%s/, c));
        return false;
    }
    return true;
}

/**
 * テキストエリアバイト数チェック(改行を含む全角)
 */
function check_length_return(v, lmax, c)
{
    if (c==undefined) {
        c = '';
    }
    if (is_null(v)) {
        return true;
    }
    var len = v.length;

    var count = 0;
    for (i=0; i<len; i++) {
        var cc = v.charAt(i);
        if (cc == '\n') {
        }
        else {
            count++;
        }
    }
    if (lmax < count) {
        var message = VALIDATE_ERR_08;
        push_error(message.replace(/%s/, lmax).replace(/%s/, c));
        return false;
    }
    return true;
}

/**
 * バイト数チェック
 */
function check_byte(v, byteLen, c)
{
    if (c==undefined) {
        c = '';
    }
    var count_byteCheck=0;
    for (i_byteCheck=0; i_byteCheck<v.length; i_byteCheck++) {
        char = escape(v.charAt(i_byteCheck));
        //改行コードCR(%0D)またはLF(%0A)の場合、CRLF(%0D%0A)と判断する
        if (char == '%0D' || char == '%0A') {
            char = '%0D%0A';
        }
        (escape(char).length< 4)?count_byteCheck++:count_byteCheck+=2;
    }
    if (count_byteCheck > byteLen) {
        var message = VALIDATE_ERR_07;
        var harfLen = parseInt(byteLen/2);
        push_error(message.replace(/%s/, byteLen).replace(/%s/, harfLen).replace(/%s/, c));
        return false;
    }
    return true;
}

/**
 * 日付チェック
 */
function check_date(v, c)
{
    if (c==undefined) {
        c = '';
    }
    return _check_date_exist(v, c);
}

/**
 * 日付存在チェック(yyyymmdd)
 */
function _check_date_exist(v, c)
{
    v = '' + v;
    if (!v.match(/^\d{4}\/\d{2}\/\d{2}$/) && !v.match(/^\d{8}$/)) {
        var message = VALIDATE_ERR_14;
        push_error(message.replace(/%s/, c));
        return false;
    }
    v = v.replace(/\//g, '');

    var year  = parseInt(v.substr(0, 4), 10);
    var month = parseInt(v.substr(4, 2), 10);
    var day   = parseInt(v.substr(6, 2), 10);
    var ret   = true;

    if (year < 1900) {
        // 1900年より過去は簡易版チェック
        if (month < 1 || month > 12 || day < 1 || day > 31) {
            var message = VALIDATE_ERR_14;
            push_error(message.replace(/%s/, c));
            return false;
        }
    }
    else {
        var dates = new Date(year, month-1, day);
        if (dates.getYear() < 1900) {
            if (year != dates.getYear()+1900) {
                ret = false;
            }
        }
        else {
            if (year != dates.getYear()) {
                ret = false;
            }
        }
        if (month != dates.getMonth()+1) {
            ret = false;
        }
        if (day   != dates.getDate()) {
            ret = false;
        }
        if (!ret) {
            var message = VALIDATE_ERR_14;
            push_error(message.replace(/%s/, c));
            return false;
        }
    }
    return true;
}

function is_null(v)
{
    if ((v == null) || (v == undefined) || (v.length == 0) || (v.value == '')) {
        return true;
    }
    return false;
}

//====================================================================================================================================---

function _cls_checker()
{
    this.target;
    this.check_list = [];
}
function _cls_checker_iif()
{
    this.type;
    this.target_list = [];
}
function _cls_checker_obj()
{
    this.error = true;
    this.target;
    this.name;
    this.value;
    this.column;
    this.iif;
    this.check_params = [];
    this.name_tmp;
}

//
//無名関数は、ECMAScriptの規格に合わないらしいので本当は良くない書き方
//
_cls_checker_obj.prototype = {
    check: function() {
        var enable = true;
        //iif,checked,id
        var iif_obj = this.iif;
        if (iif_obj != undefined && iif_obj != null) {
            for(var i=0; i < iif_obj.target_list.length; i++) {
                var target_nm = iif_obj.target_list[i];
                var doc_obj;
                if (this.target==undefined) {
                    doc_obj = document.getElementsByName(target_nm);
                }
                else {
                    doc_obj = document.forms[this.target].document.getElementsByName(target_nm);
                }
                if (doc_obj[0].name==undefined) {
                    enable = false;
                    continue;
                }
                var target_vl = doc_obj[0].getAttribute('value');
                var target_ck = doc_obj[0].checked;
                enable = this.is_iif(iif_obj.type, target_vl, target_ck);
            }
        }
        if (enable) {
            if (this.name_tmp != this.name) {
                this.name_tmp = this.name
                if (ERROR_DISPLAY_TYPE!='ALERT') {
                    this.non_display();
                }
            }
            for (var i = 0;i<this.check_params.length;i++) {
                var _check = this.check_params[i].split(",");
                var chktype = _check[0].replace(/^\s+|\s+$/g, "");
                if (chktype==''||chktype==undefined) {
                    continue;
                }
                switch(chktype) {
                case 'sql':
                    if (!check_sql(this.value, this.column)) {
                        this.error = false;
                    }
                    break;
                case 'null':
                    if (!check_null(this.value, this.column)) {
                        this.error = false;
                    }
                    break;
                case 'external':
                    if (!check_external(this.value, this.column)) {
                        this.error = false;
                    }
                    break;
                case 'range':
                    if (_check[1]==''||_check[1]==undefined
                        || _check[2]==''||_check[2]==undefined) {
                        alert('range check parameter error!!');
                        this.error = false;
                        break;
                    }
                    var para1 = _check[1].replace(/^\s+|\s+$/g, "");
                    var para2 = _check[2].replace(/^\s+|\s+$/g, "");
                    if (!check_range(this.value, para1, para2, this.column)) {
                        this.error = false;
                    }
                    break;
                case 'len':
                    if (_check[1]==''||_check[1]==undefined
                        || _check[2]==''||_check[2]==undefined) {
                        alert('len check parameter error!!');
                        this.error = false;
                        break;
                    }
                    var para1 = _check[1].replace(/^\s+|\s+$/g, "");
                    var para2 = _check[2].replace(/^\s+|\s+$/g, "");
                    if (!check_length(this.value, para1, para2, this.column)) {
                        this.error = false;
                    }
                    break;
                case 'len-return':
                    if (_check[1]==''||_check[1]==undefined
                        || _check[2]==''||_check[2]==undefined) {
                        alert('len-return check parameter error!!');
                        this.error = false;
                        break;
                    }
                    var para1 = _check[1].replace(/^\s+|\s+$/g, "");
                    var para2 = _check[2].replace(/^\s+|\s+$/g, "");
                    if (!check_length_return(this.value, para1, para2, this.column)) {
                        this.error = false;
                    }
                    break;
                case 'byte':
                    if (_check[1]==''||_check[1]==undefined) {
                        alert('byte check parameter error!!');
                        this.error = false;
                        break;
                    }
                    var para1 = _check[1].replace(/^\s+|\s+$/g, "");
                    if (!check_byte(this.value, para1, this.column)) {
                        this.error = false;
                    }
                    break;
                case 'decimal':
                    if (_check[1]==''||_check[1]==undefined
                        || _check[2]==''||_check[2]==undefined) {
                        alert('decimal check parameter error!!');
                        this.error = false;
                        break;
                    }
                    var para1 = _check[1].replace(/^\s+|\s+$/g, "");
                    var para2 = _check[2].replace(/^\s+|\s+$/g, "");
                    if (!check_decimal(this.value, para1, para2, this.column)) {
                        this.error = false;
                    }
                    break;
                case 'number':
                    if (!check_number(this.value, this.column)) {
                        this.error = false;
                    }
                    break;
                case 'number-sign':
                    if (!check_number_with_sign(this.value, this.column)) {
                        this.error = false;
                    }
                    break;
                case 'han':
                    if (!check_hankaku(this.value, this.column)) {
                        this.error = false;
                    }
                    break;
                case 'han_eisu':
                    if (!check_hankaku_eisu(this.value, this.column)) {
                        this.error = false;
                    }
                    break;
                case 'zen':
                    if (!check_zenkaku(this.value, this.column)) {
                        this.error = false;
                    }
                    break;
                case 'zen-rtn':
                    if (!check_zenkaku_return(this.value, this.column)) {
                        this.error = false;
                    }
                    break;
                case 'date':
                    if (!check_date(this.value, this.column)) {
                        this.error = false;
                    }
                    break;
                default:
                    break;
                }
                if (!this.error) {
                    if (ERROR_DISPLAY_TYPE!='ALERT') {
                        var messages = _error_list[_error_list.length-1];
                        this.display(messages);
                    }
                }
            }
        }
        return this.error;
    },
    is_iif: function(type,val,chk) {
        var enable = true;
        switch(type) {
            case 'checked':
                if (!chk) {
                    enable = false;
                }
                break;
            case 'unchecked':
                if (chk) {
                    enable = false;
                }
                break;
            case 'null':
                if (val != '') {
                    enable = false;
                }
                break;
            case 'not-null':
                if (val == '') {
                    enable = false;
                }
                break;
        }
        return enable;
    },
    is_error: function() {
        return this.error;
    },
    display: function(msg) {
        var obj = document.getElementsByName(this.name);
        var err = document.getElementById('err_'+this.name);
        if (err==null||err==undefined) {
            obj[0].insertAdjacentHTML('AfterEnd', "<p id=err_" + this.name + " class=error>" + msg + "</p>");
        }
        else {
            err.innerHTML = msg;
        }
    },
    non_display: function() {
        var obj = document.getElementsByName(this.name);
        var err = document.getElementById('err_'+this.name);
        if (err!=null && err!=undefined) {
            err.innerHTML = '';
            err.removeNode();
        }
    }
}

_cls_checker.prototype = {
    init: function(target_form_id) {
        this.clear_check_list();
        this.target = target_form_id;
        if (target_form_id==undefined) {
            for (var i=0; i<document.forms.length; i++) {
                var forms = document.forms[i];
                this._init(forms);
            }
        }
        else {
            var forms = document.forms[target_form_id];
            this._init(forms);
        }
    },
    _init: function(frm) {
        for (var i=0; i<frm.length; i++) {
            var type  = frm.elements[i].getAttribute('type');
            var name  = frm.elements[i].getAttribute('name');
            var value = frm.elements[i].value;
            //var value = frm.elements[i].getAttribute('value');
            var valid = frm.elements[i].getAttribute('validate');
            var colum = frm.elements[i].getAttribute('column');
            if (name==undefined || name=='') {
                continue;
            }
            if (valid != undefined && valid != null) {
                //iif{}を取得
                var iif_list = valid.match(/iif\{[^}]+\};*/gi);
                if (iif_list != null) {
                    for(var j=0; j<iif_list.length; j++) {
                        var iif_str = iif_list[j];
                        //valid(str)からiif条件(str)を削除
                        valid = valid.replace(iif_list[j], "");
                        //{}で囲まれた中身を取得
                        var iif_params = iif_str.match(/\{[^}]+\}/i);
                        try {
                            //var iif_obj = eval("("+iif_params[0]+")");
                            var iif_obj = JSON.parse(iif_params[0]);
                        }
                        catch(e) {
                            alert('json convert error!!');
                            break;
                        }
                        var clsCheckerIif = new _cls_checker_iif();
                        clsCheckerIif.type = iif_obj.type;
                        clsCheckerIif.target_list = iif_obj.target_list;
                        //
                        var clsChecker = new _cls_checker_obj();
                        clsChecker.target = this.target;
                        clsChecker.name = name;
                        clsChecker.value = value;
                        clsChecker.column = colum;
                        clsChecker.iif = clsCheckerIif;
                        clsChecker.check_params = iif_obj.check_list;
                        this.add_check_list(clsChecker);
                    }
                }
                //「;」で区切る
                var clsChecker = new _cls_checker_obj();
                clsChecker.target = this.target;
                clsChecker.name = name;
                clsChecker.value = value;
                clsChecker.column = colum;
                if (valid!=undefined && valid!=null) {
                    clsChecker.check_params =  valid.split(";");
                }
                this.add_check_list(clsChecker);
            }
        }
        return true;
    },
    add_check_list: function(checker) {
        this.check_list[this.check_list.length] = checker;
    },
    clear_check_list: function() {
        this.check_list = [];
    },
    check: function() {
        this.error = true;
        var chk_len = this.check_list.length;
        if (chk_len > 0) {
            for (var i=0; i < chk_len; i++) {
                var clsChecker = this.check_list[i];
                if (i>0){
                    clsChecker.name_tmp = this.check_list[i-1].name;
                }
                clsChecker.check();
                if (!clsChecker.is_error()) {
                    this.error = false;
                    if (ERROR_RETURN) {
                        return this.error;
                    }
                }
            }
        }
        return this.error;
    },
    is_error: function() {
        return this.error;
    }
}
var _clsChecker = new _cls_checker();

