#
# Sample class for Win32::Dokan
#
package Mirror;

use strict;

use base qw(Win32::Dokan::FS);
use Win32::Dokan::Const qw(:attribute);

use Fcntl qw(:DEFAULT);

use Carp;
use Win32::Dokan::Const qw(-compile :error);

sub new {
    my $class = shift;
    my $dir = shift;

    $dir =~ s/\//\\/g;
    bless {
	root_dir => $dir,
    }, $class;
}

sub encoding {
    return "X-MBCS";
    # return "CP932";
}

sub full_path {
    my $self = shift;
    my $path = shift;

    if ($path =~ /^[\\\/]/) {
	return $self->{root_dir} . $path;
    }
    else {
	return $self->{root_dir} . "/" . $path;
    }
}

sub open {
    my ($self, $path, $mode, $fileinfo) = @_;

    if ($mode & O_TRUNC) {
	CORE::open(my $fh, "> " . $self->full_path($path)) or return undef;
	binmode($fh);
	$fileinfo->context($fh);
    }
    else {
	CORE::open(my $fh, "+< " . $self->full_path($path)) or return undef;
	binmode($fh);
	$fileinfo->context($fh);
    }

    return 1;
}

sub create {
    my ($self, $path, $mode, $fileinfo) = @_;

    CORE::open(my $fh, "> " . $self->full_path($path)) or return undef;
    binmode($fh);
    $fileinfo->context($fh);

    return 0;
}

sub truncate {
    my ($self, $path, $length, $fileinfo) = @_;

    my $fh = $fileinfo->context;
    unless (CORE::truncate($fh, $length)) {
	return -1;
    }

    return 0;
}

sub opendir {
    my ($self, $path, $fileinfo) = @_;

    if (-d $self->full_path($path)) {
	return 0;
    }

    return -1*Win32::Dokan::Const::ERROR_FILE_NOT_FOUND;
}

sub mkdir {
    my ($self, $path, $fileinfo) = @_;

    if (CORE::mkdir($self->full_path($path))) {
	return 0;
    }
    return -1;
}

sub close {
    my ($self, $path, $fileinfo) = @_;

    # OK: true
    # NG: false
    my $fh = $fileinfo->context;
    if ($fh) {
	CORE::close($fh) or warn "close error: $!";
    }

    return 1;
}

sub read {
    my ($self, $path, $offset, $length, $fileinfo) = @_;

    my $fh = $fileinfo->context;

    unless (sysseek($fh, $offset, 0)) {
	return undef;
    }

    my $buf;
    my $ret = CORE::sysread($fh, $buf, $length);
    return undef unless defined($ret);


    return $buf;
}

sub write {
    my ($self, $path, $offset, $data, $fileinfo) = @_;

    my $fh = $fileinfo->context;

    unless (sysseek($fh, $offset, 0)) {
	return undef;
    }
    syswrite($fh, $data) or return undef;

    return length($data);
}

sub flush {
    my ($self, $path, $fileinfo) = @_;
    # print STDERR "fs flush: $path\n";
    return 0;
}

sub stat {
    my ($self, $path, $fileinfo) = @_;

    my $attr = -d $self->full_path($path)
	? FILE_ATTRIBUTE_DIRECTORY
	: FILE_ATTRIBUTE_NORMAL;

    my @s = stat($self->full_path($path));
    return undef unless (@s);

    my ($dev,$ino,$mode,$nlink,$uid,$gid,$rdev,$size,
	$atime,$mtime,$ctime,$blksize,$blocks) = @s;
    
    return [$size, $attr, $ctime, $atime, $mtime];
}

sub readdir {
    my ($self, $path, $fileinfo) = @_;
    CORE::opendir(my $dh, $self->full_path($path)) or return -1;
    my @ret = readdir($dh);
    closedir($dh);

    return \@ret;
}

sub setattr {
    my ($self, $path, $attr, $fileinfo) = @_;

    # print STDERR "setattr for $path to $attr\n";

    return 0;
}

sub utime {
    my ($self, $path, $ctime, $atime, $mtime, $fileinfo) = @_;

    if (CORE::utime($atime, $mtime, $self->full_path($path))) {
	return 0;
    }
    return -1;
}

sub remove {
    my ($self, $path, $fileinfo) = @_;

    if ($fileinfo->context) {
	close($fileinfo->context);
    }
    if (CORE::unlink($self->full_path($path))) {
	return 0;
    }
    return -1;
}

sub rename {
    my ($self, $path, $newpath, $fileinfo) = @_;

    if (CORE::rename($self->full_path($path), $self->full_path($newpath))) {
	return 0;
    }
    return -1;
}

sub rmdir {
    my ($self, $path, $fileinfo) = @_;

    if (CORE::rmdir($self->full_path($path))) {
	return 0;
    }
    return -1;
}

sub unmount {
    my ($self, $fileinfo) = @_;
    return;
}

sub volume_information {
    my ($self, $fileinfo) = @_;

    my ($volume_name, $serial, $component_length, $file_system_flags, $file_system_name);

    $volume_name = "dokan - mirror";

    return ($volume_name, $serial, $component_length, $file_system_flags, $file_system_name);
}

1;
