#
# Dokan constants
#
package Win32::Dokan::Const;

=head1 NAME

Win32::Dokan::Const - Win32::Dokan constants

=head1 SYNOPSIS

  # make the constants availabe only (not import in your namespace)
  use Win32::Dokan::Const qw(-compile :all);

  # or specified symbol only
  use Win32::Dokan::Const qw(-compile FILE_ATTRIBUTE_NORMAL);

  # normal import
  use Win32::Dokan::Const qw(:all);

  # or specified symbol only
  use Win32::Dokan::Const qw(DOKAN_SUCCESS);


=head1 DESCRIPTION

This package contains Win32::Dokan specific constats.
(and some Win32 constants for convenience.)

=head2 EXPORT

None by default.

=head2 Exportable constants

=head3 ":dokan" - dokan specific constats

  DOKAN_DRIVER_INSTALL_ERROR
  DOKAN_DRIVE_LETTER_ERROR
  DOKAN_ERROR
  DOKAN_MOUNT_ERROR
  DOKAN_START_ERROR
  DOKAN_SUCCESS


=head3 ":attribute" - Win32 constats for file attributes

  FILE_ATTRIBUTE_READONLY
  FILE_ATTRIBUTE_HIDDEN
  FILE_ATTRIBUTE_SYSTEM
  FILE_ATTRIBUTE_DIRECTORY
  FILE_ATTRIBUTE_ARCHIVE
  FILE_ATTRIBUTE_DEVICE
  FILE_ATTRIBUTE_NORMAL
  FILE_ATTRIBUTE_TEMPORARY
  FILE_ATTRIBUTE_SPARSE_FILE
  FILE_ATTRIBUTE_REPARSE_POINT
  FILE_ATTRIBUTE_COMPRESSED
  FILE_ATTRIBUTE_OFFLINE
  FILE_ATTRIBUTE_NOT_CONTENT_INDEXED


=head3  ":fs" - Win32 constats for filesystem or device

  FS_CASE_IS_PRESERVED
  FS_CASE_SENSITIVE
  FS_UNICODE_STORED_ON_DISK
  FS_PERSISTENT_ACLS
  FS_FILE_COMPRESSION
  FS_VOL_IS_COMPRESSED
  FILE_NAMED_STREAMS
  FILE_SUPPORTS_ENCRYPTION
  FILE_SUPPORTS_OBJECT_IDS
  FILE_SUPPORTS_REPARSE_POINTS
  FILE_SUPPORTS_SPARSE_FILES
  FILE_VOLUME_QUOTAS


=head3 ":iomode" - Win32 constats for file I/O

  GENERIC_READ
  GENERIC_WRITE
  GENERIC_EXECUTE
  CREATE_NEW
  CREATE_ALWAYS
  OPEN_EXISTING
  OPEN_ALWAYS
  TRUNCATE_EXISTING
  FILE_SHARE_READ
  FILE_SHARE_WRITE
  FILE_SHARE_DELETE


=head3 ":error" - Win32 constats to identify error

  ERROR_INVALID_FUNCTION
  ERROR_FILE_NOT_FOUND
  ERROR_PATH_NOT_FOUND
  ERROR_TOO_MANY_OPENFILES
  ERROR_ACCESS_DENIED
  ERROR_INVALID_BLOCK
  ERROR_INVALID_ACCESS
  ERROR_WRITE_PROTECT
  ERROR_NOT_READY
  ERROR_SEEK
  ERROR_WRITE_FAULT
  ERROR_READ_FAULT
  ERROR_GEN_FAILURE
  ERROR_LOCK_VIOLATION
  ERROR_NOT_SUPPORTED
  ERROR_DISK_FULL
  ERROR_DIR_NOT_EMPTY
  ERROR_PATH_BUSY
  ERROR_ALREADY_EXISTS
  ERROR_FILE_EXISTS
  ERROR_DRIVE_LOKED
  ERROR_NOT_LOCKED
  ERROR_LOCK_FAILED
  ERROR_LOCKED

=cut

use Carp;
use Win32::Dokan;

require Exporter;

our @ISA = qw(Exporter);


our %EXPORT_TAGS = ( 'dokan' => [ qw(DOKAN_DRIVER_INSTALL_ERROR
				     DOKAN_DRIVE_LETTER_ERROR
				     DOKAN_ERROR
				     DOKAN_MOUNT_ERROR
				     DOKAN_START_ERROR
				     DOKAN_SUCCESS)],

		     'attribute' => [ qw(FILE_ATTRIBUTE_READONLY
					 FILE_ATTRIBUTE_HIDDEN
					 FILE_ATTRIBUTE_SYSTEM
					 FILE_ATTRIBUTE_DIRECTORY
					 FILE_ATTRIBUTE_ARCHIVE
					 FILE_ATTRIBUTE_DEVICE
					 FILE_ATTRIBUTE_NORMAL
					 FILE_ATTRIBUTE_TEMPORARY
					 FILE_ATTRIBUTE_SPARSE_FILE
					 FILE_ATTRIBUTE_REPARSE_POINT
					 FILE_ATTRIBUTE_COMPRESSED
					 FILE_ATTRIBUTE_OFFLINE
					 FILE_ATTRIBUTE_NOT_CONTENT_INDEXED)],

		     'fs' => [ qw(FS_CASE_IS_PRESERVED
				  FS_CASE_SENSITIVE
				  FS_UNICODE_STORED_ON_DISK
				  FS_PERSISTENT_ACLS
				  FS_FILE_COMPRESSION
				  FS_VOL_IS_COMPRESSED
				  FILE_NAMED_STREAMS
				  FILE_SUPPORTS_ENCRYPTION
				  FILE_SUPPORTS_OBJECT_IDS
				  FILE_SUPPORTS_REPARSE_POINTS
				  FILE_SUPPORTS_SPARSE_FILES
				  FILE_VOLUME_QUOTAS) ],

		     'iomode' => [ qw(GENERIC_READ
				      GENERIC_WRITE
				      GENERIC_EXECUTE
				      CREATE_NEW
				      CREATE_ALWAYS
				      OPEN_EXISTING
				      OPEN_ALWAYS
				      TRUNCATE_EXISTING
				      FILE_SHARE_READ
				      FILE_SHARE_WRITE
				      FILE_SHARE_DELETE) ],

		     'error' => [ qw(ERROR_INVALID_FUNCTION
				     ERROR_FILE_NOT_FOUND
				     ERROR_PATH_NOT_FOUND
				     ERROR_TOO_MANY_OPENFILES
				     ERROR_ACCESS_DENIED
				     ERROR_INVALID_BLOCK
				     ERROR_INVALID_ACCESS
				     ERROR_WRITE_PROTECT
				     ERROR_NOT_READY
				     ERROR_SEEK
				     ERROR_WRITE_FAULT
				     ERROR_READ_FAULT
				     ERROR_GEN_FAILURE
				     ERROR_LOCK_VIOLATION
				     ERROR_NOT_SUPPORTED
				     ERROR_DISK_FULL
				     ERROR_DIR_NOT_EMPTY
				     ERROR_PATH_BUSY
				     ERROR_ALREADY_EXISTS
				     ERROR_FILE_EXISTS
				     ERROR_DRIVE_LOKED
				     ERROR_NOT_LOCKED
				     ERROR_LOCK_FAILED
				     ERROR_LOCKED ) ],
 );

{
    my %seen;

    push @{$EXPORT_TAGS{all}},
	 grep {!$seen{$_}++} @{$EXPORT_TAGS{$_}} foreach keys %EXPORT_TAGS;

}

our @EXPORT_OK = @{$EXPORT_TAGS{all}};

sub AUTOLOAD {
    # This AUTOLOAD is used to 'autoload' constants from the constant()
    # XS function.

    my $constname;
    our $AUTOLOAD;
    ($constname = $AUTOLOAD) =~ s/.*:://;
    croak "&Win32::Dokan::Const::constant not defined" if $constname eq 'constant';
    my ($error, $val) = constant($constname);
    if ($error) { croak $error; }
    {
	no strict 'refs';
	# Fixed between 5.005_53 and 5.005_61
#XXX	if ($] >= 5.00561) {
#XXX	    *$AUTOLOAD = sub () { $val };
#XXX	}
#XXX	else {
	    *$AUTOLOAD = sub { $val };
#XXX	}
    }
    goto &$AUTOLOAD;
}

sub import {
    my $class = shift;
    if ($_[0] && $_[0] eq '-compile') {
	$class->_compile_symbols(@_);
    }
    else {
	unshift @_, $class;
	goto &Exporter::import;
    }
}

sub _compile_symbols {
    my $class = shift;

    shift;
    my @symbols;
    for (@_) {
	my $tag = $_;
	if ($tag =~ s/^://) {
	    if ($EXPORT_TAGS{$tag}) {
		push(@symbols, @{$EXPORT_TAGS{$tag}});
	    }
	    else {
		croak "$tag is not exported tag";
	    }
	}
	else {
	    if (defined(EXPORT_OK{$tag})) {
		push(@symbols, $tag);
	    }
	    else {
		croak "$tag is not exported";
	    }
	}
    }
    my $x = join(";", map { "sub $_" } @symbols);
    eval $x;
}

1;
