/*
 * "Peko" Visual Novel System
 *
 * All Rights Reserved.
 * Copyright (c) 1999-2003 Tsukuba Bunko.
 *
 * $Id: SceneProcessor.java,v 1.1.2.2 2003/12/11 10:16:15 ppoi Exp $
 */
package tsukuba_bunko.peko.scenario;

import	java.io.InputStream;

import	java.net.URL;

import	javax.xml.parsers.ParserConfigurationException;
import	javax.xml.parsers.SAXParser;
import	javax.xml.parsers.SAXParserFactory;

import	org.xml.sax.Attributes;
import	org.xml.sax.InputSource;
import	org.xml.sax.SAXException;
import	org.xml.sax.SAXParseException;
import	org.xml.sax.helpers.DefaultHandler;

import	tsukuba_bunko.peko.ActionControler;
import	tsukuba_bunko.peko.Logger;
import	tsukuba_bunko.peko.PekoSystem;

import	tsukuba_bunko.peko.resource.ResourceManager;

import	tsukuba_bunko.peko.scenario.select.SelectCoordinator;

import	tsukuba_bunko.peko.scenario.stage.StageCoordinator;

import	tsukuba_bunko.peko.scenario.text.TextCoordinator;

import	tsukuba_bunko.peko.session.Session;


/**
 * V[̏svZbTW[łB
 * @author	$Author: ppoi $
 * @version	$Revision: 1.1.2.2 $
 */
public class SceneProcessor	extends DefaultHandler	implements Runnable	{

	/**
	 * PSML 1.0 Scene DTD Public ID
	 */
	public static final String	PSML_PUBLICID = "-//Tsukuba Bunko//DTD PSML Scene 1.0//EN";

	/**
	 * PSML 1.0 Scene DTD System ID
	 */
	public static final String	PSML_SYSTEMID = "http://softlab.tsukuba-bunko.org/dtd/psml10-scene.dtd";


	/**
	 * PSML ͂s SAX Parser
	 */
	protected SAXParser	_parser = null;

	/**
	 * ԃtO
	 */
	protected boolean	_running = false;

	/**
	 * ~tO
	 */
	protected boolean	_aborted = false;

	/**
	 * ̃V[
	 */
	protected URL	_sceneURL = null;

	/**
	 * ̃V[̃ReNXg
	 */
	protected SceneContext	_sceneContext = null;

	/**
	 *  SceneProcessor 𐧌䂷 ScenarioProcessor
	 */
	protected ScenarioProcessor	_owner = null;

	/**
	 * ݂̃c[̃x
	 */
	protected int	_level = -1;

	/**
	 * ElementHandler ̃Tuc[̊Jnx
	 */
	protected int	_processStartedLevel = -1;

	/**
	 * ]ȂTuc[̊Jnx
	 */
	protected int	_ignoreStartedLevel = -1;

	/**
	 * nhWXg
	 */
	protected HandlerRegistry	_registry = null;

	/**
	 * ݗvf ElementHandler
	 */
	protected ElementHandler	_handler = null;

	/**
	 * X^[g̃m[hʒu
	 */
	protected SceneContext.Node	_startNode = null;


	/**
	 * TextCanvas R[fBl[^
	 */
	protected TextCoordinator	_textCoordinator = null;

	/**
	 * StageCanvas R[fBl[^
	 */
	protected StageCoordinator	_stageCoordinator = null;

	/**
	 * SelectCanvas R[fBl[^
	 */
	protected SelectCoordinator	_selectCoordinator = null;


	/**
	 * <code>SceneProcessor</code> ̃CX^X𐶐܂B
	 * @param	owner	 SceneProcessor CX^X𐧌䂷 ScenarioProcessor
	 */
	public SceneProcessor( ScenarioProcessor owner )
	{
		super();
		_owner = owner;
		_registry = HandlerRegistry.newInstance();
		getSAXParser();
		Logger.debug( "[scenario] create SceneProcessor." );
	}


	/**
	 * V[ReNXg擾܂B
	 * @return	V[ReNXg
	 */
	public SceneContext getSceneContext()
	{
		return _sceneContext;
	}


	/**
	 * nhWXg擾܂B
	 * @return	nhWXg
	 */
	public HandlerRegistry getHandlerRegistry()
	{
		return _registry;
	}

	/**
	 * TextCanvas R[fBl[^ݒ肵܂B
	 * @param	coordinator	TextCanvas R[fBl[^
	 */
	public void setTextCoordinator( TextCoordinator coordinator )
	{
		_textCoordinator = coordinator;
	}	

	/**
	 * TextCanvas R[fBl[^擾܂B
	 * @return	TextCanvas R[fBl[^
	 */
	public TextCoordinator getTextCoordinator()
	{
		return _textCoordinator;
	}

	/**
	 * StageCanvas R[fBl[^ݒ肵܂B
	 * @param	coordinator	StageCanvas R[fBl[^
	 */
	public void setStageCoordinator( StageCoordinator coordinator )
	{
		_stageCoordinator = coordinator;
	}

	/**
	 * StageCanvas R[fBl[^擾܂B
	 * @return	StageCanvas R[fBl[^
	 */
	public StageCoordinator getStageCoordinator()
	{
		return _stageCoordinator;
	}

	/**
	 * SelectCanvas R[fBl[^ݒ肵܂B
	 * @param	coodinator	SelectCanvas R[fBl[^
	 */
	public void setSelectCoordinator( SelectCoordinator coordinator )
	{
		_selectCoordinator = coordinator;
	}

	/**
	 * SelectCanvas R[fBl[^擾܂B
	 * @return	SelectCanvas R[fBl[^
	 */
	public SelectCoordinator getSelectCoordinator()
	{
		return _selectCoordinator;
	}


	/**
	 * V[̏Jn܂B
	 * @param	sceneName	V[̃V[
	 */
	public void process( String sceneName, Session session )
	{
		if( _running )	{
			IllegalStateException	ise = new IllegalStateException( "scene processor is still running." );
			Logger.fatal( MessageIDs.SCN0007F );
			PekoSystem.showErrorDialog( MessageIDs.SCN0007F.getMessage(), ise.fillInStackTrace(), true );
		}
		_sceneURL = getSceneURL( sceneName );
		_sceneContext = new SceneContext( sceneName, session, this );
		_startNode = _sceneContext.getLastCommittedNode();
		if( _startNode != null )	{
			Logger.debug( "[scenario.scene] start from: " + _startNode.getPath() );
		}

		Thread	thread = new Thread( this );
		session.setSceneContext( _sceneContext, thread );

		_textCoordinator.prepare( _sceneContext, thread );
		_stageCoordinator.prepare( _sceneContext, thread );
		_selectCoordinator.prepare( _sceneContext, thread );

		_aborted = false;
		thread.start();
	}

	/**
	 * V[̏𒆎~܂B
	 */
	public void abort()
	{
		_aborted = true;
		_textCoordinator.dormantize();
		_stageCoordinator.dormantize();
		_selectCoordinator.dormantize();
		Logger.debug( "[scenario] release stopped thread." );
		ActionControler	controler = PekoSystem.getInstance().getActionControler();
		controler.start();
	}

	/**
	 * V[~Ă邩ǂ𔻒肵܂B
	 * @return	V[~Ăꍇ <code>true</code>AȊȌꍇ <code>false</code>
	 */
	public boolean isAborted()
	{
		return _aborted;
	}

	/**
	 * PSML Scene f[^̍\͂s SAX p[U擾܂B
	 * @return	PSML Scene f[^̍\͂s SAX p[U
	 */
	protected SAXParser getSAXParser()
	{
		if( _parser == null )	{
			try	{
				SAXParserFactory	factory = SAXParserFactory.newInstance();
				factory.setNamespaceAware( true );
				if( "on".equals(System.getProperty("debug", "off")) )	{
					Logger.debug( "[scenario.scene] set PSML validating on." );
					factory.setValidating( true );
				}
				else	{
					Logger.debug( "[scenario.scene] set PSML validating off." );
					factory.setValidating( false );
				}
				_parser = factory.newSAXParser();
			}
			catch( SAXException se )	{
				Logger.fatal( MessageIDs.SCN0000F );
				PekoSystem.showErrorDialog( MessageIDs.SCN0000F.getMessage(), se, true );
			}
			catch( ParserConfigurationException pce )	{
				Logger.fatal( MessageIDs.SCN0000F );
				PekoSystem.showErrorDialog( MessageIDs.SCN0000F.getMessage(), pce, true );
			}
		}
		return _parser;
	}

	/**
	 * V[f[^ URL 擾܂B
	 * @param	scene	V[
	 * @return	V[f[^ URL
	 */
	protected URL getSceneURL( String scene )
	{
		ResourceManager	resources = ResourceManager.getInstance();
		URL	sceneDir = resources.getLocationResources().getScenesDirecotryURL();
		try	{
			return new URL( sceneDir, scene );
		}
		catch( Exception e )	{
			Logger.fatal( MessageIDs.SCN0006F, e );
			PekoSystem.showErrorDialog( MessageIDs.SCN0006F.getMessage(), e, true );
			return null;
		}
	}

//
//	Runnable ̎
//
	public void run()
	{
		_running = true;
		try	{
			Logger.debug( _sceneURL.toString() );
			SAXParser	parser = getSAXParser();
			parser.parse( _sceneURL.toString(), this );
		}
		catch( Exception e )	{
			Logger.fatal( MessageIDs.SCN0005F, new Object[]{_sceneContext.getCurrentPath()}, e );
			PekoSystem.showErrorDialog( MessageIDs.SCN0005F.getMessage(new Object[]{_sceneContext.getCurrentPath()}), e, true );
		}
		_running = false;
		_owner.pushSceneProcessor( this );
	}


//
//	ContentHandler ̎
//
	public void startDocument()
		throws SAXException
	{
		if( _aborted )	{
			return;
		}

		_level  = 0;
		_processStartedLevel = -1;
		_ignoreStartedLevel = -1;
		_handler = null;
	}

	public void endDocument()
		throws SAXException
	{
		if( _aborted )	{
			return;
		}
		else	{
			_owner.sceneEnded( this );
		}
	}

	public void startElement( String namespaceURI, String localName, String qName, Attributes attrs )
		throws SAXException
	{
		if( _aborted )	{
			return;
		}

		++_level;
		_sceneContext.pushNode( qName );

		if( _startNode != null )	{
			if( _sceneContext.isCurrentNode(_startNode) )	{
				_startNode = null;
			}
			else	{
				return;
			}
		}
		else if( _ignoreStartedLevel != -1 )	{
			return;
		}
		else if( !PSMLUtil.isEvaluatable(attrs, _sceneContext) )	{
			_ignoreStartedLevel = _level;
			return;
		}

		if( _handler == null )	{
			_handler = _registry.getElementHandler( namespaceURI, localName );
			if( _handler == null )	{
				return;
			}
			else	{
				_sceneContext.saveCurrentNode();
				_handler.setSceneContext( _sceneContext );
				_processStartedLevel = _level;
				_handler.startDocument();
			}
		}
		try	{
			_handler.startElement( namespaceURI, localName, qName, attrs );
		}
		catch( Exception e )	{
			Logger.fatal( MessageIDs.SCN0005F, new Object[]{_sceneContext.getCurrentPath()}, e );
			PekoSystem.showErrorDialog( MessageIDs.SCN0005F.getMessage(new Object[]{_sceneContext.getCurrentPath()}), e, true );
		}
	}

	public void endElement( String namespaceURI, String localName, String qName )
		throws SAXException
	{
		if( _aborted )	{
			return;
		}

		if( _startNode != null )	{
			//NOOP
		}
		else if( _level == _ignoreStartedLevel )	{
			_ignoreStartedLevel = -1;
		}
		else if( _ignoreStartedLevel != -1 )	{
			//NOOP
		}
		else if( _handler != null )	{
			try	{
				_handler.endElement( namespaceURI, localName, qName );
				if( _level == _processStartedLevel )	{
					_handler.endDocument();
					if( _handler.isEndOfScene() )	{
						_owner.sceneEnded( this );
						_aborted = true;
					}
					_handler = null;
					_processStartedLevel = -1;
				}
			}
			catch( Exception e )	{
				Logger.fatal( MessageIDs.SCN0005F, new Object[]{_sceneContext.getCurrentPath()}, e );
				PekoSystem.showErrorDialog( MessageIDs.SCN0005F.getMessage(new Object[]{_sceneContext.getCurrentPath()}), e, true );
			}
		}

		--_level;
		_sceneContext.popNode();
	}

	public void characters( char[] ch, int begin, int length )
		throws SAXException
	{
		if( _aborted )	{
			return;
		}

		if( (_ignoreStartedLevel != -1 ) || (_handler == null) )	{
			return;
		}
		else if( _handler != null )	{
			_handler.characters( ch, begin, length );
		}
	}

	public void processingInstruction( String target, String data )
	{
	}

//
//	ErrorHandler ̎
//
	public void warning( SAXParseException exception )
	{
		Logger.warn( MessageIDs.SCN0002W, new Object[]{exception.getMessage()}, exception );
	}

	public void error( SAXParseException exception )
	{
		Logger.error( MessageIDs.SCN0003E, new Object[]{exception.getMessage()}, exception );
	}

	public void fatalError( SAXParseException exception )
	{
		Logger.fatal( MessageIDs.SCN0004F, new Object[]{exception.getMessage()}, exception );
		PekoSystem.showErrorDialog( "fail to parsing scene data.", exception, true );
	}


//
//	EntityResolver ̎
//
	public InputSource resolveEntity( String publicId, String systemId )
	{
		boolean	isPSMLScene = false;
		if( publicId != null )	{
			if( SceneProcessor.PSML_PUBLICID.equals(publicId) )	{
				isPSMLScene = true;
			}
		}
		else if( systemId != null )	{
			if( SceneProcessor.PSML_SYSTEMID.equals(systemId) )	{
				isPSMLScene = true;
			}
		}

		if( isPSMLScene )	{
			InputStream	is = null;
			try	{
				is = getClass().getClassLoader().getResourceAsStream( "psml10-scene.dtd" );
			}
			catch( Exception e )	{
				Logger.warn( MessageIDs.SCN0001W, e );
				return null;
			}

			if( is != null )	{
				return new InputSource( is );
			}
			else	{
				Logger.warn( MessageIDs.SCN0001W );
				return null;
			}
		}
		else	{
			return null;
		}
	}
}
