/*
 * Paraselene
 * Copyright (c) 2009-2011  Akira Terasaki
 * このファイルは同梱されているLicense.txtに定めた条件に同意できる場合にのみ
 * 利用可能です。
 */
package paraselene.supervisor;


import java.util.*;
import java.util.concurrent.locks.*;
import java.io.*;
import paraselene.*;
import paraselene.ajax.data.*;
import paraselene.ajax.*;
import paraselene.ui.*;
import paraselene.tag.*;


class CallStack {
	private static final long serialVersionUID = 2L;
	Child	child;
	private Page	dialog;
	Forward	fw;
	volatile Object	result = null;
	CallStack( Child base, Page call, EphemeralPosition p, Tag hint ) {
		child = base;
		dialog = call.getParentPage();
		fw = new EphemeralPopup( dialog, p, hint );
	}
	boolean isMyself( Child ch ) {
		return child == ch;
	}
	boolean isMyself( Page p ) {
		p = p.getParentPage();
		Option.trace( "isMyself:dialog(%s) <-> p(%s)", dialog.getUniqueKey(), p.getUniqueKey() );
		return dialog.getUniqueKey().equals( p.getUniqueKey() );
	}
}

/**
 * 画面遷移履歴。<br>
 * 現在のポップアップ表示中ページも保持します。
 * ポップアップページは履歴には入りません。
 * <br>スレッドセーフです。
 */
public class History implements Serializable {
	private static final long serialVersionUID = 2L;
	private ArrayList<Page>	hist = new ArrayList<Page>();
	private int	histry_no;
	private ReentrantLock	lock = new ReentrantLock();
	private ArrayList<Page>	popup = new ArrayList<Page>();
	volatile boolean drop_f = false;
	volatile String	session_id;

	private transient LinkedList<CallStack>	callstack = null;

	void callStack( Child base, Page call, EphemeralPosition p, Tag hint ) {
		Option.trace( "callStack:%s %s", base, call.getID() );
		synchronized( callstack ) {
			callstack.push( new CallStack( base, call, p, hint ) );
		}
	}
	Object popStack( Child base ) {
		synchronized( callstack ) {
			while( callstack.size() > 0 ) {
				CallStack	cs = callstack.pop();
				if ( cs.isMyself( base ) ) {
					Object	ret = cs.result;
					if ( ret == null )	return EphemeralPage.NO_ANSWER;
					return ret;
				}
				if ( cs.child == null )	continue;
				synchronized( cs.child ) {
					unlock();
					cs.child.notifyAll();
				}
			}
		}
		return EphemeralPage.ABORT;
	}
	private void deadStack( Page p ) {
		if ( hist.contains( p ) )	return;
		Option.traceWithStack( "deadStack: %s", p.getID() );
		synchronized( callstack ) {
			boolean	flag = false;
			for ( CallStack	cs: callstack ) {
				if ( flag = cs.isMyself( p ) )	break;
			}
			if ( !flag )	return;
			while( callstack.size() > 0 ) {
				CallStack	cs = callstack.pop();
				flag = cs.isMyself( p );
				if ( cs.child == null )	continue;
				synchronized( cs.child ) {
					unlock();
					cs.child.notifyAll();
				}
				if ( flag )	return;
			}
		}
	}
	private Child continueStack( Page p, Object result ) {
		Option.traceWithStack( "continueStack:%s / return %s", p.getID(), result );
		synchronized( callstack ) {
			boolean	flag = false;
			for ( CallStack	cs: callstack ) {
				if ( flag = cs.isMyself( p ) )	break;
			}
			if ( !flag ) {
				Option.trace( "continueStack null return" );
				return null;
			}
			while( callstack.size() > 0 ) {
				CallStack	cs = callstack.pop();
				if ( cs.child == null )	continue;
				if ( cs.isMyself( p ) ) {
					cs.result = result;
					callstack.push( cs );
					return cs.child;
				}
				synchronized( cs.child ) {
					unlock();
					cs.child.notifyAll();
				}
			}
		}
		Option.trace( "continueStack null return" );
		return null;
	}
	private Forward returnStack( Closure fw ) throws Throwable {
		Page	page = ((Closure)fw).close_page;
		Object	result = ( fw instanceof EphemeralClosure )?
			((EphemeralClosure)fw).result_data:	null;
		Child	child = continueStack( page, result );
		if ( child != null )	return Forward.merge( fw, SandBox.input_sub( child, false, this ) );
		return fw;
	}
	Forward returnStack( Forward fw, boolean call_f ) throws Throwable {
		if ( callstack == null )	callstack = new LinkedList<CallStack>();
		if ( call_f && fw instanceof Closure )	return returnStack( (Closure)fw );
		if ( fw.isAjaxHold() ) {
			ArrayList<Forward>	list = new ArrayList<Forward>();
			for ( AjaxForward af: fw.getAjaxForward( true ) ) {
				list.add( (Forward)af );
				if ( af instanceof Closure ) list.add( returnStack( (Closure)af, call_f ) );
			}
			return Forward.merge( list.toArray( new Forward[0] ) );
		}
		return fw;
	}
	Forward getStackForward( Child c ) {
		synchronized( callstack ) {
			for ( CallStack cs:	callstack ) {
				if ( cs.isMyself( c ) )	return cs.fw;
			}
		}
		return null;
	}

	private void comet_stop( Page p ) {
		StringBuilder	buf = new StringBuilder( "paraselene.comet_stop('" );
		buf = buf.append( p.getUniqueKey() );
		buf = buf.append( "');" );
		Ajax.addComet( session_id, new PostBack( new Exec( buf.toString() ) ) );
	}

	boolean tryLock() {
		if ( lock.isHeldByCurrentThread() )	return true;
		Option.trace( "call tryLock " );
		boolean	ret = lock.tryLock();
		Option.trace( "tryLock %s ", ret  );
		return ret;
	}

	void lock() {
		if ( lock.isHeldByCurrentThread() )	return;
		Option.trace( "call lock " );
		lock.lock();
		Option.trace( "locked " );
	}

	void unlock() {
		if ( !lock.isHeldByCurrentThread() )	return;
		Option.trace( "call unlock " );
		lock.unlock();
		Option.trace( "unlocked " );
	}

	/**
	 * コンストラクタ。
	 */
	private History() {
		callstack = new LinkedList<CallStack>();
	}
	History( int n, String s ) {
		this();
		histry_no = n;
		session_id = s;
	}

	/**
	 * サテライトページ追加。
	 * @param page 追加ページ。
	 */
	public void addSatellitePage( Page page ) {
		if ( page == null )	return;
		synchronized( popup ) {
			String	key = page.getUniqueKey();
			for ( Page p: popup ) {
				if ( p.getUniqueKey().equals( key ) )	return;
			}
			setInfo( page );
			popup.add( page );
		}
		Option.trace( "addSatellitePage %d %s", hashCode(), page.getUniqueKey() );
	}

	private void initSatellitePage( Page parent ) {
		if ( parent == null )	return;
		PageBypassedDiv[]	div = parent.getPageBypassedDiv();
		for ( int i = 0; i < div.length; i++ ) {
			initSatellitePage( div[i].getBypassPage() );
		}
		if ( parent != parent.getParentPage() ) {
			addSatellitePage( parent );
		}
	}

	/**
	 * サテライトページ削除。
	 * @param page 削除ページ。
	 */
	public void removeSatellitePage( Page page ) {
		if ( page == null )	return;
		if ( page.getAjaxSupport() == Page.AjaxSupport.SERVER_PUSH ) {
			comet_stop( page );
		}
		removePopup( page );
	}

	private boolean addPopup( Page page ) {
		if ( page.getPopupType() == null )	return false;
		addSatellitePage( page );
		return true;
	}

	void removePopup( Page page ) {
		String	key = page.getUniqueKey();
		synchronized( popup ) {
			int	cnt = popup.size();
			for ( int i = 0; i < cnt; i++ ) {
				if ( popup.get( i ).getUniqueKey().equals( key ) ) {
					deadStack( popup.remove( i ) );
					Option.trace( "removePopup %s", key );
					return;
				}
			}
		}
		Option.trace( "donot remove, not found %s", key );
	}

	private void clearPopup() {
		synchronized( popup ) {
			for ( Page p: popup )	deadStack( p );
			popup.clear();
			Option.traceWithStack( "popup clear" );
		}
	}

	/**
	 * 追加。<br>
	 * ポップアップページは履歴に追加されません。
	 * 新しく画面遷移が発生すると、ポップアップ情報はクリアされます。
	 * @param page 追加するページ。
	 */
	public void add( Page page ) {
		if ( page == null )	return;
		page = page.getParentPage();
		if ( addPopup( page ) )	return;
		synchronized( hist ) {
			int	cnt = getHistoryCount();
			Page	old = null;
			if ( cnt > 0 ) {
				old = getPage();
			}
			if ( page.isHistoryClear() ) {
				for ( int i = 0; i < cnt; i++ ) {
					Page	h = hist.get( i );
					if ( page.equals( h ) ) {
						for ( int j = cnt -1; j >= i; j-- ) {
							Page	tmp = hist.remove( j );
							tmp.setHistoryKey( HistorySet.ROOT );
							if ( page != tmp )	deadStack( tmp );
						}
						break;
					}
				}
			}
			if ( page.isAllowHistoryAdd() ) {
				setInfo( page );
				hist.add( page );
				page.setHistoryKey( histry_no );
			}
			cnt = getHistoryCount();
			if ( cnt == 0 || old != getPage() ) {
				if ( cnt > 0 && old != null ) {
					Option.trace(
						String.format( "old=%s(%s) / new=%s(%s)",
							old.getID().toString(), old.getUniqueKey(),
							getPage().getID().toString(),
							getPage().getUniqueKey()
						)
					);
				}
				clearPopup();
				initSatellitePage( getPage() );
			}
		}
	}

	private void setInfo( Page page ) {
		if ( page.getPageServerInformation() != null )	return;
		Page	old = getPage();
		if ( old == null )	return;
		page.setPageServerInformation( old.getPageServerInformation() );
	}

	/**
	 * 履歴の件数を取得。
	 * @return 履歴件数。
	 */
	public int getHistoryCount() {
		return hist.size();
	}

	/**
	 * 履歴を取得。指定ページと同じものを履歴から取得します。
	 * @param id 指定ページ。
	 * @return 指定ページと同じページの配列。無ければ0個の配列。
	 */
	public Page[] getAllPage( PageID id ) {
		if ( id == null ) return new Page[0];
		ArrayList<Page>	list = new ArrayList<Page>();
		synchronized( hist ) {
			int	cnt = getHistoryCount();
			for ( int i = 0; i < cnt; i++ ) {
				Page	h = hist.get( i );
				if ( h.equals( id ) ) {
					int	hld = list.size();
					boolean	flag = true;
					for ( int j = 0; j < hld; j++ ) {
						if ( list.get( j ).getUniqueKey() == h.getUniqueKey() ) {
							flag = false;
							break;
						}
					}
					if ( flag )	list.add( h );
				}
			}
		}
		return list.toArray( new Page[0] );
	}

	/**
	 * 履歴を取得。指定ページと同じものを履歴から取得します。
	 * @param id 指定ページ。
	 * @return 指定ページと同じページ。無ければnullを返す。
	 */
	public Page getPage( PageID id ) {
		Page[]	p = getAllPage( id );
		if ( p.length == 0 )	return null;
		return p[p.length - 1];
	}

	/**
	 * 現在カレントのブラウザで表示されているページ一覧を返します。
	 * @return ポップアップを含めた表示ページ。
	 */
	public Page[] getBrowsingPage() {
		ArrayList<Page>	list = new ArrayList<Page>();
		lock();
		synchronized( popup ) {
			int	cnt = popup.size();
			for ( int i = 0; i < cnt; i++ ){
				list.add( popup.get( i ) );
			}
		}
		Page	p = getPage();
		if ( p != null )	list.add( p );
		return list.toArray( new Page[0] );
	}

	private Page[] getBrowsingPage( Page page, PageID ... id ) {
		Page[]	dmy = new Page[0];
		if ( page == null )	return dmy;
		ArrayList<Page>	list = new ArrayList<Page>();
		PageID	pid = page.getID();
		for ( PageID i: id ) {
			if ( pid.getID() == i.getID() ) {
				list.add( page );
				break;
			}
		}
		for ( PageBypassedDiv div: page.getPageBypassedDiv() ) {
			for ( Page b: getBrowsingPage( div.getBypassPage(), id ) ) {
				list.add( b );
			}
		}
		return list.toArray( dmy );
	}

	/**
	 * 現在カレントのブラウザで表示されているページから、ページIDが一致する
	 * ものを返します。
	 * ポップアップやバイパスページを含めて探します。
	 * @param id 検索するID。
	 * @return 該当ページ。
	 */
	public Page[] getBrowsingPage( PageID ... id ) {
		ArrayList<Page>	list = new ArrayList<Page>();
		for ( Page p: getBrowsingPage() ) {
			for ( Page p2: getBrowsingPage( p, id ) ) {
				initSatellitePage( p2 );
				list.add( p2 );
			}
		}
		return list.toArray( new Page[0] );
	}

	/**
	 * 現在カレントのブラウザで表示されているページから、ユニークキー
	 * ({@link paraselene.Page#getUniqueKey()})が一致するものを返します。
	 * ポップアップやバイパスページを含めて探します。
	 * @param uni ユニークキー。
	 * @return 一致したページ。
	 */
	public Page getBrowsingPage( String uni ) {
		Option.trace( "getBrowsingPage  %d >> %s", hashCode(), uni );
		Page	page = null;
		for ( Page p: getBrowsingPage() ) {
			if ( p.getUniqueKey().equals( uni ) )	return p;
			p = p.getBypassPage( uni );
			if ( p != null ) {
				initSatellitePage( page );
				return p;
			}
		}
		return null;
	}

	/**
	 * 履歴を削除。直前のページを削除すると、ポップアップ情報もクリアされます。
	 * @param last_no
	 * 0:直前のページ ～ getHistoryCount()-1:最古のページ。
	 */
	public void removePage( int last_no ) {
		if ( last_no < 0 )	return;
		if ( last_no == 0 )	clearPopup();
		synchronized( hist ) {
			int	cnt = getHistoryCount();
			for ( int i = cnt - 1; i >= 0; i-- ) {
				if ( last_no == 0 )	{
					Page	old = hist.remove( i );
					old.setHistoryKey( HistorySet.ROOT );
					deadStack( old );
					return;
				}
				last_no--;
			}
		}
		if ( last_no == 0 )	initSatellitePage( getPage() );
	}

	/**
	 * 履歴を削除。指定ページIDなら、全て削除する。
	 * 指定ページIDが直前のページなら、ポップアップ情報もクリアされます。
	 * @param page_id ページID。
	 */
	public void removePage( PageID page_id ) {
		boolean	init_f = false;
		synchronized( hist ) {
			int	cnt = getHistoryCount();
			for ( int i = cnt - 1; i >= 0; i-- ) {
				Page	page = hist.get( i );
				if ( page.getID() == page_id )	{
					Page	old = hist.remove( i );
					old.setHistoryKey( HistorySet.ROOT );
					deadStack( old );
					if ( i == (cnt - 1) ) {
						clearPopup();
						init_f = true;
					}
				}
			}
		}
		if ( init_f )	initSatellitePage( getPage() );
	}

	/**
	 * 履歴を全て削除。
	 * ポップアップ情報もクリアされます。
	 */
	public void removePage() {
		clearPopup();
		synchronized( hist ) {
			int	cnt = getHistoryCount();
			for ( int i = 0; i < cnt; i++ ) {
				Page	old = hist.get( i );
				old.setHistoryKey( HistorySet.ROOT );
				deadStack( old );
			}
			hist.clear();
		}
	}

	/**
	 * 履歴を取得。
	 * @param last_no
	 * 0:直前のページ ～ getHistoryCount()-1:最古のページ。<br>
	 * 範囲外であるとnullを返す。
	 * @return 指定ページの内容。
	 */
	public Page getPage( int last_no ) {
		if ( last_no < 0 )	return null;
		synchronized( hist ) {
			int	cnt = getHistoryCount();
			for ( int i = cnt - 1; i >= 0; i-- ) {
				if ( last_no == 0 )	{
					return hist.get( i );
				}
				last_no--;
			}
		}
		return null;
	}

	/**
	 * 直前のページを取得。getPage(0)と等価。
	 * @return 遷移元ページ。
	 */
	public Page getPage() {
		return getPage( 0 );
	}
}

