/*
 * Paraselene
 * Copyright (c) 2009  Akira Terasaki
 * このファイルは同梱されているLicense.txtに定めた条件に同意できる場合にのみ
 * 利用可能です。
 */
package paraselene.tag.attr;

import java.net.*;
import java.io.*;
import java.util.*;
import paraselene.*;
import paraselene.supervisor.*;


/**
 * HTMLタグの属性(URIを示すもの)。
 */
public class URIAttribute extends Attribute {
	private static final long serialVersionUID = 1L;
	private static final String DOWNLOAD_DIR = "rabbit/";

	private static String[] uri_name = new String[] {
		"href", "codebase", "src", "cite", "action", "profile", "background",
		"code", "data",
	};
	private static HashMap<String,String>	uri_map = new HashMap<String,String>();
	static {
		for ( int i = 0; i < uri_name.length; i++ ) {
			uri_map.put( uri_name[i], uri_name[i] );
		}
	}

	/**
	 * URI属性であるか？
	 * @param n 属性名。
	 * @return true:URI属性、false:URI属性ではない。
	 */
	public static boolean isURI( String n ) {
		return ( uri_map.get( n.toLowerCase( Locale.ENGLISH ) ) != null );
	}

	private URIAttribute() {}

	private String out_enc;

	/**
	 * 複製の作成。
	 * @return 複製。
	 */
	public Attribute getReplica() {
		try {
			return new URIAttribute( getName(), getURI(), out_enc, out_enc );
		}
		catch ( Exception e ) {}
		return null;
	}

	/**
	 * コンストラクタ。URIはdecに従いデコードします。
	 * @param n 属性名。
	 * @param u URI。
	 * @param dec デコード文字コード。
	 * @param enc setEncode()参照。
	 */
	public URIAttribute( String n, URI u, String dec, String enc ) throws UnsupportedEncodingException {
		setName( n );
		setEncode( enc );
		set( u, dec );
	}

	/**
	 * URIのエンコード。
	 * @param uri URI。
	 * @param enc 文字コード。
	 * @return エンコード後の文字列。
	 */
	public static String encode( String uri, String enc ) throws UnsupportedEncodingException {
		if ( enc == null )	return uri;
		String[]	frag = uri.split( "#" );
		if ( frag.length == 0 )	return uri;
		String[]	query = frag[0].split( "\\?" );
		if ( query.length == 0 )	return uri;
		String[]	test = query[0].split( ":" );
		String	scheme = null;
		String	main_path = test[test.length - 1];
		if ( test.length > 1 ) {
			if ( test.length == 2 ) {
				StringBuffer	sc = new StringBuffer( test[0] );
				if ( main_path.indexOf( "//" ) == 0 ) {
					main_path = main_path.substring( 2 );
					sc = sc.append( "://" );
				}
				else {
					sc = sc.append( ":" );
				}
				scheme = sc.toString();
			}
			else {
				StringBuilder	sc = new StringBuilder();
				for ( int i = 0; i < test.length - 1; i++ ) {
					sc = sc.append( test[i] );
					sc = sc.append( ":" );
				}
				scheme = sc.toString();
			}
		}
		else {
			int	len = query[0].length();
			if ( len > 0 ) {
				if ( query[0].charAt( len - 1 ) == ':' ) {
					scheme = query[0];
					main_path = "";
				}
			}
		}
		StringBuffer	buf = new StringBuffer( " " );
		buf = buf.append( main_path );
		buf = buf.append( " " );
		String[]	path = buf.toString().split( "/" );
		buf = new StringBuffer( URLEncoder.encode( path[0], enc ) );
		for ( int i = 1; i < path.length; i++ ) {
			buf = buf.append( "/" );
			buf = buf.append( URLEncoder.encode( path[i], enc ) );
		}
		String	tmp = buf.toString();
		buf = new StringBuffer( tmp.substring( 1, tmp.length() -1 ) );
		if ( query.length > 1 ) {
			buf = buf.append( "?" );
			String[]	param = query[1].split( "&" );
			for ( int i = 0 ;i < param.length; i++ ) {
				if ( i > 0 )	buf = buf.append( "&" );
				String[]	data = param[i].split( "=" );
				buf = buf.append( URLEncoder.encode( data[0], enc ) );
				buf = buf.append( "=" );
				if ( data.length > 1 ) {
					buf = buf.append( URLEncoder.encode( data[1], enc ) );
				}
			}
		}
		if ( frag.length > 1 ) {
			 buf = buf.append( "#" );
			 buf = buf.append( URLEncoder.encode( frag[1], enc ) );
		}
		if ( scheme != null ) {
			StringBuffer	full = new StringBuffer( scheme );
			buf = full.append( buf );
		}
		return buf.toString();
	}

	/**
	 * コンストラクタ。エスケープされていないURIを格納します。
	 * @param n 属性名。
	 * @param u URI。
	 * @param enc setEncode()参照。
	 */
	public URIAttribute( String n, String u, String enc ) throws UnsupportedEncodingException, URISyntaxException {
		this( n, u != null?	new URI( encode( u, enc ) ):	null, enc, enc );
	}

	/**
	 * 属性値の設定。エスケープがあればdecに従いデコードします。
	 * @param u URI。
	 * @param dec デコード文字コード。
	 */
	public void set( URI u, String dec ) throws UnsupportedEncodingException {
		if ( u == null ) {
			super.set( (Text)null );
		}
		else {
			set( URLDecoder.decode( u.toString(), dec ) );
		}
	}

	/**
	 * 属性値の取得。
	 * return 属性値。
	 */
	public URI getURI() {
		if ( super.get() == null )	return null;
		try {
			return new URI( encode( super.getValueString(), out_enc ) );
		}
		catch ( Exception e ) {}
		return null;
	}

	/**
	 * クエリー部分の取得。<br>
	 * http://xxx.com/yyy.cgi?a=1&b=2 ならば、<br>
	 * [0] &quot;a=1&quot;<br>
	 * [1] &quot;b=2&quot;<br>
	 * を返します。エスケープされていない状態で返します。
	 * @return クエリー部分。クエリー部が無ければ0個の配列。
	 */
	public String[] getQuery() {
		URI	uri = getURI();
		String	q = uri.getQuery();
		if ( q == null ) {
			return new String[0];
		}
		return q.split( "&" );
	}

	/**
	 * URIのクエリ設定。元のURIにフラグメントを含んでいる場合、それを引き継ぎます。
	 * @param query getQuery()の戻り値と同じ形式。エスケープしないで下さい。
	 * null を渡すと、0個の配列と同じ意味となります。
	 */
	public void setQuery( String[] query ) {
		set( setQuery( getURI(), query ) );
	}

	/**
	 * 自サイト内ページへのURI生成。
	 * @param pid ページID。
	 * @param fragment フラグメント。#は含めないで下さい。
	 * 不要な場合はnullを指定して下さい。
	 * @param query getQuery()の戻り値と同じ形式。エスケープしないで下さい。
	 * @return URI。相対パスです。
	 */
	public static String pageToURI( PageID pid, String fragment, String ... query ) {
		StringBuffer	buf = new StringBuffer( Supervisor.LINK_DEF[0] );
		buf = buf.append( pid.getID() );
		if ( fragment != null ) {
			buf = buf.append( "#" );
			buf = buf.append( fragment );
		}
		try {
			return setQuery( new URI( buf.toString() ), query );
		}
		catch( Exception e ){
			return null;
		}
	}

	/**
	 * 自サイト内ページへのURI生成。ダウンロード用。
	 * @param pid ページID。
	 * @param filename ブラウザに認識させるファイル名。
	 * @param query getQuery()の戻り値と同じ形式。エスケープしないで下さい。
	 * @return URI。相対パスです。
	 */
	public static String pageToDownloadURI( PageID pid, String filename, String ... query ) {
		StringBuffer	buf = new StringBuffer( DOWNLOAD_DIR );
		buf = buf.append( pageToURI( pid, null ) );
		buf = buf.append( "/" );
		buf = buf.append( filename );
		try {
			return setQuery( new URI( buf.toString() ), query );
		}
		catch( Exception e ){
			return null;
		}
	}

	/**
	 * スキーマ。
	 */
	public enum Scheme {
		HTTP( Supervisor.LINK_DEF[1], "http", 80 ),
		HTTPS( Supervisor.LINK_DEF[2], "https", 443 );
		private static final long serialVersionUID = 1L;
		String	str;
		String	prot;
		int	port;
		private Scheme( String s, String p, int pt ) {
			str = s;
			prot = p;
			port = pt;
		}
		/**
		 * インスタンスの比較。
		 * 渡されたものがStringならば、プロトコル名とみなし比較します。
		 * @param o 比較対象。
		 * @return true:一致、false:不一致。
		 */
		private boolean equalsString( String o ) {
			return prot.equals( o );
		}
		/**
		 * URIのスキーマを返します。
		 * @param uri URI。
		 * @return スキーマ。
		 */
		public static Scheme getScheme( URI uri ) {
			Scheme[]	s = values();
			String	p = uri.getScheme();
			for ( int i = 0; i < s.length; i++ ) {
				if ( s[i].equalsString( p ) )	return s[i];
			}
			return null;
		}
	}
	/**
	 * プロトコル名を伴う絶対パスの生成。
	 * サーブレットコンテナに問い合わせ、ホスト名、コンテキストパスを取得し
	 * 絶対パスを生成します。<br>
	 * 例えば、makeAbsolutePath( URIAttribute.Scheme.HTTP, 0, "abc.html" )は<br>
	 * http://ホスト名/コンテキストパス/abc.html<br>
	 * を生成します。<br>
	 * makeAbsolutePath( URIAttribute.Scheme.HTTPS, 0, pageToURI( pid, null ) )<br>
	 * のように、プロトコルをHTTPSに切り替えたい場合に有用です。
	 * @param prt スキーマ。
	 * @param port ポート。0以下ならばHTTPなら80のように標準的なポートを使用します。
	 * 特殊なポートを指定したい場合に値を設定して下さい。
	 * @param path 相対パス。スキーマやホスト名を含んだ文字列を指定しないで下さい。
	 */
	public static String makeAbsolutePath( Scheme prt, int port, String path ) {
		StringBuffer	buf = new StringBuffer( prt.str );
		if ( port == prt.port )	port = 0;
		buf = buf.append( port );
		if ( path == null )	path = "/";
		if ( path.charAt( 0 ) != '/' ) {
			buf = buf.append( "/" );
		}
		buf = buf.append( path );
		return buf.toString();
	}

	private static String setQuery( URI uri, String[] query ) {
		if ( query == null )	query = new String[0];
		StringBuffer	buf = new StringBuffer();
		String	tmp;
		tmp = uri.getScheme();
		if ( tmp != null ) {
			buf = buf.append( tmp );
			buf = buf.append( "://" );
		}
		tmp = uri.getUserInfo();
		if ( tmp != null ) {
			buf = buf.append( tmp );
			buf = buf.append( "@" );
		}
		tmp = uri.getHost();
		if ( tmp != null ) {
			buf = buf.append( tmp );
		}
		int	port = uri.getPort();
		if ( port != -1 ) {
			buf = buf.append( ":" );
			buf = buf.append( port );
		}
		buf = buf.append( uri.getPath() );
		for ( int i = 0; i < query.length; i++ ) {
			if ( i == 0 ) {
				buf = buf.append( "?" );
			}
			else {
				buf = buf.append( "&" );
			}
			buf = buf.append( query[i] );
		}
		tmp = uri.getFragment();
		if ( tmp != null ) {
			buf = buf.append( "#" );
			buf = buf.append( tmp );
		}
		return buf.toString();
	}

	/**
	 * 出力文字コード指定。
	 * 出力時にエスケープする際の文字コードを設定します。
	 * @param enc 例:&quot;UTF-8&quot;、
	 * &quot;Windows-31J&quot;、&quot;EUC-JP&quot;
	 * @exception UnsupportedEncodingException サポートされない文字コード。
	 */
	public void setEncode( String enc ) throws UnsupportedEncodingException {
		URLEncoder.encode( "テスト", enc );
		out_enc = enc;
	}

	protected String getValueString() {
		try {
			return Supervisor.makeWithSessionURI( super.getValueString(), out_enc );
		}
		catch( Exception e ) {
			return null;
		}
		catch( NoClassDefFoundError nc ) {
			try {
				return URIAttribute.encode( super.getValueString(), out_enc );
			}
			catch( Exception e2 ) {
				return null;
			}
		}
	}

}


