/*
 * Paraselene
 * Copyright (c) 2009  Akira Terasaki
 * このファイルは同梱されているLicense.txtに定めた条件に同意できる場合にのみ
 * 利用可能です。
 */
package paraselene;

import java.net.*;
import java.io.*;
import paraselene.tag.*;
import paraselene.css.*;

/**
 * 色を示す文字列。
 */
public class Color extends PlainText implements AttributeValuable, CSSValuable {
	private static final long serialVersionUID = 2L;
	private int red = -1, green = -1, blue = -1;

	/**
	 * 複製の作成。
	 * @return 複製。
	 */
	public HTMLPart getReplica() {
		return new Color( red, green, blue );
	}

	/**
	 * コンストラクタ。
	 * @param s 文字列。
	 */
	public Color( String s ) {
		setText( s );
	}

	private int toRGB( NumberValue num ) {
		if ( num.getUnit() == NumberValue.Unit.PERCENTAGE ) {
			num.multiply( 256 ).divide( 100 );
		}
		int	v = num.toInteger();
		if ( v > 256 )	v = 255;
		return v;
	}

	private boolean set_rgb_text( String str ){
		String[]	all = str.split( "[\\(\\)]" );
		if ( all.length != 2 )	return false;
		if ( !"rgb".equalsIgnoreCase( all[0].trim() ) )	return false;
		String[]	rgb = all[1].split( "," );
		if ( rgb.length != 3 )	return false;
		NumberValue[]	num = new NumberValue[3];
		for ( int i = 0; i < 3; i++ ) {
			num[i] = new NumberValue( rgb[i] );
		}
		setColor( toRGB( num[0] ), toRGB( num[1] ), toRGB( num[2] ) );
		return true;
	}

	/**
	 * 色の設定。
	 * @param s 文字列。
	 */
	public void setText( String s ) {
		if ( s == null ) {
			super.setText( s );
			return;
		}
		s = s.trim();
		if ( s.isEmpty() ) {
			super.setText( s );
			return;
		}
		if ( set_rgb_text( s ) )	return;
		if ( s.charAt( 0 ) != '#' ) {
			setColor( WebColor.valueOf( s.toUpperCase() ) );
			return;
		}
		char[]	ch = s.substring( 1 ).toCharArray();
		int[]	rgb = new int[3];
		if ( ch.length == 3 ) {
			for ( int i = 0; i < 3; i++ ) {
				StringBuilder	buf = new StringBuilder( ch[i] );
				buf = buf.append( ch[i] );
				rgb[i] = Integer.parseInt( buf.toString(), 16 );
			}
		}
		else {
			for ( int i = 0; i < 3; i++ ) {
				rgb[i] = Integer.parseInt( new String( ch, i * 2, 2 ), 16 );
			}
		}
		setColor( rgb[0], rgb[1], rgb[2] );
	}

	/**
	 * コンストラクタ。
	 * @param wc 規定色。
	 */
	public Color( WebColor wc ) {
		setColor( wc );
	}

	/**
	 * 色の設定。
	 * @param wc 規定色。
	 */
	public void setColor( WebColor wc ) {
		setColor( wc.red, wc.green, wc.blue );
	}

	/**
	 * コンストラクタ。全てのパラメータは下位１バイトのみ取り出して使用する。
	 * @param r 赤(0x00 ～ 0xff)。負数の場合、transparent となります。
	 * @param g 緑(0x00 ～ 0xff)。負数の場合、transparent となります。
	 * @param b 青(0x00 ～ 0xff)。負数の場合、transparent となります。
	 */
	public Color( int r, int g, int b ) {
		setColor( r, g, b );
	}

	/**
	 * 色の設定。全てのパラメータは下位１バイトのみ取り出して使用します。
	 * @param r 赤(0x00 ～ 0xff)。負数の場合、transparent となります。
	 * @param g 緑(0x00 ～ 0xff)。負数の場合、transparent となります。
	 * @param b 青(0x00 ～ 0xff)。負数の場合、transparent となります。
	 */
	public void setColor( int r, int g, int b ) {
		if ( r < 0 || g < 0 || b < 0 ) {
			red = green = blue = -1;
			super.setText( "transparent" );
		}
		else {
			red = r & 0xff;
			green = g & 0xff;
			blue = b & 0xff;
			super.setText( String.format( "#%06X", (red << (8*2)) | (green << 8) | blue ) );
		}
	}

	/**
	 * 赤の取得。
	 * @return 赤。transparent の場合、-1 を返します。
	 */
	public int getRed() {
		return red;
	}

	/**
	 * 緑の取得。
	 * @return 緑。transparent の場合、-1 を返します。
	 */
	public int getGreen() {
		return green;
	}

	/**
	 * 青の取得。
	 * @return 青。transparent の場合、-1 を返します。
	 */
	public int getBlue() {
		return blue;
	}

	private static int unsigned_fix( int v ) {
		if ( v < 0 )	return 0;
		if ( v > 255 )	return 255;
		return v;
	}

	private static int signed_fix( int v ) {
		if ( v < -128 )	return -128;
		if ( v > 127 )	return 127;
		return v;
	}

	/**
	 * YUVの取得。
	 * @return YUV変換した値。
	 */
	public YUV getYUV() {
		double[]	s = new double[] { red, green, blue };
		double[]	d = new double[] {
			0.299 * s[0] + 0.587 * s[1] + 0.114 * s[2],
			-0.169 * s[0] - 0.331 * s[1] + 0.5 * s[2],
			0.5 * s[0] - 0.419 * s[1] - 0.081 * s[2]
		};
		YUV	yuv = new YUV( unsigned_fix( (int)d[0] ), signed_fix( (int)d[1] ), signed_fix( (int)d[2] ) );
		return yuv;
	}

	/**
	 * コンストラクタ。
	 * 全てのパラメータは下位１バイトのみ取り出して使用します。
	 * @param yuv YUV。
	 */
	public Color( YUV yuv ) {
		setColor( yuv );
	}

	/**
	 * 色の設定。
	 * 全てのパラメータは下位１バイトのみ取り出して使用します。
	 * @param yuv YUV。
	 */
	public void setColor( YUV yuv ) {
		double[]	s = new double[] {
			unsigned_fix( yuv.y) , signed_fix( yuv.u ), signed_fix( yuv.v )
		};
		double[]	d = new double[] {
			s[0] + 1.402 * s[2],
			s[0] - 0.344 * s[1] - 0.714 * s[2],
			s[0] + 1.772 * s[1]
		};
		setColor( unsigned_fix( (int)d[0] ), unsigned_fix( (int)d[1] ), unsigned_fix( (int)d[2] ) );
	}
}


