/*
 * Paraselene
 * Copyright (c) 2009  Akira Terasaki
 * このファイルは同梱されているLicense.txtに定めた条件に同意できる場合にのみ
 * 利用可能です。
 */
package paraselene.tag.attr;

import java.io.*;
import java.net.*;
import java.util.*;
import paraselene.*;
import paraselene.tag.*;

/**
 * HTMLタグの属性(汎用的なもの)。
 */
public class Attribute implements Serializable {
	private static final long serialVersionUID = 1L;
	private String	name;
	private Text	value;

	protected Attribute() {}

	/**
	 * 複製の作成。
	 * @return 複製。
	 */
	public Attribute getReplica() {
		return new Attribute( name, (Text)value.getReplica() );
	}

	/**
	 * コンストラクタ。nは小文字に変換される。
	 * @param n 属性名。
	 * @param v 属性値。
	 */
	public Attribute( String n, String v ) {
		setName( n );
		set( v );
	}

	/**
	 * コンストラクタ。nは小文字に変換される。
	 * 値を持たない属性を作成します。
	 * @param n 属性名。
	 */
	public Attribute( String n ) {
		setName( n );
	}

	/**
	 * コンストラクタ。nは小文字に変換される。
	 * @param n 属性名。
	 * @param v 属性値。
	 */
	public Attribute( String n, Text v ) {
		setName( n );
		set( v );
	}

	/**
	 * 属性名の設定。
	 * @param n 属性名。
	 */
	protected void setName( String n ) {
		name = n.toLowerCase( Locale.ENGLISH );
	}

	/**
	 * 属性名の取得。小文字で返す。
	 * return 属性名。
	 */
	public String getName() {
		return name;
	}

	/**
	 * 属性値の設定。
	 * @param v 属性値。
	 */
	public void set( String v ) {
		if ( value != null ) {
			value.setText( v );
			return;
		}
		if ( v == null ) {
			value = null;
		}
		else {
			value = new Text( v, HTMLPart.StringMode.ATTRIBUTE );
		}
	}

	/**
	 * 属性値の設定。
	 * @param v 属性値。
	 */
	public void set( Text v ) {
		value = v;
	}

	/**
	 * 属性値の取得。
	 * @return 属性値。無ければnull。
	 */
	public Text get() {
		return value;
	}

	/**
	 * 属性値の取得。文字列で返します。
	 * @return 属性値。無ければnull。
	 */
	public String getString() {
		Text	t = get();
		if ( t == null )	return null;
		return t.toString();
	}

	/**
	 * 文字列生成。
	 * @return 値を出力用に正規化したもの。
	 */
	protected String getValueString() {
		if ( value == null )	return null;
		return value.toString( HTMLPart.StringMode.ATTRIBUTE );
	}

	/**
	 * 文字列化。
	 * @param xml true:xhtmlである、false:htmlである。
	 * @return 文字列。
	 */
	public String toHtmlString( boolean xml ) {
		StringBuffer	buf = new StringBuffer( name );
		if ( getValueString() != null ) {
			buf = buf.append( "=" );
			buf = buf.append( "\"" );
			buf = buf.append( getValueString() );
			buf = buf.append( "\"" );
		}
		else if ( xml ) {
			buf = buf.append( "=" );
			buf = buf.append( "\"" );
			buf = buf.append( name );
			buf = buf.append( "\"" );
		}
		return buf.toString();
	}

	/**
	 * 文字列化。xxx=xxx形式で返します。
	 * @return 文字列。
	 */
	public String toString() {
		try {
			return toHtmlString ( false );
		}
		catch( Exception e ) {
			return e.toString();
		}
	}
}

