/*
 * Paraselene
 * Copyright (c) 2009  Akira Terasaki
 * このファイルは同梱されているLicense.txtに定めた条件に同意できる場合にのみ
 * 利用可能です。
 */
package paraselene;

import java.net.*;
import java.io.*;
import java.util.*;
import java.text.*;

/**
 * テキスト。<br>
 * 単純な文字列情報を持ちますが、HTMLのタグ種別に合わせ
 * 出力フォーマットを変換します。
 */
public class Text extends HTMLPart {
	private static final long serialVersionUID = 1L;
	private String	text;

	protected Text() {}

	/**
	 * コンストラクタ。StringMode.PLAIN扱い。
	 * @param txt テキスト。
	 */
	public Text( String txt ) {
		this( txt, StringMode.PLAIN );
	}

	/**
	 * コンストラクタ。数値を文字列にします。<BR>
	 * 例えばnew Text( 1234678, &quot;\\#,###&quot; ) は
	 * &quot;\12,345,678&quot;という文字列にして格納します。
	 * @param num 数値。
	 * @param format フォーマット。
	 */
	public Text( long num, String format ) {
		this( new Long( num ), format );
	}

	/**
	 * コンストラクタ。数値を文字列にします。<BR>
	 * 例えばnew Text( 1234678.9, &quot;#,##0.00&quot; ) は
	 * &quot;12,345,678.90&quot;という文字列にして格納します。
	 * @param num 数値。
	 * @param format フォーマット。
	 */
	public Text( double num, String format ) {
		this( new Double( num ), format );
	}

	/**
	 * コンストラクタ。数値を文字列にします。<BR>
	 * 例えばnew Text(new BigInteger( &quot;12345678&quot; ), &quot;\\#,###&quot;) は
	 * &quot;\12,345,678&quot;という文字列にして格納します。
	 * @param num 数値。
	 * @param format フォーマット。
	 */
	public Text( Number num, String format ) {
		this( new DecimalFormat( format ).format( num ) );
	}

	/**
	 * コンストラクタ。日時を文字列にします。<BR>
	 * 例えばnew Text(new Date(), &quot;yyyy/MM/dd HH:mm:ss&quot;) は
	 * &quot;2009/01/01 15:00:00&quot;のような文字列を格納します。
	 * @param date 日時。
	 * @param format フォーマット。
	 */
	public Text( Date date, String format ) {
		this( new SimpleDateFormat( format ).format( date ) );
	}

	/**
	 * コンストラクタ。
	 * @param txt テキスト。
	 * @param mode 入力文字列種別。
	 */
	public Text( String txt, StringMode mode ) {
		setText( txt, mode );
	}

	/**
	 * テキストを設定する。StringMode.PLAIN扱い。
	 * @param txt テキスト。
	 */
	public void setText( String txt ) {
		setText( txt, StringMode.PLAIN );
	}

	/**
	 * テキストを設定する。
	 * @param txt テキスト。
	 * @param mode 入力文字列種別。
	 */
	public void setText( String txt, StringMode mode ) {
		if ( txt == null || mode == StringMode.PLAIN ) {
			text = txt;
			return;
		}

		char[]	line = txt.toCharArray();
		StringBuffer	buf = new StringBuffer();
		StringBuffer	esc = null;
		int	spc_cnt = 0;
		boolean	nbsp_f = false;
		for ( int i = 0; i < line.length; i++ ) {
			if ( line[i] == ' ' ) {
				if ( mode == StringMode.TEXTAREA ) {
					buf = buf.append( ' ' );
					continue;
				}
				spc_cnt++;
				continue;
			}
			if ( spc_cnt > 0 ) {
				if ( mode == StringMode.TABLE ) {
					if ( nbsp_f ) {
						buf = buf.append( ' ' );
						nbsp_f = false;
					}
				}
				else {
					buf = buf.append( ' ' );
				}
				spc_cnt = 0;
			}
			if ( esc != null ) {
				esc = esc.append( line[i] );
				if ( line[i] != ';' ) continue;
				String	amp = esc.toString();
				if ( "&lt;".equalsIgnoreCase( amp ) ) {
					buf = buf.append( '<' );
				}
				else if ( "&gt;".equalsIgnoreCase( amp ) ) {
					buf = buf.append( '>' );
				}
				else if ( "&amp;".equalsIgnoreCase( amp ) ) {
					buf = buf.append( '&' );
				}
				else if ( "&quot;".equalsIgnoreCase( amp ) ) {
					buf = buf.append( '"' );
				}
				else if ( "&nbsp;".equalsIgnoreCase( amp ) ) {
					buf = buf.append( ' ' );
					nbsp_f = true;
				}
				else {
					buf = buf.append( esc );
				}
				esc = null;
				continue;
			}
			nbsp_f = false;
			if ( line[i] == '&' ) {
				esc = new StringBuffer( "&" );
				continue;
			}
			spc_cnt = 0;
			buf = buf.append( line[i] );
			nbsp_f = false;
		}
		if ( esc != null ) {
			buf = buf.append( esc );
		}
		text = buf.toString();
	}

	/**
	 * テキスト化。
	 * @param mode 出力モード。
	 * @return 文字列。
	 */
	public String toString( StringMode mode ) {
		return toString( text, mode );
	}

	/**
	 * テキスト化。
	 * @param str 文字列。
	 * @param mode 出力モード。
	 * @return 文字列。
	 */
	private String toString( String str, StringMode mode ) {
		if ( str == null )	return null;
		if ( mode == StringMode.PLAIN ) {
			return str;
		}
		StringBuffer	buf = new StringBuffer();
		char[]	line = str.toCharArray();
		int	spc_cnt = 0;
		for ( int i = 0; i < line.length; i++ ) {
			if ( line[i] == ' ' ) {
				String	add = " ";
				switch( mode ) {
				case TABLE:
				case BODY:
					add = ((spc_cnt % 2) == 0)?	"&nbsp;": " ";
					break;
				}
				spc_cnt++;
				buf = buf.append( add );
				continue;
			}
			spc_cnt = 0;
			if ( line[i] == '"' ) {
				buf = buf.append( "&quot;" );
				continue;
			}
			if ( mode != StringMode.ATTRIBUTE ) {
				WebChar	wc = WebChar.getWebChar( line[i] );
				if ( wc != null ) {
					buf = buf.append( wc.getRef() );
					continue;
				}
				if ( mode != StringMode.TEXTAREA ) {
					if ( line[i] == '\n' || line[i] == '\r' ) {
						if ( i > 0 ) {
							if ( line[i -1] == '\n' || line[i - 1] == '\r' ) {
								if ( line[i] != line[i - 1] )	continue;
							}
						}
						String	br = "<br>";
						if ( embed != null ) {
							if ( embed.isXML() ) {
								br = "<br />";
							}
						}
						buf = buf.append( br );
						continue;
					}
				}
			}
			buf = buf.append( line[i] );
		}
		return buf.toString();
	}

	/**
	 * 複製の作成。
	 * @return 複製。
	 */
	public HTMLPart getReplica() {
		return new Text( text );
	}

	/**
	 * 文字列の追加。
	 * @param add 追加文字列。
	 */
	public void append( String add ) {
		if ( add == null )	return;
		StringBuffer	buf = new StringBuffer( text );
		buf = buf.append( add );
		text = buf.toString();
	}

	/**
	 * 文字列の追加。
	 * @param add 追加文字列。
	 */
	public void append( Text add ) {
		if ( add == null )	return;
		append( add.text );
	}
}


