#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "snp_RAT.h"

dsfmt_t snp_dsfmt_global_data;

int main(int argc, char* argv[])
{
    InputRAT inputRAT={"", "", "", 0, "", 0, 0, 0, 0};

    if(argc != 10){
        printf("[usage]main.exe [InputFile1] [InputFile2] [OutputFile] [AreaFileType] [BlockAreaFile] [Score] [Generation] [Burnin] [dataType] \n");
        return 255;
    }
    strcpy(inputRAT.inputFile1, argv[1]);
    strcpy(inputRAT.inputFile2, argv[2]);
    strcpy(inputRAT.outputFile1, argv[3]);
    inputRAT.areaFileType = atoi(argv[4]);
    strcpy(inputRAT.blockAreaFile, argv[5]);
    inputRAT.score = atoi(argv[6]);
    inputRAT.gen = atol(argv[7]);
    inputRAT.burnin = atol(argv[8]);
    inputRAT.dataType = atoi(argv[9]);

    /* スコア計算方法の指定 */
    iWay = inputRAT.score;

    /* 乱数発生関数初期化 */
    initMyRand(&snp_dsfmt_global_data);

    /* RATのPermutation検定を行う */
    MainProgramRAT(&inputRAT);

}


/* RATのPermutation検定を行う */
void MainProgramRAT(InputRAT *inputRAT)
{
    int retval = 0;
    int flag = 0;
    long i = 0;
	long j = 0;
    long fileLine1 = 0; /* 入力ファイル（case）のライン数 */
    long fileLine2 = 0; /* 入力ファイル（contorl）のライン数 */
    long areaFileLine = 0; /* haplotypeブロック領域指定ファイルのライン数 */
    long a = 0;         /* number of haplotype copies (=sequences) in case */
    long b = 0;         /* number of haplotype copies (=sequences) in control */
    long n = 0;
    long dataNum;       /* case、controlデータ整合後の総SNP数 */
    long jStart = 0;    /* haplotypeブロックの最初のSNPを示す座位 */
    long jEnd = 0;      /* haplotypeブロックの最後のSNPを示す座位 */
    long blockNum = 0;  /* haplotypeブロック数 */
    long gen = 0;
    long startPos = 0;
    long endPos = 0;
    double S = 0;

    int **T = NULL;             /* 偶現表 */
    int *populationType = NULL;
    long *blockArea = NULL;     /* 各haplotypeブロック領域格納 */
    long *linkSNPNum = NULL;    /* 各haplotypeブロックのSNP数 */
    long *linkSNPStart = NULL;  /* 各haplotypeブロックの最初のSNPを示す座位 */
    long *maxScoreIndex = NULL; /* 各haplotypeブロック内のスコア最大値の座位 */
    double *Sobs = NULL;        /* 各haplotypeブロック毎のスコア最大値 */
    double *p = NULL;

    FILE *fpCase = NULL;    /* 入力（case）ファイルポインタ */
    FILE *fpCntl = NULL;    /* 入力（control）ファイルポインタ */
    FILE *fpOut = NULL;     /* 出力ファイルポインタ */
    FILE *fpArea = NULL;    /* haplotypeブロック領域指定ファイルポインタ */

    char *caseData = NULL;      /* サンプルデータ（case）格納用 */
    char *controlData = NULL;   /* サンプルデータ（control）格納用 */

    SnpData *snpTmpData1 = NULL;
    SnpData *snpTmpData2 = NULL;
    SnpData *snpData1 = NULL;
    SnpData *snpData2 = NULL;

    gen = inputRAT->gen;

    /* ファイルオープン */
    retval = InputFileOpen(&fpCase, inputRAT->inputFile1);
    if (retval != 0){
        goto finalize;
    }
    retval = InputFileOpen(&fpCntl, inputRAT->inputFile2);
    if (retval != 0){
        goto finalize;
    }
    retval = OutputFileOpen(&fpOut, inputRAT->outputFile1);
    if (retval != 0){
        goto finalize;
    }
    retval = InputFileOpen(&fpArea, inputRAT->blockAreaFile);
    if (retval != 0){
        goto finalize;
    }

/****************************************************************/
/* データ入力                                                   */
/****************************************************************/
 
    /* haplotypeブロック領域指定ファイルのライン数を取得 */
    areaFileLine = DataReaderCountFileLine(fpArea);
    /* haplotypeブロック領域格納用配列のメモリ確保 */
    blockArea = (long*)malloc1Dim(sizeof(long), areaFileLine);
    if (NULL == blockArea){ goto finalize; }
    /* ファイルポインタを先頭に戻す */
    fseek(fpArea, 0L, SEEK_SET);
    /* haplotypeブロック領域を配列に収める */
    DataReaderSetHaplotypeBlockArea(fpArea, blockArea);

    /* 入力ファイルのライン数を取得 */
    fileLine1 = DataReaderCountFileLine(fpCase);
    fileLine2 = DataReaderCountFileLine(fpCntl);

    /* ファイルポインタを先頭に戻す */
    fseek(fpCase, 0L, SEEK_SET);
    fseek(fpCntl, 0L, SEEK_SET);

    /* データ一時格納用構造体のメモリ確保 */
    snpTmpData1 = (SnpData*)malloc1Dim(sizeof(SnpData), fileLine1);
    if (NULL == snpTmpData1){ goto finalize; }
    snpTmpData2 = (SnpData*)malloc1Dim(sizeof(SnpData), fileLine2);
    if (NULL == snpTmpData2){ goto finalize; }

    /* データをファイルから読み込み構造体に収める */
    DataReaderSetAllData(fpCase, snpTmpData1, fileLine1, inputRAT->dataType);
    DataReaderSetAllData(fpCntl, snpTmpData2, fileLine2, inputRAT->dataType);

    /* 入力データの整合性をチェックして並列化用の入力データを作成する */
    /* MPI_Bcastの回数を減らすためにサンプルデータは別配列（caseData, controlData）で保持する */
    dataNum = DataReaderMakeParallelData(snpTmpData1, snpTmpData2, fileLine1, fileLine2, &snpData1, &snpData2, &caseData, &controlData);

    /* 使用しない配列のメモリ開放 */
    /* 構造体SnpDataメンバのメモリを開放する */
    DataReaderSnpDataMemoryFree(snpTmpData1, fileLine1);
    DataReaderSnpDataMemoryFree(snpTmpData2, fileLine2);
    snpTmpData1 = NULL;
    snpTmpData2 = NULL;

    /* 入力データのサンプル数取得 */
    a = snpData1[0].dataNum;
    b = snpData2[0].dataNum;
    n = a + b;

    /* 構造体SnpDataにサンプルデータをコピーする */
    DataReaderDataCopyToSnpData(snpData1, caseData, dataNum, a);
    DataReaderDataCopyToSnpData(snpData2, controlData, dataNum, b);

/****************************************************************/
/* メモリ確保                                                   */
/****************************************************************/

    /* haplotypeブロック数 */
    if (inputRAT->areaFileType == 0){
        blockNum = areaFileLine - 1;
    }
    else {
        /* haplotypeブロックの個数を計算する */
        if (blockArea[0] > dataNum){
            blockNum = 1;
            blockArea[1] = dataNum; /* linkSNPNumの値がデータ数だけになる */
        }
        else{
            blockNum = (dataNum - (blockArea[0] - blockArea[1]) ) / blockArea[1];
        }
    }
    /* 各haplotypeブロック領域内のSNP数格納用配列のメモリ確保 */
    linkSNPNum = (long*)malloc1Dim(sizeof(long), blockNum);
    if (NULL == linkSNPNum){ goto finalize; }
    /* 各haplotypeブロック領域内の最初のSNPを示す座位格納用配列のメモリ確保 */
    linkSNPStart = (long*)malloc1Dim(sizeof(long), blockNum);
    if (NULL == linkSNPStart){ goto finalize; }

    /* 各haplotypeブロック領域に該当するSNPデータを決定 */
    if (inputRAT->areaFileType == 0){
        for (i = 0; i < blockNum; i++){ /* 将来、領域の重複を許す場合も考える */
            startPos = blockArea[i];
            endPos = blockArea[i+1];
            flag = 0;
            /* 入力データはポジションでソートされていると仮定 */
            for (j = 0; j < dataNum; j++){
                if (startPos <= snpData1[j].pos){
                    if (snpData1[j].pos < endPos){
                        /* 領域内のSNP数カウント */
                        linkSNPNum[i]++;
                        /* 領域内の最初のSNPを示す座位を保持 */
                        if (0 == flag ){
                            linkSNPStart[i] = j;
                            flag = 1;
                        }
                    }
                    /* これ以降、領域に該当するデータは出現しないので次のブロックを調べる */
                    else {
                        break;
                    }
                }
            }
        }
    }
     else {
        for (i = 0; i < blockNum; i++){
            linkSNPNum[i] = blockArea[0];
            linkSNPStart[i] = i * blockArea[1];
        }
    }

    /* 偶現表Tのメモリ確保 */
    T = (int**)mallocInt2Dim(ROW, COLUMN);
    if (NULL == T){ goto finalize; }
    /* populationTypeのメモリ確保 */
    populationType = (int*)malloc1Dim(sizeof(int), a + b);
    if (NULL == populationType) { goto finalize; }
    /* Haplotypeブロック内のスコア最大値の座位のメモリ確保 */
    maxScoreIndex = (long*)malloc1Dim(sizeof(long), blockNum);
    if (NULL == maxScoreIndex) { goto finalize; }
    /* Haplotypeブロック毎のスコア最大値のメモリ確保 */
    Sobs = (double*)malloc1Dim(sizeof(double), blockNum);
    if (NULL == Sobs) { goto finalize; }
    /* pのメモリ確保 */
    p = (double*)malloc1Dim(sizeof(double), blockNum);
    if (NULL == p) { goto finalize; }

/****************************************************************/
/* 検定処理                                                     */
/****************************************************************/

    /* haplotypeブロック単位でループ */
    jStart = 0;
    for (i = 0; i < blockNum; i++){
        jStart = linkSNPStart[i];
        jEnd = jStart + linkSNPNum[i];
        /* 各haplotypeブロック内でループ */
        for (j = jStart; j < jEnd; j++){
            /* 観測値から偶現表を作成する */
            DataReaderPopulationType(&snpData1[j], &snpData2[j], populationType);
            DataReaderMakeTableDi(&snpData1[j], &snpData2[j], populationType, T);
            /* スコアを計算　スコア計算中0割になってしまう場合は-1を返す */
            S = TableCalcScore(T);
            /* 最大スコアを決定する */
            if (S > Sobs[i]){
                Sobs[i] = S;
                maxScoreIndex[i] = j;
            }
        }

        /* RATのPermutation検定を実行する */
        if (Sobs[i] > 0.0) {
            p[i] = RATExecute(snpData1, snpData2, Sobs[i], a, b, jStart, jEnd, gen, inputRAT->burnin);
        }
    }

/****************************************************************/
/* 検定結果出力                                                 */
/****************************************************************/

    fprintf(fpOut, "CaseData    = %s\n", inputRAT->inputFile1);
    fprintf(fpOut, "ControlData = %s\n", inputRAT->inputFile2);
    fprintf(fpOut, "Genoration  = %ld\n", inputRAT->gen);
    fprintf(fpOut, "BlockArea\tSNPNum\trsNumber\tPosition\tScore\tP\n");
    if (inputRAT->areaFileType == 0){
        for (i = 0; i < blockNum; i++){
            if (Sobs[i] != 0){
                fprintf(fpOut, "%ld-%ld\t%ld\t%s\t%ld\t%.10lf\t%.10lf\n", 
                    blockArea[i],
                    blockArea[i+1],
                    linkSNPNum[i],
                    snpData1[ maxScoreIndex[i] ].rsNumber,
                    snpData1[ maxScoreIndex[i] ].pos,
                    Sobs[i],
                    p[i]);
            }
            else {
                fprintf(fpOut, "%ld-%ld\t%ld\tNoData\n", 
                    blockArea[i],
                    blockArea[i+1],
                    linkSNPNum[i]);
            }
        }
    }
    else {
        for (i = 0; i < blockNum; i++){
            if (Sobs[i] != 0){
                fprintf(fpOut, "%ld-%ld\t%ld\t%s\t%ld\t%.10lf\t%.10lf\n", 
                    snpData1[ linkSNPStart[i] ].pos,
                    snpData1[ linkSNPStart[i] + linkSNPNum[i] - 1 ].pos,
                    linkSNPNum[i],
                    snpData1[ maxScoreIndex[i] ].rsNumber,
                    snpData1[ maxScoreIndex[i] ].pos,
                    Sobs[i],
                    p[i]);
            }
            else {
                fprintf(fpOut, "%ld-%ld\t%ld\tNoData\n", 
                    snpData1[ linkSNPStart[i] ].pos,
                    snpData1[ linkSNPStart[i] + linkSNPNum[i] - 1 ].pos,
                    linkSNPNum[i]);
            }
        }
    }

/****************************************************************/
/* 終了処理                                                     */
/****************************************************************/

finalize:;
    /* ファイルクローズ */
    FileClose(fpCase);
    FileClose(fpCntl);
    FileClose(fpOut);
    FileClose(fpArea);
    /* 確保したメモリを開放する */
    free1Dim(blockArea);
    free1Dim(linkSNPNum);
    free1Dim(linkSNPStart);
    freeInt2Dim(T, ROW);
    free1Dim(populationType);
    free1Dim(maxScoreIndex);
    free1Dim(Sobs);
    free1Dim(p);
    DataReaderSnpDataMemoryFree(snpTmpData1, fileLine1);
    DataReaderSnpDataMemoryFree(snpTmpData2, fileLine2);
    free1Dim(caseData);
    free1Dim(controlData);
    free1Dim(snpData1);
    free1Dim(snpData2);

    return;
}

/* RATのPermutation検定を実行する */
double RATExecute(SnpData *snpData1, SnpData *snpData2, double Sobs, long a, long b, long jStart, long jEnd, long gen, long burnin)
{
    long h = 0;
    long i = 0;
    long j = 0;
    long m = 0;
    long n = 0;
    long gen2 = 0;
    double sumOfQj = 0;
    double sumOfinvQj = 0;
    double S = 0;
    double Srat = 0;
    double F = 0;
    double logF = 0;
    double Qvalue = 0;
    double result = 0;
    SnpTable T;

    int *type = NULL;
    int *di = NULL;
    int **genotype = NULL;
    int ***Tj = NULL;
    double *Sj = NULL;
    double *numberOfDiInHj = NULL;

    n = a + b;
    m = jEnd - jStart;

    /* significantではない場合、終了 */
    if ( Sobs < 3.84){
        printf("P > 0.05\n");
        return 0.05;
    }

    /* log テーブルを作成する */
    FactorialSetFactorial(n);

    /* typeのメモリ確保 */
    type = (int*)malloc1Dim(sizeof(int), n);
    if (NULL == type) { goto finalize; }
    /* diのメモリ確保 */
    di = (int*)malloc1Dim(sizeof(int), n);
    if (NULL == di) { goto finalize; }
    /* genotypeのメモリ確保 */
    genotype = (int**)mallocInt2Dim(m, n);
    if (NULL == genotype) { goto finalize; }
    /* Tjのメモリ確保 */
    Tj = (int***)mallocInt3Dim(m, ROW, COLUMN);
    if (NULL == Tj) { goto finalize; }
    /* Sjのメモリ確保 */
    Sj = (double*)malloc1Dim(sizeof(double), m);
    if (NULL == Sj) { goto finalize; }
    /* numberOfDiInHjのメモリ確保 */
    numberOfDiInHj = (double*)malloc1Dim(sizeof(double), m);
    if (NULL == numberOfDiInHj) { goto finalize; }
         
    for (j = jStart; j < jEnd; j++){
        /* 観測値から偶現表を作成する */
        DataReaderPopulationType(&snpData1[j], &snpData2[j], type);
        DataReaderMakeTableDi(&snpData1[j], &snpData2[j], type, Tj[i]);
        /* genotypeを計算 */
        RATGenotype(&snpData1[j], &snpData2[j], genotype[i]);
        i++;
    }

    for (i = 0; i < m; i++){
        TableMakeTableIntArray(&T, Tj[i]);
        numberOfDiInHj[i] = RATNumberOfDiInHj(T, Sobs);
        sumOfQj += numberOfDiInHj[i];
        /* 構造体tableのメモリ開放 */
        TableFinalTable(&T);
    }

    for (h= 0; h < burnin; h++){
        for (i = 0; i < m; i++){
            TableMakeTableIntArray(&T, Tj[i]);
            RATAlgorithmB(T, Sobs, Tj[i], &snp_dsfmt_global_data);
            /* 構造体tableのメモリ開放 */
            TableFinalTable(&T);
        }
    }

    for (h = 0; h < gen; h++){
        j = RATAlgorithmA(numberOfDiInHj, m, &snp_dsfmt_global_data);
        TableMakeTableIntArray(&T, Tj[j]);
        RATAlgorithmB(T, S, Tj[j], &snp_dsfmt_global_data);
        RATAlgorithmC(genotype[j], Tj[j], di, &snp_dsfmt_global_data);
        Qvalue = RATCalcQvalue(genotype, di, Sobs, n, m);
        if (Qvalue != 0){
            sumOfinvQj += 1 / Qvalue;
            gen2++;
        }
        /* 構造体tableのメモリ開放 */
        TableFinalTable(&T);
    }

    //F = FactorialGetCombination(n, a);
    logF = FactorialGetLogCombination(n, a);
    if (0 == gen2){
        result = -1;
    }
    else{
        result = exp( log(sumOfQj) + log(sumOfinvQj) - log(gen2) - logF );
        //result = sumOfQj * sumOfinvQj / gen2 / F;
    }


finalize:;
    /* 確保したメモリを開放する */
    free1Dim(type);
	freeInt2Dim(genotype, m);
    freeInt3Dim(Tj, m, ROW);
    free1Dim(Sj);
    free1Dim(numberOfDiInHj);
    free1Dim(di);
    /* logテーブルクリア */
    FactorialDeleteFactorial();
    /* 構造体tableのメモリ開放 */
    TableFinalTable(&T);

    return result;
}

/* マーカーjをnumberOfDiInHj[j]に比例した確率で返す */
long RATAlgorithmA(double *numberOfDiInHj, long m, dsfmt_t *snp_dsfmt_data)
{
    long j = 0;
    double p = 0;
    double sumOfNumberOfHj = 0;


    for (j = 0; j < m; j++){
        sumOfNumberOfHj += numberOfDiInHj[j];
    }
    p = sumOfNumberOfHj * myRand(snp_dsfmt_data);  

    for (j = 0; j < m; j++){
        if (p < numberOfDiInHj[j]){
            return j;
        }
        p -= numberOfDiInHj[j];
    }

    return 0;
}

/* 超幾何分布に従うMCMCで、以前のテープルから次のテープルを作成 */
int RATAlgorithmB(SnpTable T, double Score, int **table, dsfmt_t *snp_dsfmt_data)
{
    if (FLAG_TRUE == TableTwoParts(T, Score)){
        TableMarkov2(T, Score, table, snp_dsfmt_data);
    }
    else{
        TableMarkov1(T, Score, table, snp_dsfmt_data);
    }

    return 0;
}

/* 偶現表から１つのpermutation eventを等確率で抽出 */
int RATAlgorithmC(int *genotype, int **table, int *di, dsfmt_t *snp_dsfmt_data)
{
    PermutationConditionedDi(genotype, table, di, snp_dsfmt_data);
    return 0;
}

/* Qの値を計算 */
long RATCalcQvalue(int **genotype, int *di, double Sobs, long n, long dataNum)
{
    int retval = 0;
    long j = 0;
    long result = 0;
    double S = 0;
    int **T = NULL;

    /* Tのメモリ確保 */
    T = (int**)mallocInt2Dim(ROW, COLUMN);
    if (NULL == T){ goto finalize; }

    for (j = 0; j < dataNum; j++){
        retval = PermutationCalcT(di, genotype[j], T, n);
        S = TableCalcScore(T);
        if (S >= Sobs){
            result++;
        }      	
    }

finalize:;
    /* 確保したメモリを開放する */
    freeInt2Dim(T, ROW);

    return result;
}

double RATNumberOfDiInHj(SnpTable table, double Sobs)
{
    double result = 0;
    int l = 0;
    int u = 0;
    int retval = 0;
    int **T = NULL;

    int i = 0;
    int a1 = 0;

    /* Tのメモリ確保 */
    T = (int**)mallocInt2Dim(ROW, COLUMN);
    if (NULL == T) {
        retval = 1;
        goto finalize;
    }

    l = TableCalcL(table);
    u = TableCalcU(table);

    for (a1 = l; a1 <= u; a1++){
        retval = TableNewTable(table, a1, T);
        if (FLAG_TRUE == TableInCj(Sobs, T, ROW, COLUMN)){
            result += TableCalcMu(T);
        }
    }

finalize:;
    /* 確保したメモリを開放する */
    freeInt2Dim(T, ROW);

    return result;
}

/* genotypeを計算 */
int RATGenotype(SnpData *snpData1, SnpData *snpData2, int *genotype)
{
    long h = 0;
    long a = 0;
    long b = 0;
    char reference = 0;

    int *genotype1 = NULL;
    int *genotype2 = NULL;

    a = snpData1->dataNum;
    b = snpData2->dataNum;
   
    /* genotype1のメモリ確保 */
    genotype1 = (int*)malloc1Dim(sizeof(int), a);
    if (NULL == genotype1) { goto finalize; }
    /* genotype2のメモリ確保 */
    genotype2 = (int*)malloc1Dim(sizeof(int), b);
    if (NULL == genotype2) { goto finalize; }

    /* 参照文字を決定 */
    reference = DataReaderMaximumNuc(snpData1);

    /* genotypeを計算 */
    DataReaderCalcGenotype(snpData1, reference, genotype1);
    DataReaderCalcGenotype(snpData2, reference, genotype2);
    for (h = 0; h < a; h++){
        genotype[h] = genotype1[h];
    }
    for (h = 0; h < b; h++){
        genotype[h + a] = genotype2[h];
    }

finalize:;
    /* 確保したメモリを開放する */
    free1Dim(genotype1);
    free1Dim(genotype2);

    return 0;
}
