#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "snp_Primitive.h"

dsfmt_t snp_dsfmt_global_data;

int main(int argc, char* argv[])
{
    InputPrimitive inputPrimitive={"", "", "", 0, "", 0, 0, 0};

    if(argc != 9){
        printf("[usage]main.exe [InputFile1] [InputFile2] [outputFile] [AreaFileType] [BlockAreaFile] [Score] [Repeat] [dataType] \n");
        return 255;
    }
    strcpy(inputPrimitive.inputFile1, argv[1]);
    strcpy(inputPrimitive.inputFile2, argv[2]);
    strcpy(inputPrimitive.outputFile1, argv[3]);
    inputPrimitive.areaFileType = atoi(argv[4]);
    strcpy(inputPrimitive.blockAreaFile, argv[5]);
    inputPrimitive.score = atoi(argv[6]);
    inputPrimitive.repeat = atol(argv[7]);
    inputPrimitive.dataType = atoi(argv[8]);

    /* スコア計算方法の指定 */
    iWay = inputPrimitive.score;

    /* 乱数発生関数初期化 */
    initMyRand(&snp_dsfmt_global_data);

    /* PrimitiveなPermutation検定を行う */
    MainProgramPrimitive(&inputPrimitive);

}


/* PrimitiveなPermutation検定を行う */
void MainProgramPrimitive(InputPrimitive *inputPrimitive)
{
    int retval = 0;
    int flag = 0;
    long i = 0;
	long j = 0;
    long fileLine1 = 0; /* 入力ファイルのライン数 */
    long fileLine2 = 0; /* 入力ファイルのライン数 */
    long areaFileLine = 0; /* haplotypeブロック領域指定ファイルのライン数 */
    long a = 0;         /* number of haplotype copies (=sequences) in case */
    long b = 0;         /* number of haplotype copies (=sequences) in control */
    long n = 0;
    long dataNum;       /* case、controlデータ整合後の総SNP数 */
    long jStart = 0;    /* haplotypeブロックの最初のSNPを示す座位 */
    long jEnd = 0;      /* haplotypeブロックの最後のSNPを示す座位 */
    long blockNum = 0;  /* haplotypeブロック数 */
    long repeat = 0;
    long startPos = 0;
    long endPos = 0;
    double S = 0;

    int **T = NULL;             /* 偶現表 */
    int *populationType = NULL;
    long *blockArea = NULL;     /* 各haplotypeブロック領域格納 */
    long *linkSNPNum = NULL;    /* 各haplotypeブロックのSNP数 */
    long *linkSNPStart = NULL;  /* 各haplotypeブロックの最初のSNPを示す座位 */
    long *maxScoreIndex = NULL; /* 各haplotypeブロック内のスコア最大値の座位 */
    long *count = NULL;         /* 各haplotypeブロック内のスコア最大値を超えるスコアの出現回数 */
    double *Sobs = NULL;        /* 各haplotypeブロック毎のスコア最大値 */

    FILE *fpCase = NULL;    /* 入力（case）ファイルポインタ */
    FILE *fpCntl = NULL;    /* 入力（control）ファイルポインタ */
    FILE *fpOut = NULL;     /* 出力ファイルポインタ */
    FILE *fpArea = NULL;    /* haplotypeブロック領域指定ファイルポインタ */

    char *caseData = NULL;      /* サンプルデータ（case）格納用 */
    char *controlData = NULL;   /* サンプルデータ（control）格納用 */

    SnpData *snpTmpData1 = NULL;
    SnpData *snpTmpData2 = NULL;
    SnpData *snpData1 = NULL;
    SnpData *snpData2 = NULL;

    repeat = inputPrimitive->repeat;

    /* ファイルオープン */
    retval = InputFileOpen(&fpCase, inputPrimitive->inputFile1);
    if (retval != 0){
        goto finalize;
    }
    retval = InputFileOpen(&fpCntl, inputPrimitive->inputFile2);
    if (retval != 0){
        goto finalize;
    }
    retval = OutputFileOpen(&fpOut, inputPrimitive->outputFile1);
    if (retval != 0){
        goto finalize;
    }
    retval = InputFileOpen(&fpArea, inputPrimitive->blockAreaFile);
    if (retval != 0){
        goto finalize;
    }

/****************************************************************/
/* データ入力                                                   */
/****************************************************************/
 
    /* haplotypeブロック領域指定ファイルのライン数を取得 */
    areaFileLine = DataReaderCountFileLine(fpArea);
    /* haplotypeブロック領域格納用配列のメモリ確保 */
    blockArea = (long*)malloc1Dim(sizeof(long), areaFileLine);
    if (NULL == blockArea){ goto finalize; }
    /* ファイルポインタを先頭に戻す */
    fseek(fpArea, 0L, SEEK_SET);
    /* haplotypeブロック領域を配列に収める */
    DataReaderSetHaplotypeBlockArea(fpArea, blockArea);
 
    /* 入力ファイルのライン数を取得 */
    fileLine1 = DataReaderCountFileLine(fpCase);
    fileLine2 = DataReaderCountFileLine(fpCntl);

    /* ファイルポインタを先頭に戻す */
    fseek(fpCase, 0L, SEEK_SET);
    fseek(fpCntl, 0L, SEEK_SET);

    /* データ一時格納用構造体のメモリ確保 */
    snpTmpData1 = (SnpData*)malloc1Dim(sizeof(SnpData), fileLine1);
    snpTmpData2 = (SnpData*)malloc1Dim(sizeof(SnpData), fileLine2);

    /* データをファイルから読み込み構造体に収める */
    DataReaderSetAllData(fpCase, snpTmpData1, fileLine1, inputPrimitive->dataType);
    DataReaderSetAllData(fpCntl, snpTmpData2, fileLine2, inputPrimitive->dataType);

    /* 入力データの整合性をチェックして並列化用の入力データを作成する */
    /* MPI_Bcastの回数を減らすためにサンプルデータは別配列（caseData, controlData）で保持する */
    dataNum = DataReaderMakeParallelData(snpTmpData1, snpTmpData2, fileLine1, fileLine2, &snpData1, &snpData2, &caseData, &controlData);

    /* 使用しない配列のメモリ開放 */
    /* 構造体SnpDataメンバのメモリを開放する */
    DataReaderSnpDataMemoryFree(snpTmpData1, fileLine1);
    DataReaderSnpDataMemoryFree(snpTmpData2, fileLine2);
    snpTmpData1 = NULL;
    snpTmpData2 = NULL;

    /* 入力データのサンプル数取得 */
    a = snpData1[0].dataNum;
    b = snpData2[0].dataNum;
    n = a + b;

    /* 構造体SnpDataにサンプルデータをコピーする */
    DataReaderDataCopyToSnpData(snpData1, caseData, dataNum, a);
    DataReaderDataCopyToSnpData(snpData2, controlData, dataNum, b);

/****************************************************************/
/* メモリ確保                                                   */
/****************************************************************/

    /* haplotypeブロック数 */
    if (inputPrimitive->areaFileType == 0){
        blockNum = areaFileLine - 1;
    }
    else {
        /* haplotypeブロックの個数を計算する */
        if (blockArea[0] > dataNum){
            blockNum = 1;
            blockArea[1] = dataNum; /* linkSNPNumの値がデータ数だけになる */
        }
        else{
            blockNum = (dataNum - (blockArea[0] - blockArea[1]) ) / blockArea[1];
        }
    }
    /* 各haplotypeブロック領域内のSNP数格納用配列のメモリ確保 */
    linkSNPNum = (long*)malloc1Dim(sizeof(long), blockNum);
    if (NULL == linkSNPNum){ goto finalize; }
    /* 各haplotypeブロック領域内の最初のSNPを示す座位格納用配列のメモリ確保 */
    linkSNPStart = (long*)malloc1Dim(sizeof(long), blockNum);
    if (NULL == linkSNPStart){ goto finalize; }

    /* 各haplotypeブロック領域に該当するSNPデータを決定 */
    if (inputPrimitive->areaFileType == 0){
        for (i = 0; i < blockNum; i++){ /* 将来、領域の重複を許す場合も考える */
            startPos = blockArea[i];
            endPos = blockArea[i+1];
            flag = 0;
            /* 入力データはポジションでソートされていると仮定 */
            for (j = 0; j < dataNum; j++){
                if (startPos <= snpData1[j].pos){
                    if (snpData1[j].pos < endPos){
                        /* 領域内のSNP数カウント */
                        linkSNPNum[i]++;
                        /* 領域内の最初のSNPを示す座位を保持 */
                        if (0 == flag ){
                            linkSNPStart[i] = j;
                            flag = 1;
                        }
                    }
                    /* これ以降、領域に該当するデータは出現しないので次のブロックを調べる */
                    else {
                        break;
                    }
                }
            }
        }
    }
    else {
        for (i = 0; i < blockNum; i++){
            linkSNPNum[i] = blockArea[0];
            linkSNPStart[i] = i * blockArea[1];
        }
    }

    /* 偶現表Tのメモリ確保 */
    T = (int**)mallocInt2Dim(ROW, COLUMN);
    if (NULL == T){ goto finalize; }
    /* populationTypeのメモリ確保 */
    populationType = (int*)malloc1Dim(sizeof(int), a + b);
    if (NULL == populationType) { goto finalize; }
    /* Haplotypeブロック内のスコア最大値の座位のメモリ確保 */
    maxScoreIndex = (long*)malloc1Dim(sizeof(long), blockNum);
    if (NULL == maxScoreIndex) { goto finalize; }
    /* Haplotypeブロック毎のスコア最大値のメモリ確保 */
    Sobs = (double*)malloc1Dim(sizeof(double), blockNum);
    if (NULL == Sobs) { goto finalize; }
    /* Haplotypeブロック内のスコア最大値を超えるスコアの出現回数のメモリ確保 */
    count = (long*)malloc1Dim(sizeof(long), blockNum);
    if (NULL == count) { goto finalize; }

/****************************************************************/
/* 検定処理                                                     */
/****************************************************************/

    /* haplotypeブロック単位でループ */
    jStart = 0;
    for (i = 0; i < blockNum; i++){
        jStart = linkSNPStart[i];
        jEnd = jStart + linkSNPNum[i];
        /* 各haplotypeブロック内でループ */
        for (j = jStart; j < jEnd; j++){
            /* 観測値から偶現表を作成する */
            DataReaderPopulationType(&snpData1[j], &snpData2[j], populationType);
            DataReaderMakeTableDi(&snpData1[j], &snpData2[j], populationType, T);
            /* スコアを計算　スコア計算中0割になってしまう場合は-1を返す */
            S = TableCalcScore(T);
            /* 最大スコアを決定する */
            if (S > Sobs[i]){
                Sobs[i] = S;
                maxScoreIndex[i] = j;
            }
        }

        /* PrimitiveなPermutation検定を実行する */
        if (Sobs[i] > 0.0) {
            count[i] = PrimitiveExecute(snpData1, snpData2, Sobs[i], a, b, jStart, jEnd, repeat);
        }
    }

/****************************************************************/
/* 検定結果出力                                                 */
/****************************************************************/

    fprintf(fpOut, "CaseData    = %s\n", inputPrimitive->inputFile1);
    fprintf(fpOut, "ControlData = %s\n", inputPrimitive->inputFile2);
    fprintf(fpOut, "Repeat      = %ld\n", inputPrimitive->repeat);
    fprintf(fpOut, "BlockArea\tSNPNum\trsNumber\tPosition\tScore\tP\n");
    if (inputPrimitive->areaFileType == 0){
        for (i = 0; i < blockNum; i++){
            if (Sobs[i] != 0){
                fprintf(fpOut, "%ld-%ld\t%ld\t%s\t%ld\t%.10lf\t%.10lf\n", 
                    blockArea[i],
                    blockArea[i+1],
                    linkSNPNum[i],
                    snpData1[ maxScoreIndex[i] ].rsNumber,
                    snpData1[ maxScoreIndex[i] ].pos,
                    Sobs[i],
                    (double)count[i] / repeat);
            }
            else {
                fprintf(fpOut, "%ld-%ld\t%ld\tNoData\n", 
                    blockArea[i],
                    blockArea[i+1],
                    linkSNPNum[i]);
            }
        }
    }
    else {
        for (i = 0; i < blockNum; i++){
            if (Sobs[i] != 0){
                fprintf(fpOut, "%ld-%ld\t%ld\t%s\t%ld\t%.10lf\t%.10lf\n", 
                    snpData1[ linkSNPStart[i] ].pos,
                    snpData1[ linkSNPStart[i] + linkSNPNum[i] - 1 ].pos,
                    linkSNPNum[i],
                    snpData1[ maxScoreIndex[i] ].rsNumber,
                    snpData1[ maxScoreIndex[i] ].pos,
                    Sobs[i],
                    (double)count[i] / repeat);
            }
            else {
                fprintf(fpOut, "%ld-%ld\t%ld\tNoData\n", 
                    snpData1[ linkSNPStart[i] ].pos,
                    snpData1[ linkSNPStart[i] + linkSNPNum[i] - 1 ].pos,
                    linkSNPNum[i]);
            }
        }
    }

/****************************************************************/
/* 終了処理                                                     */
/****************************************************************/

finalize:;
    /* ファイルクローズ */
    FileClose(fpCase);
    FileClose(fpCntl);
    FileClose(fpOut);
    FileClose(fpArea);
    /* 確保したメモリを開放する */
    free1Dim(blockArea);
    free1Dim(linkSNPNum);
    free1Dim(linkSNPStart);
    freeInt2Dim(T, ROW);
    free1Dim(populationType);
    free1Dim(maxScoreIndex);
    free1Dim(Sobs);
    free1Dim(count);
    DataReaderSnpDataMemoryFree(snpTmpData1, fileLine1);
    DataReaderSnpDataMemoryFree(snpTmpData2, fileLine2);
    free1Dim(caseData);
    free1Dim(controlData);
    free1Dim(snpData1);
    free1Dim(snpData2);

    return;
}

/* PrimitiveなPermutation検定を実行する */
long PrimitiveExecute(SnpData *snpData1, SnpData *snpData2, double Sobs, long a, long b, long jStart, long jEnd, long repeat)
{
    int retval = 0;
    long i = 0;
    long j = 0;
    long n = 0;
    long count = 0;

    double S = 0;
    double Sperm = 0;

    int **Tperm = NULL;             /* 偶現表 */
    int *omega = NULL;
    int *di = NULL;

    n = a + b;

    /* 偶現表Tpermのメモリ確保 */
    Tperm = (int**)mallocInt2Dim(ROW, COLUMN);
    if (NULL == Tperm){ goto finalize; }
    /* omegaのメモリ確保 */
    omega = (int*)malloc1Dim(sizeof(int), n);
    if (NULL == omega) { goto finalize; }
    /* diのメモリ確保 */
    di = (int*)malloc1Dim(sizeof(int), n);
    if (NULL == di) { goto finalize; }

    /* permutation */
    for (i = 0; i < repeat; i++){
        Sperm = 0;
        /* 順番をシャッフル */
        retval = PermutationCalcOmega(n, omega, &snp_dsfmt_global_data);
        if (retval != 0){
            printf("can not create omega\n");
            return -1;
        }
        /* diを計算 */
        retval =  PermutationCalcDi(n, a, omega, di);
        if (retval != 0){
            printf("can not create di\n");
            return -1;
        }
        /* haplotypeブロックの各SNPを計算 */
        for (j = jStart; j < jEnd; j++){
            /* 観測値から偶現表を作成する */
            DataReaderMakeTableDi(&snpData1[j], &snpData2[j], di, Tperm);
            /* スコアを計算　スコア計算中0割になってしまう場合を-1を返す */
            S = TableCalcScore(Tperm);
            /* 最大スコアを決定する */
            if (S > Sperm){
                Sperm = S;
            }
        }
        if (Sperm >= Sobs){
            count++;
        }
    }

finalize:;
    /* 確保したメモリを開放する */
    free1Dim(omega);
    free1Dim(di);
    freeInt2Dim(Tperm, ROW);

    return count;
}
