#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <math.h>

#include "DataReader.h"
#include "Config.h"
#include "MemoryControl.h"

/**********************/
/* 入力ファイルを開く */
/**********************/
int InputFileOpen(FILE **fp, char *file)
{
    int retval = 0;

    /* ファイルオープン */
    *fp = fopen(file, "r");
    if (NULL == *fp) {
        printf("input file open error! FileName:%s\n", file);
        retval = 1;
    }

    return retval;
}

/**********************/
/* 出力ファイルを開く */
/**********************/
int OutputFileOpen(FILE **fp, char *file)
{
    int retval = 0;

    /* ファイルオープン */
    *fp = fopen(file, "w");
    if (NULL == *fp) {
        printf("output file open error! FileName:%s\n", file);
        retval = 1;
    }

    return retval;
}

/********************/
/* ファイルを閉じる */
/********************/
void FileClose(FILE *fp)
{
    if (fp != NULL) {
        fclose(fp);
    }
}


/********************************/
/* ファイルの行数をカウントする */
/********************************/
long DataReaderCountFileLine(FILE *fp)
{
    long index = 0;
    long count = 0;
    char buf[MAX_LEN];

    /* 特定文字数ずつ読み込む */
    while ( fgets(buf, MAX_LEN, fp) != NULL ) {
        index = strlen(buf);
        if ('\n' == buf[index-1]) {
            count++;
        }
    }

    return count;
}

/********************************/
/* 入力データを構造体に収める */
/********************************/
int DataReaderSetAllData(FILE *fp, SnpData *snpData, long line, int dataType)
{
    int retval = 0;

    switch(dataType) {
        case 0:
            /* 入力データがHapmap */
            retval = DataReaderSetAllHapmapData(fp, snpData, line);
            break;
        case 1:
            /* 入力データがPhasingHapmap */
            retval = DataReaderSetAllPhasingHapmapData(fp, snpData, line);
            break;
        default:
            break;
    }
    return retval;
}

/********************************/
/* Hapmapデータを構造体に収める */
/********************************/
int DataReaderSetAllHapmapData(FILE *fp, SnpData *snpData, long line)
{
    int retval = 0;
    long i = 0;
    long sampleDataNum = 0;

    /* HapmapDataのサンプルデータ数を取得する */
    sampleDataNum = DataReaderGetHapmapSampleDataNum(fp);
    /* ファイルから読み込み */
    for (i = 1; i < line; i++) {
        /* SNPデータ格納用配列のメモリ確保 */
        snpData[i].SNPdata = (char*)malloc1Dim(sizeof(char), sampleDataNum);
        retval = DataReaderSetHapmapData(fp, &snpData[i], i);
    }

    return retval;
}

/************************************/
/* Hapmapデータ一行を構造体に収める */
/************************************/
int DataReaderSetHapmapData(FILE *fp, SnpData *snpData, long line)
{
    int i = 0;
    int j = 0;
    int retval = -1;    /* ファイル最後尾まで来ている場合は-1を返す */
    int column = 1;     /* Hapmapデータの列 */
    int index = 0;
    int maxIndex = 0;
    int dataIndex = 0;
    char buf[1024];
    char tmp[20];


    /* 複数文字ずつ読み込む */
    while ( fgets(buf, 1024, fp) != NULL ) {
        maxIndex = strlen(buf);
        j = 0;
        /* １行読み込んだら値を保持して終了 */
        for (j = 0; j < maxIndex; j++) {
            if ('\n' == buf[j]) {
                snpData->sampleNum = dataIndex / 2;
                snpData->sampleDataNum = dataIndex;
                retval = 0;
                dataIndex = 0;
                return retval;
            }
            /* ファイルの２行目以降から処理を行う */
            else if (line > 0) {
                /* ファイル内の列をカウント */
                if (isspace(buf[j]) != 0) {
                    if (HAPMAP_RS_NUM == column) {
                        snpData->rsNumber[index] = '\0';
                    }
                    else if (HAPMAP_ALLELES == column){
                        snpData->alleles[index] = '\0';
                    }
                    else if (HAPMAP_CHROM == column){
                        snpData->chrom[index] = '\0';
                    }
                    else if (HAPMAP_POS == column) {
                        tmp[index] = '\0';
                        snpData->pos = atol(tmp);
                    }
                    column++;
                    index = 0;
                }
                /* rs#を取得する */
                else if (HAPMAP_RS_NUM == column) {
                    if (isalnum(buf[j]) != 0) {
                        snpData->rsNumber[index] = buf[j];
                        index++;
                    }
                }
                /* allelsを取得する */
                else if (HAPMAP_ALLELES == column){
                    snpData->alleles[index] = buf[j];
                    index++;
                }
                /* chromを取得する */
                else if (HAPMAP_CHROM == column){
                    if (isalnum(buf[j]) != 0){
                        snpData->chrom[index] = buf[j];
                        index++;
                    }
                }
                /* posを取得する */
                else if (HAPMAP_POS == column) {
                    tmp[index] = buf[j];
                    index++;
                }
                /* 12列目移行のデータを参照する */
                else if (column >= HAPMAP_DATA) {
                    /* 遺伝子データを格納 */
                    snpData->SNPdata[dataIndex] = buf[j];
                    dataIndex++;
                }
            }
        }
    }

    return retval;
}

/*********************************************/
/* phasingされたHapmapデータを構造体に収める */
/*********************************************/
int DataReaderSetAllPhasingHapmapData(FILE *fp, SnpData *snpData, long line)
{
    int retval = 0;
    long i = 0;
    long sampleDataNum = 0;

    /* phasingされたHapmapデータのサンプル数を取得する */
    sampleDataNum = DataReaderGetPhasingHapmapSampleDataNum(fp);
    /* ファイルから読み込み */
    for (i = 1; i < line; i++) {
        /* SNPデータ格納用配列のメモリ確保 */
        snpData[i].SNPdata = (char*)malloc1Dim(sizeof(char), sampleDataNum);
        retval = DataReaderSetPhasingHapmapData(fp, &snpData[i], i);
    }

    return retval;
}

/*************************************************/
/* phasingされたHapmapデータ一行を構造体に収める */
/*************************************************/
int DataReaderSetPhasingHapmapData(FILE *fp, SnpData *snpData, long line)
{
    int i = 0;
    int j = 0;
    int retval = -1;    /* ファイル最後尾まで来ている場合は-1を返す */
    int column = 1;     /* データの列 */
    int refNum = 0;     /* SNPallelesのパターン数 */
    int index = 0;
    int maxIndex = 0;
    int dataIndex = 0;
    int flag = 0;
    char buf[1024];
    char tmp[20];

    /* 複数文字ずつ読み込む */
    while ( fgets(buf, 1024, fp) != NULL ) {
        maxIndex = strlen(buf);
        j = 0;
        /* １行読み込んだら値を保持して終了 */
        for (j = 0; j <= maxIndex; j++) {
            if ('\n' == buf[j]) {
                snpData->sampleNum = dataIndex / 2;
                snpData->sampleDataNum = dataIndex;
                retval = 0;
                dataIndex = 0;
                return retval;
            }
            /* ファイルの２行目以降から処理を行う */
            else if (line > 0) {
                /* ファイル内の列をカウント */
                if (isspace(buf[j]) != 0) {
                    if (PHASE_RS_NUM == column) {
                        snpData->rsNumber[index] = '\0';
                    }
                    else if (PHASE_POS == column) {
                        tmp[index] = '\0';
                        snpData->pos = atol(tmp);
                    }
                    column++;
                    index = 0;
                }
                /* rs#を取得する */
                else if (PHASE_RS_NUM == column) {
                    if (isalnum(buf[j]) != 0) {
                        snpData->rsNumber[index] = buf[j];
                        index++;
                    }
                }
                /* posを取得する */
                else if (PHASE_POS == column) {
                    tmp[index] = buf[j];
                    index++;
                }
                /* 3列目以降のデータを参照する */
                else if (column >= PHASE_DATA) {
                    /* Nと一致するデータをカウント */
                    if ('N' != buf[j]){
                        for (i = 0; i < refNum; i+=2){
                            /* SNPallelesと一致するデータをカウント */
                            if (snpData->alleles[i] == buf[j]){
                                flag = 1;
                            }
                        } 
                        /* フラグがたっていない場合、２つ目（３つ目）のSNPallelesを取得 */
                        if (flag == 0) {
                            if (refNum != 0) {
                                snpData->alleles[refNum-1] = '/';
                            }
                            /* SNPallelesを取得 */
                            snpData->alleles[refNum] = buf[j];
                            snpData->alleles[refNum+1] = '\0';
                            refNum += 2;
                        }
                        /* フラグのクリア */
                        flag = 0;
                    }
                    /* 遺伝子データを格納 */
                    snpData->SNPdata[dataIndex] = buf[j];
                    dataIndex++;
                }
            }
        }
    }

    return retval;
}

/*****************************************/
/* 構造体SnpDataのメモリを確保する */
/*****************************************/
int DataReaderSnpDataMemoryAllocate(SnpData *snpData, long dataNum, long sampleDataNum)
{
    long i = 0;

    /* ファイルから読み込み */
    for (i = 0; i < dataNum; i++) {
        /* SNPデータ格納用配列のメモリ確保 */
        snpData[i].SNPdata = (char*)malloc1Dim(sizeof(char), sampleDataNum);
    }

    return 0;
}

/*****************************************/
/* 構造体SnpDataメンバのメモリを開放する */
/*****************************************/
int DataReaderSnpDataMemoryFree(SnpData *snpData, long dataNum)
{
    long i = 0;

    if (NULL != snpData) {
        for (i = 0; i < dataNum; i++) {
            /* SNPデータ格納用配列のメモリ開放 */
            free1Dim(snpData[i].SNPdata);
        }
        free1Dim(snpData);
    }

    return 0;
}

/*****************************************/
/* LDブロックを配列に格納する            */
/*****************************************/
int DataReaderSetLDBlock(FILE *fp, long *ldBlock)
{
    long index = 0;
    long strIndex = 0;
    long valIndex = 0;
    char string[MAX_LEN];
    char value[MAX_LEN];

    /* 一行ずつ読み込む */
    while ( (fgets(string, MAX_LEN, fp)) != NULL) {
        strIndex = 0;
        valIndex = 0;
        while (string[strIndex] != '\n') {
            value[valIndex] = string[strIndex];
            strIndex++;
            valIndex++;
        }
        value[valIndex] = '\0';
        ldBlock[index] = atol(value);
        index++;
    }
    return 0;
}

/******************************************/
/* 入力データのサンプルデータ数を取得する */
/******************************************/
long DataReaderGetSampleDataNum(FILE *fp, int dataType)
{
    long retval = 0;

    switch(dataType) {
        case 0:
            /* 入力データがHapmap */
            retval = DataReaderGetHapmapSampleDataNum(fp);
            break;
        case 1:
            /* 入力データがPhasingHapmap */
            retval = DataReaderGetPhasingHapmapSampleDataNum(fp);
            break;
        default:
            break;
    }

    return retval;
}

/******************************************/
/* HapmapDataのサンプルデータ数を取得する */
/******************************************/
long DataReaderGetHapmapSampleDataNum(FILE *fp)
{
    int character = 0;
    long column = 1;     /* Haplotypeデータの列 */
    long dataNum = 0;    /* サンプル人数 */

    /* １文字ずつ読み込む */
    while ( feof(fp) == 0 ) {
        character = fgetc(fp);
        /* １行読み込んだら値を返して終了 */
        if ('\n' == character) {
            return dataNum * 2;
        }
        /* ファイル内の列をカウント */
        else if (isspace(character) != 0) {
            column++;
            /* 12列目移行のデータ数をカウント */
            if (column >= HAPMAP_DATA) {
                dataNum++;
            }
        }
    }

    /* HapmapDataは一人（１列）で２つの遺伝子データを持っているので２倍する */
    return dataNum * 2;
}

/*********************************************************/
/* phasingされたHapmapデータのサンプルデータ数を取得する */
/*********************************************************/
long DataReaderGetPhasingHapmapSampleDataNum(FILE *fp)
{
    int character = 0;
    long column = 1;     /* phasingされたHapmapデータの列 */
    long dataNum = 0;    /* サンプル数 */

    /* １文字ずつ読み込む */
    while ( feof(fp) == 0 ) {
        character = fgetc(fp);
        /* １行読み込んだら値を返して終了 */
        if ('\n' == character) {
            return dataNum;
        }
        /* ファイル内の列をカウント */
        else if (isspace(character) != 0) {
            column++;
            /* 3列目移行のデータ数をカウント */
            if (column >= PHASE_DATA){
                dataNum++;
            }
        }
    }
    return dataNum;
}
