/***********************************************************************
 * Copyright(C) 2006 Valtech Co.,Ltd.
 * All Rights Reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 ***********************************************************************/
package jp.valtech.bts.util;

import java.io.BufferedOutputStream;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.OutputStream;
import java.io.Serializable;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;

import jp.valtech.bts.ui.BtsPlugin;

/**
 * <dl><dt><b>I/O操作のためのユーティリティクラス</b></dt>
 * <dd></dd>
 * </dl>
 * 
 * @author		<A href="mailto:m_sugitou@valtech.jp">M.Sugito</A>
 * @version	Ver.0.8
 */
public class IOUtility implements Logging{



    /**
     * <DL><DT><B>バイナリデータをGZIP圧縮します。</B></DL>
     *  <DD></DD>
     * </DL>
     * @param data			圧縮前のバイナリデータ
     * @return				圧縮されたデータ
     * @throws IOException	I/Oエラー
     */
    public static byte[] compress( byte[] data ) throws IOException{
    	byte[] result = null;
    	ByteArrayOutputStream baos = null;
    	GZIPOutputStream os = null;
    	
    	try{
        	baos = new ByteArrayOutputStream( data.length );
        	os = new GZIPOutputStream( baos , data.length );
        	
        	os.write( data );
        	os.finish();
        	os.flush();
        	
        	result = baos.toByteArray();

    	}finally{
    		if( baos != null ){
    			try {
					baos.flush();
				} catch ( IOException e ) {
					;
				}
    			try {
					baos.close();
				} catch ( IOException e ) {
					;
				}
    		}
    		if( os != null ){
    			try{
    				os.flush();
    			} catch ( IOException e ) {
					;
				}
    			try {
					os.close();
				} catch ( IOException e ) {
					;
				}
    		}
    	}

    	return result;
    }

    /**
     * <DL><DT><B>GZIP圧縮されたデータを展開します</B></DL>
     *  <DD></DD>
     * </DL>
     * @param data			GZIP圧縮されたバイナリデータ
     * @return				展開されたデータ
     * @throws IOException	I/Oエラー
     */
    public static byte[] decompress( byte[] data ) throws IOException{
    	byte[] result = null;
    	GZIPInputStream is = null;

    	try{
    		is = new GZIPInputStream( new ByteArrayInputStream( data ) );
    		result = read( is );
    	}finally{
    		if( is != null ){
    			try {
					is.close();
				} catch ( IOException e ) {
					;
				}
    		}
    	}
    	
    	return result;
    }

    /**
     * 引数の入力ストリームからバイナリデータを読み込みます。
     * 
     * @param is    入力ストリーム
     * @return  バイナリデータ
     * @throws IOException  ストリーム読み込み時に例外が発生した場合
     */
    public static byte[] read( InputStream is ) throws IOException{
        byte[] b = null;
        
        ByteArrayOutputStream baos = null;
        
        try{
            baos = new ByteArrayOutputStream();
            int     buf = 1024 * 32;
            byte[]  data = new byte[ buf ];
            int     size = 0;
            
            while( ( size = is.read( data , 0 , buf ) ) > 0 ){
                baos.write( data , 0 , size );
            }

            baos.flush();
            b = baos.toByteArray();
        }finally{
            if( baos != null ){
                baos.close();
            	baos = null;
            }
        }
        
        return b;
    }

    public static byte[] read(String filePath)  throws IOException ,FileNotFoundException{
    	
    	byte[] data = null;
    	FileInputStream is = null;
    	try {
    		File file = new File(filePath);
    		data = new byte[(int)file.length()];
    		
    		is = new FileInputStream(file);
    		is.read(data);
    		
    	} finally {
    		is.close();
    	}
    	return data;
	}


	/**
	 * 引数の出力ストリームに対して、バイナリデータを書き出します。
	 * 
	 * @param os				出力ストリーム
	 * @param data				出力ストリームに向けて書き出すバイナリデータ
	 * @throws IOException		ストリーム読み込み時に例外が発生した場合
	 */
	public static void write( OutputStream os , byte[] data ) throws IOException{

		os.write( data );

	}
	

	/**
	 * バイナリデータをファイルとして書き込みます。
	 * 
	 * @param f		ファイルのパス。存在する場合は上書きされます。
	 * @param data		書き込むデータ(バイナリ形式)
	 * @throws IOException	ファイル書き込み中に例外が発生した場合
	 */
	public static void write( File f , byte[] data ) throws IOException{
		BufferedOutputStream bos = null;
		
		try {
			bos = new BufferedOutputStream( new FileOutputStream( f ) );
			write( bos , data );
		} catch (IOException e) {
			throw e;
		} finally {
			if( bos != null ){
				try{
					bos.flush();
					bos.close();
					bos = null;
				}catch( IOException e ){
				}
			}
		}
	}
	public static Object toObject(byte[] data){
		try {
			ByteArrayInputStream byteIn =new ByteArrayInputStream(data);
			ObjectInputStream in = new ObjectInputStream(byteIn);
			
			return in.readObject();
			
		} catch (Exception e) {
			BtsPlugin.getInstance().errorlog(e);
			return null;
		}
	}
	
	public static byte[] toByte(Serializable obj){
		if (obj == null) {
			return null;
		}
		try {
			ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
			ObjectOutputStream out = new ObjectOutputStream(byteOut);
			out.writeObject(obj);
			return byteOut.toByteArray();
		} catch (Exception e) {
			BtsPlugin.getInstance().errorlog(e);
			return null;
		}
	}


}
