/***********************************************************************
 * Copyright(C) 2006 Valtech Co.,Ltd.
 * All Rights Reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 ***********************************************************************/
package jp.valtech.bts.util;

import java.io.FileOutputStream;
import java.io.ObjectOutput;
import java.io.ObjectOutputStream;
import java.util.Date;
import java.util.Properties;

import javax.activation.DataHandler;
import javax.activation.FileDataSource;
import javax.mail.Message;
import javax.mail.Session;
import javax.mail.Transport;
import javax.mail.internet.InternetAddress;
import javax.mail.internet.MimeBodyPart;
import javax.mail.internet.MimeMessage;
import javax.mail.internet.MimeMultipart;

import jp.valtech.bts.data.CurrentProject;
import jp.valtech.bts.data.Issue;
import jp.valtech.bts.data.IssueHistory;
import jp.valtech.bts.data.IssueStatus;
import jp.valtech.bts.ui.BtsPlugin;
import jp.valtech.bts.ui.navigator.ProjectConfig;

import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.widgets.Display;

/**
 * 課題票配信時に課題票を添付した更新通知メールを送ります。
 * メールの設定はプロジェクト設定ダイアログにて行います。
 * 
 * @author		<A href="mailto:s_imai@valtech.jp">S.Imai</A>
 * @version	Ver.0.8
 */
public class BtsMailSender implements Runnable {

	/** 課題票 */
	private Issue issue;
	
	/** 現在開いているプロジェクト情報 */
	private ProjectConfig projectConfig;


	/**
	 * メールを送信します。
	 */
	public void run() {

		try {
			
			// プロジェクト情報設定
			projectConfig = CurrentProject.getInsance().getProjectConfig();
			
			// 現在開いているプロジェクトのメール設定取得
			String smtp = projectConfig.getSmtp();		// SMTPサーバ名
			String from = projectConfig.getSender();	// 送信元メールアドレス
			String to = projectConfig.getRecipient();	// 送信先メールアドレス
			String subject = projectConfig.getPrefix(); // メール題名
			
			// メール本文作成
			String body = createBody();
			
			// Issueオブジェクトのパスを設定
			String currentDir = BtsPlugin.getInstance().getStateLocation() + "/";
			String attachmentFilePath = currentDir + "tmp.bts";
			
			// Issueオブジェクトの出力
			FileOutputStream fileOutputStream = new FileOutputStream(attachmentFilePath);
			ObjectOutput objectOutput = new ObjectOutputStream(fileOutputStream);
			objectOutput.writeObject(issue);
			objectOutput.flush();
			
			// 添付ファイル名
			String tmpFileName = BTSUtility.formatDate(issue.getUpdateDate(), "yyyy.MM.dd.HH.mm.ss") + ".bts";
			
			
			// SMTPサーバ名を設定
			Properties props = new Properties();
			props.put("mail.smtp.host", smtp);
			
			// メールセッションを確立
			Session session = Session.getInstance(props, null);
			
			// タイトル
			subject = subject + Messages.getString("BtsMailSender.15") + issue.getDisplayIssueID() + "]" +issue.getTitle(); //$NON-NLS-1$
			
			// メール作成
			MimeMessage mimeMessage = new MimeMessage(session);
			mimeMessage.addFrom(InternetAddress.parse(from));
			mimeMessage.setRecipients(Message.RecipientType.TO,InternetAddress.parse(to));
			mimeMessage.setSubject(subject, "iso-2022-jp");
			mimeMessage.setSentDate(new Date());

			// マルチパートメッセージ生成
			MimeMultipart content = new MimeMultipart();

			// テキスト部作成
			MimeBodyPart text = new MimeBodyPart();
			text.setText(body, "iso-2022-jp");
			content.addBodyPart(text);

			// 添付ファイル部作成
			MimeBodyPart attachmentFile = new MimeBodyPart();
			attachmentFile.setDataHandler(new DataHandler(new FileDataSource(attachmentFilePath)));
			attachmentFile.setFileName(tmpFileName);
			content.addBodyPart(attachmentFile);

			// メールにマルチパートメッセージをセット
			mimeMessage.setContent(content);

			// メッセージ送信
			Transport.send(mimeMessage);
			
			
		} catch (Exception e) {

			Display.getDefault().syncExec(new Runnable() {
				public void run() {
					MessageDialog.openError(BtsPlugin.getInstance().getActiveEditor().getSite().getShell(),
					                        Messages.getString("BtsMailSender.0"),  //$NON-NLS-1$
					                        Messages.getString("BtsMailSender.1")); //$NON-NLS-1$
				}
			});
		}
	}

	/**
	 * メール本文を作成します。
	 * 
	 * @return			メール本文
	 */
	public String createBody() throws Exception {
		
		// 更新履歴取得
		IssueHistory[] issueHistories = issue.getIssueHistories();
		
		// 本文を作成
		StringBuffer body = new StringBuffer();
		body.append(Messages.getString("BtsMailSender.2")); //$NON-NLS-1$
		body.append(Messages.getString("BtsMailSender.7")); //$NON-NLS-1$
		body.append(Messages.getString("BtsMailSender.16")); //$NON-NLS-1$
		body.append(Messages.getString("BtsMailSender.8")); //$NON-NLS-1$
		body.append("==================================================\n");
		body.append(Messages.getString("BtsMailSender.3") + projectConfig.getProjectName() + "\n"); //$NON-NLS-1$
		body.append(Messages.getString("BtsMailSender.4") + CurrentProject.getInsance().getSSID() + "\n"); //$NON-NLS-1$
		body.append(Messages.getString("BtsMailSender.5") + issueHistories[0].getUpdateUser() + "\n"); //$NON-NLS-1$
		body.append(Messages.getString("BtsMailSender.6") + BTSUtility.formatDate(issueHistories[0].getUpdateDate(), "yyyy/MM/dd HH:mm:ss") + "\n"); //$NON-NLS-1$
		body.append("==================================================" + "\n\n");
		body.append(Messages.getString("BtsMailSender.9")); //$NON-NLS-1$
		body.append(Messages.getString("BtsMailSender.17") + issue.getVersion()); //$NON-NLS-1$
		body.append(Messages.getString("BtsMailSender.18") + issue.getDisplayIssueID()); //$NON-NLS-1$
		body.append(Messages.getString("BtsMailSender.19") + issue.getTitle()); //$NON-NLS-1$
		body.append(Messages.getString("BtsMailSender.20") + issue.getPriority()); //$NON-NLS-1$
		body.append(Messages.getString("BtsMailSender.21") + IssueStatus.getEnumFromValue(issue.getStatus()).getDescription()); //$NON-NLS-1$
		body.append(Messages.getString("BtsMailSender.22") + issue.getDescription()); //$NON-NLS-1$
		
		body.append(Messages.getString("BtsMailSender.23")); //$NON-NLS-1$
		
		for(int i=0; i<issueHistories.length; i++) {
			
			// 課題票更新日時と履歴日時が同じ履歴のみ表示
			if(issue.getUpdateDate().equals(issueHistories[i].getUpdateDate())) {
				
				// 更新属性取得
				String updateAttribute = issueHistories[i].getUpdateAttribute();
				
				// 更新属性が「(起票)」,「(更新なし配信)」,「(ごみ箱に移動)」の場合
				if(IssueHistory.ISSUE_NON_UPDATE.equals(updateAttribute)
					|| IssueHistory.ISSUE_INITIAL_REGIST.equals(updateAttribute)
					|| IssueHistory.ISSUE_MOVE_GARBAGE.equals(updateAttribute)) {
					
					body.append(updateAttribute + Messages.getString("BtsMailSender.10")); //$NON-NLS-1$
					
				}
				
				else {
					
					// 更新前の完了期限
					String beforeDate = issueHistories[i].getUpdateBefore();
					if(beforeDate == null) {
						beforeDate = "";
					}
					
					// 更新後の完了期限
					String afterDate = issueHistories[i].getUpdateAfter();
					if(afterDate == null) {
						afterDate = "";
					}
					
					body.append(Messages.getString("BtsMailSender.11") + issueHistories[i].getUpdateAttribute() + Messages.getString("BtsMailSender.12")); //$NON-NLS-1$ //$NON-NLS-2$
					body.append(Messages.getString("BtsMailSender.13")); //$NON-NLS-1$
					body.append(beforeDate + "\n\n");
					body.append(Messages.getString("BtsMailSender.14")); //$NON-NLS-1$
					body.append(afterDate + "\n\n");	
					
				}
				
				body.append("--------------------------------------------------\n");
				
			}
			
		}

		return body.toString();

	}
	
	
	public void setIssue(Issue issue) {
		this.issue = issue;
	}

}
