/***********************************************************************
 * Copyright(C) 2006 Valtech Co.,Ltd.
 * All Rights Reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 ***********************************************************************/
package jp.valtech.bts.ui.navigator;

import jp.valtech.bts.data.Issue;
import jp.valtech.bts.data.IssueType;
import jp.valtech.bts.network.ClientInfo;
import jp.valtech.bts.ui.BtsPlugin;

import org.eclipse.jface.viewers.TreeViewer;

/**
 * ツリー操作の実装を補助する機能を集めたクラスです。
 *
 * @author		<A href="mailto:m_sugitou@valtech.jp">M.Sugito</A>
 * @version	Ver.0.8
 */
public class TreeHelper {

	
	/**
	 * プロジェクトノードの表示を更新します。配下のノードまでは更新しません。
	 * 
	 * @param		ssid			SSID
	 * @param		projectName		プロジェクト名
	 */
	public static void modifyProjectNode(String ssid, String projectName) {
		TreeViewer treeViewer = BtsPlugin.getInstance().getNavigatorView().getViewer();

		// ビューアからツリーオブジェクトを取得
		BtsTreeNode rootNode = (BtsTreeNode)treeViewer.getInput();

		// 「現在の参加者」をあらわすノードを取得
		ProjectNode projectNode = (ProjectNode)rootNode.getChild(ssid);
		projectNode.setTextName(projectName);
		
		treeViewer.refresh(projectNode);
	}
	
	
	/**
	 * プロジェクトノードの表示を更新します。配下のノードまでは更新しません。
	 * 
	 * @param		ssid			SSID
	 * @param		projectName		プロジェクト名
	 */
	public static void addProjectNode(String ssid, String projectName) {

		TreeViewer treeViewer = BtsPlugin.getInstance().getNavigatorView().getViewer();
		
		// ビューアからツリーオブジェクトを取得
		BtsTreeNode rootNode = (BtsTreeNode)treeViewer.getInput();

		// プロジェクトノード生成
		ProjectNode projectNode = new ProjectNode(rootNode, projectName, ssid);
    	rootNode.addChild(projectNode);

		// ビューを更新する
    	treeViewer.add(rootNode, projectNode);
	}

	
	/**
	 * 指定のSSIDのプロジェクトノードの表示をOPENにします。
	 * 
	 * @param		ssid			SSID
	 */
	public static void openProjectNode(String ssid) {
		TreeViewer treeViewer = BtsPlugin.getInstance().getNavigatorView().getViewer();

		// ビューアからツリーオブジェクトを取得
		BtsTreeNode rootNode = (BtsTreeNode)treeViewer.getInput();

		// 「現在の参加者」をあらわすノードを取得
		ProjectNode projectNode = (ProjectNode)rootNode.getChild(ssid);
		projectNode.open();
		
		treeViewer.update(projectNode, null);
	}

	
	/**
	 * 指定のSSIDのプロジェクトノードの表示をCLOSEにします。
	 * 
	 * @param		ssid			SSID
	 */
	public static void closeProjectNode(String ssid) {
		TreeViewer treeViewer = BtsPlugin.getInstance().getNavigatorView().getViewer();

		// ビューアからツリーオブジェクトを取得
		BtsTreeNode rootNode = (BtsTreeNode)treeViewer.getInput();

		// 指定のSSIDのプロジェクトノード取得
		ProjectNode projectNode = (ProjectNode)rootNode.getChild(ssid);

		// ツリーのクローズ処理
		projectNode.close();
		treeViewer.update(projectNode, null);
		treeViewer.refresh();
		treeViewer.collapseAll();
	}


	/**
	 * 指定のSSIDのプロジェクトノードを削除します。。
	 * 
	 * @param		ssid			SSID
	 */
	public static void deleteProjectNode(String ssid) {
		TreeViewer treeViewer = BtsPlugin.getInstance().getNavigatorView().getViewer();

		// ビューアからツリーオブジェクトを取得
		BtsTreeNode rootNode = (BtsTreeNode)treeViewer.getInput();

		// 指定のSSIDのプロジェクトノード取得
		ProjectNode projectNode = (ProjectNode)rootNode.getChild(ssid);

		// ツリーからプロジェクトを削除
		rootNode.removeChild(projectNode);
		treeViewer.remove(projectNode);
	}
	
	
	/**
	 * ユーザ名表示を更新します。
	 * 
	 * @param		ssid			SSID
	 * @param		clientInfo		クライアント情報
	 */
	public static void modifyMemberName(String ssid, ClientInfo clientInfo) {
		TreeViewer treeViewer = BtsPlugin.getInstance().getNavigatorView().getViewer();

		// ビューアからツリーオブジェクトを取得
		BtsTreeNode rootNode = (BtsTreeNode)treeViewer.getInput();

		// 参加者リストノードを取得
		ProjectNode projectNode = (ProjectNode)rootNode.getChild(ssid);
		MemberListNode memberListNode = projectNode.getMemberListNode();
		
		// ユーザ名を更新
		BtsTreeNode membernode = memberListNode.getChild(clientInfo.getClientAddress());
		if(membernode != null) {
			membernode.setTextName(clientInfo.getClientName());
			treeViewer.refresh(memberListNode);
		}
	}

	
	/**
	 * 課題票のカウントをインクリメントします。
	 * インクリメントするフォルダノードは<code>newIssue</code>が持つ{@link IssueType 課題票種別}で判断します。
	 * 
	 * @param		issue				新規追加の課題票
	 * @param		projectNode			対象となるプロジェクト
	 */
	public static void incrementIssueCount(Issue issue, ProjectNode projectNode) {

		// 下書き保存の場合
		if( IssueType.DRAFT_VALUE.equals(issue.getType()) ){
			DraftNode draftNode = projectNode.getDraftNode();
			draftNode.incrementChildCount();
			return;
		}

		// 競合の場合
		else if( IssueType.CONFLICT_VALUE.equals(issue.getType()) ){
			ConflictNode conflictNode = projectNode.getConflictNode();
			conflictNode.incrementChildCount();
			return;
		}

		// ごみばこの場合
		else if( IssueType.GARBAGE_VALUE.equals(issue.getType()) ){
			GarbageBoxNode garbageBoxNode = projectNode.getGarbageBoxNode();
			garbageBoxNode.incrementChildCount();
			return;
		}

		// 通常の課題票（カテゴリに追加する）
		else {
			// カテゴリノード取得
			ReleaseNode releaseNode = projectNode.getReleaseNode();
			CategoryNode ctgryNode = releaseNode.getCategoryNode(issue.getCategory());
	
			// 新しいカテゴリの場合は新規カテゴリノードを作る
			if(ctgryNode == null) {
				ctgryNode = new CategoryNode(releaseNode, issue.getCategory());
				releaseNode.addChild(ctgryNode);
			}
			ctgryNode.incrementChildCount();
		}
	}

	
	/**
	 * 課題票のカウントをデクリメントします。
	 * デクリメントするフォルダノードは<code>newIssue</code>が持つ{@link IssueType 課題票種別}で判断します。
	 * 
	 * @param		issue				新規追加の課題票
	 * @param		projectNode			対象となるプロジェクト
	 */
	public static void decrementIssueCount(Issue issue, ProjectNode projectNode) {

		// 下書き保存の場合
		if( IssueType.DRAFT_VALUE.equals(issue.getType()) ){
			DraftNode draftNode = projectNode.getDraftNode();
			draftNode.decrementChildCount();
			return;
		}

		// 競合の場合
		else if( IssueType.CONFLICT_VALUE.equals(issue.getType()) ){
			ConflictNode conflictNode = projectNode.getConflictNode();
			conflictNode.decrementChildCount();
			return;
		}

		// ごみばこの場合
		else if( IssueType.GARBAGE_VALUE.equals(issue.getType()) ){
			GarbageBoxNode garbageBoxNode = projectNode.getGarbageBoxNode();
			garbageBoxNode.decrementChildCount();
			return;
		}

		// 通常の課題票（カテゴリに追加する）
		else {
			// カテゴリノード取得
			ReleaseNode releaseNode = projectNode.getReleaseNode();
			CategoryNode ctgryNode = releaseNode.getCategoryNode(issue.getCategory());
			
			// 念のためnullチェック
			if(ctgryNode != null) {
				ctgryNode.decrementChildCount();
			}
		}
		
	}

}
