/***********************************************************************
 * Copyright(C) 2006 Valtech Co.,Ltd.
 * All Rights Reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 ***********************************************************************/
package jp.valtech.bts.ui.msglist;

import java.util.List;

import jp.valtech.bts.data.Issue;
import jp.valtech.bts.data.MessagePacket;
import jp.valtech.bts.facade.MessagePacketFacade;
import jp.valtech.bts.network.ClientInfo;
import jp.valtech.bts.ui.BtsPlugin;
import jp.valtech.bts.ui.IBtsListener;
import jp.valtech.bts.ui.issuelist.Messages;
import jp.valtech.bts.util.Logging;

import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.widgets.Control;

/**
 * {@link MessageListView メッセージ一覧ビュー}のプロバイダクラスです。
 * 
 * {@link jp.valtech.bts.ui.IBtsListener リスナーインタフェース}を実装することにより、BTS情報に更新があった際には更新通知を受けとることができます。<BR>
 * 更新通知を受けとると{@link MessageListView メッセージ一覧ビュー}に表示を反映します。
 * 
 * @author		<A href="mailto:m_sugitou@valtech.jp">M.Sugito</A>
 * @version	Ver.0.8
 */
public class MessageListViewContentProvider implements IStructuredContentProvider, IBtsListener, Logging {

	
	/** メッセージ一覧ビューのビューアオブジェクト */
	private TableViewer viewer;

	
	/**
	 * 当該リスナインスタンスををプラグインに登録します。
	 * 
	 */
	public MessageListViewContentProvider() {
		super();
		
		// 当該リスナインスタンスをプラグインに登録します。
		BtsPlugin.getInstance().addListener(this);
	}

	/*
	 *  (非 Javadoc)
	 * @see org.eclipse.jface.viewers.IContentProvider#dispose()
	 */
	public void dispose() {
		BtsPlugin.getInstance().removeListener(this);
	}
	
	/*
	 *  (非 Javadoc)
	 * @see org.eclipse.jface.viewers.IContentProvider#inputChanged(org.eclipse.jface.viewers.Viewer, java.lang.Object, java.lang.Object)
	 */
	public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {
		this.viewer= (TableViewer)viewer;
	}
	
	
	/*
	 *  (非 Javadoc)
	 * @see org.eclipse.jface.viewers.IStructuredContentProvider#getElements(java.lang.Object)
	 */
	public Object[] getElements(Object inputElement) {
		return ((List)inputElement).toArray();
	}

	
	/*
	 *  (非 Javadoc)
	 * @see jp.valtech.bts.ui.IBtsListener#addIssue(jp.valtech.bts.data.Issue)
	 */
	public void addIssue(Issue issue) {
		// 何もしない		
	}

	/*
	 *  (非 Javadoc)
	 * @see jp.valtech.bts.ui.IBtsListener#updateIssue(jp.valtech.bts.data.Issue)
	 */
	public void updateIssue(Issue oldIssue, Issue newIssue) {
		// 何もしない		
	}

	/*
	 *  (非 Javadoc)
	 * @see jp.valtech.bts.ui.IBtsListener#deleteIssue(java.lang.String, java.lang.String)
	 */
	public void deleteIssue(Issue deleteIssue) {
		// 何もしない
	}
	

	/*
	 *  (非 Javadoc)
	 * @see jp.valtech.bts.ui.IBtsListener#joinMember(jp.valtech.bts.network.ClientInfo)
	 */
	public void joinMember(ClientInfo clientInfo) {
		// 何もしない
	}
	
	/*
	 *  (非 Javadoc)
	 * @see jp.valtech.bts.ui.IBtsListener#leaveMember(jp.valtech.bts.network.ClientInfo)
	 */
	public void leaveMember(ClientInfo clientInfo) {
		// 何もしない
	}
	
	
	/**
	 * メッセージ追加。
	 * {@link MessagePacket 指定のメッセージ}をビューに追加します。
	 * 
	 * @param		messagePacket		指定のメッセージ 	
	 */
	public void addMessage(final MessagePacket messagePacket) {
		// テーブルビューアがnullの時は何もしない
		if(viewer == null ) {
			return ;
		}
		// 実データの更新
		final List msgList = (List)viewer.getInput();
		msgList.add(messagePacket);

		// メインスレッドで実行されるようにRunnableをメインスレッドに渡します。
		Control ctrl= viewer.getControl();
		ctrl.getDisplay().syncExec(new Runnable() {
			public void run() {
				try {
					viewer.insert(messagePacket, 0);

				} catch (Exception e) {
					String msg = Messages.getString("MessageListViewContentProvider.0"); //$NON-NLS-1$
					logger.fatal(msg, e);
					BtsPlugin.getInstance().error(msg, e);
				}
			}
		});
	}


	/**
	 * 指定のメッセージを削除します。
	 * 
	 * @param		messagePacket			メッセージ情報
	 */
	public void deleteMessage(final MessagePacket messagePacket) {
		// テーブルビューアがnullの時は何もしない
		if(viewer == null ) {
			return ;
		}

		// メインスレッドで実行されるようにRunnableをメインスレッドに渡します。
		Control ctrl= viewer.getControl();
		ctrl.getDisplay().syncExec(new Runnable() {
			public void run() {
				try {
					// テーブルの実データを削除
					List messageList = (List)viewer.getInput();
					messageList.remove(messagePacket);

					// 一覧表示の更新
					viewer.remove(messagePacket);
	
				}catch (Exception e) {
					String msg = Messages.getString("MessageListViewContentProvider.1"); //$NON-NLS-1$
					logger.fatal(msg, e);
					BtsPlugin.getInstance().error(msg, e);
				}
			}
		});
	}

	
	public void refresh(List issueList) {
		// テーブルビューアがnullの時は何もしない
		if(viewer == null ) {
			return ;
		}
		
		// メインスレッドで実行されるようにRunnableをメインスレッドに渡します。
		Control ctrl= viewer.getControl();
		ctrl.getDisplay().syncExec(new Runnable() {
			public void run() {
				try {
					// 一旦データをクリアする　
					List oldMessageList = (List)viewer.getInput();
					if(oldMessageList != null) {
						oldMessageList.clear();
					}
					viewer.refresh();

					// DBから取得したデータを表示する
					MessagePacketFacade messagePacketFacade = new MessagePacketFacade();
					List newMessageList = messagePacketFacade.getMessageList();
					viewer.setInput( newMessageList );

				} catch (Exception e) {
					String msg = Messages.getString("MessageListViewContentProvider.2"); //$NON-NLS-1$
					logger.fatal(msg, e);
					BtsPlugin.getInstance().error(msg, e);
				}
			}
		});
	}

	
	/**
	 * メッセージ一覧ビューをクリアします。
	 */
	public void clear() {
		// テーブルビューアがnullの時は何もしない
		if(viewer == null ) {
			return ;
		}
		
		// メインスレッドで実行されるようにRunnableをメインスレッドに渡します。
		Control ctrl= viewer.getControl();
		ctrl.getDisplay().syncExec(new Runnable() {
			public void run() {
				try {
					// 一旦データをクリアする　
					List list = (List)viewer.getInput();
					list.clear();
					viewer.refresh();

				} catch (Exception e) {
					String msg = Messages.getString("MessageListViewContentProvider.3"); //$NON-NLS-1$
					logger.fatal(msg, e);
					BtsPlugin.getInstance().error(msg, e);
				}
			}
		});
	}
}
