/***********************************************************************
 * Copyright(C) 2006 Valtech Co.,Ltd.
 * All Rights Reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 ***********************************************************************/
package jp.valtech.bts.ui.msglist;


import java.util.ArrayList;

import jp.valtech.bts.data.CurrentProject;
import jp.valtech.bts.data.MessagePacket;
import jp.valtech.bts.data.MessageType;
import jp.valtech.bts.facade.MessagePacketFacade;
import jp.valtech.bts.ui.BtsPlugin;
import jp.valtech.bts.ui.IBtsViewPart;
import jp.valtech.bts.ui.action.DeleteMessageAction;
import jp.valtech.bts.ui.action.OpenIssueAction;
import jp.valtech.bts.ui.action.OpenMessageDlgAction;
import jp.valtech.bts.ui.action.OpenMsgFilterDlgAction;
import jp.valtech.bts.ui.action.DownloadMessageAttachmentAction;
import jp.valtech.bts.util.Logging;

import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.IToolBarManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.dialogs.IDialogSettings;
import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.IDoubleClickListener;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.TableLayout;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.SashForm;
import org.eclipse.swt.events.KeyAdapter;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.IWorkbenchActionConstants;
import org.eclipse.ui.part.ViewPart;


/**
 * メッセージ一覧ビューです。
 * 
 * @author		<A href="mailto:m_sugitou@valtech.jp">M.Sugito</A>
 * @version	Ver.0.8
 */
public class MessageListView extends ViewPart implements IBtsViewPart, Logging {
	
	/** 「メッセージ送信」のアクション */
	private OpenMessageDlgAction openMsgDlgAction;
	/** 「メッセージ返信」のアクション */
	private OpenMessageDlgAction replyMsgDlgAction;
	/** 「フィルター」ボタンを押した時のアクション */
	private OpenMsgFilterDlgAction openFilterDlgAction;
	/** 添付ファイル保存アクション */
	private DownloadMessageAttachmentAction saveAttachmentAction;
	/** 課題票を開くアクション */
	private OpenIssueAction openIssueAction;
	/** メッセージを削除するアクション */
	private DeleteMessageAction deleteMessageAction;
	
	/** メッセージ一覧テーブルのビューアインスタンス */
	private TableViewer viewer;
	
	/** メッセージを表示するテキスト */
	private Text msgTxt;
	
	
	/**
	 * ビュー生成時に最初に呼ばれるメソッドです。
	 * 画面を組み立てます。
	 * 
	 * @see			org.eclipse.ui.part.WorkbenchPart#createPartControl(org.eclipse.swt.widgets.Composite)
	 * @param		parent				親コンポジット
	 */
	public void createPartControl(Composite parent) {
		try {
			// 左右分割したパネルを使う
			SashForm sashForm = new SashForm(parent, SWT.HORIZONTAL);
			sashForm.setLayoutData(new GridData(GridData.FILL_BOTH));

			// 左ペインにテーブルを配置する
			createMessageListTable(sashForm);
			
			// 右ペインに「メッセージ」と「参加者リスト」を配置する
			createRightPain(sashForm);
			
			// SashFormの左右の比率を設定
			sashForm.setWeights(new int[]{62, 38});
			
			// このビューに結びつくアクションを登録
			makeActions();
			
			// 右クリックメニューを設定
			hookContextMenu();
			
			// ダブルクリックの動作を設定
			hookDoubleClick();
			
			// シングルクリックの動作を設定
			hookSingleClick();
			
			// ツールバーを設定
			fillLocalToolBar();
			
			// フィルタ実行
			IDialogSettings section 
				= BtsPlugin.getInstance().getDialogSettings().getSection(MessageFilterManager.SECTION_NAME);
			if(section != null){
				MessageListViewFilter filter = new MessageListViewFilter(viewer); 
				filter.addMessageFilters();
			}

			
		}catch (Exception e) {
			String msg = Messages.getString("MessageListView.0"); //$NON-NLS-1$
			logger.fatal(msg, e);
			BtsPlugin.getInstance().error(msg, e);
		}
	}

	
	/**
	 * 選択しているレコードを返します。
	 * 選択しているレコードを{@link MessagePacket}にキャストして返します。
	 * 
	 * @return			キャスト済み選択レコード
	 */
	private MessagePacket getSelectionRecord() {
		
		// 選択しているレコードを取得します。
		ISelection selection = viewer.getSelection();
		Object obj = ((IStructuredSelection)selection).getFirstElement();
		if(obj instanceof MessagePacket) {
			// キャストして返します。
			return (MessagePacket)obj;
		}
		return null;
	}

	
	/**
	 * シングルクリックの動作を設定します。
	 * <ul>
	 *  <li>メッセージ本文を右ペインに表示します。
	 *  <li>選択したメッセージを「既読」に設定します。
	 * </ul>
	 */
	private void hookSingleClick() {
		viewer.addSelectionChangedListener(new ISelectionChangedListener() {
			public void selectionChanged(SelectionChangedEvent event) {
				Control ctrl= viewer.getControl();
				ctrl.getDisplay().syncExec(new Runnable() {
					public void run() {
						try {
							// 選択レコードを取得
							MessagePacket msg = getSelectionRecord();
							if(msg==null) {
								return;
							}
							// 右ペインのテキストにメッセージ全文を表示する
							msgTxt.setText(msg.getMessage());

							// 選択したメッセージが「未読」の場合は「既読」に設定する。
							if(!msg.isOpend().booleanValue()) {

								// DBのデータを「既読」に設定する
								MessagePacketFacade facade = new MessagePacketFacade();
								facade.setOpend(msg.getMessageID());

								// 「既読」表示にする 
								msg.setOpened(true);
								viewer.refresh();
							}
						} catch (Exception e) {
							String msg = Messages.getString("MessageListView.1"); //$NON-NLS-1$
							logger.fatal(msg, e);
							BtsPlugin.getInstance().error(msg, e);
						}
					}
				});
			}
		});
	}

	
	/**
	 * ダブルクリックの動作を設定します。
	 * 課題票関連のメッセージの場合は、当該課題票を開きます。
	 * 
	 */
	private void hookDoubleClick() {
		viewer.addDoubleClickListener(new IDoubleClickListener() {
			public void doubleClick(DoubleClickEvent event) {
				// 選択レコードを取得
				
				MessagePacket msg = getSelectionRecord();
				if(msg==null) {
					return;
				}
				
				// メッセージ送信画面を開くアクション
				replyMsgDlgAction.setFromUser(msg.getFromUser());
				replyMsgDlgAction.setReplyMessage(msg.getMessage());
				replyMsgDlgAction.run();
			}
		});
	}
	
	/**
	 * テーブルを右クリックしたときの動作です。
	 */
	private void hookContextMenu() {
		MenuManager menuMgr = new MenuManager("#PopupMenu"); 
		menuMgr.setRemoveAllWhenShown(true);
		menuMgr.addMenuListener(new IMenuListener() {
			public void menuAboutToShow(IMenuManager manager) {
				fillContextMenu(manager);
			}
		});
		Menu menu = menuMgr.createContextMenu(viewer.getControl());
		viewer.getControl().setMenu(menu);
		getSite().registerContextMenu(menuMgr, viewer);
	}

	
	/**
	 * テーブルを右クリックしたときの動作です。
	 * 全部仮実装
	 * 
	 * @param		manager			右クリックメニューのマネージャインスタンス
	 */
	private void fillContextMenu(IMenuManager manager) {
		
		// 選択レコードを取得
		MessagePacket msg = getSelectionRecord();
		if(msg==null) {
			// Other plug-ins can contribute there actions here
			manager.add(new Separator(IWorkbenchActionConstants.MB_ADDITIONS));
			return;
		}
		
		String msgType = msg.getMessageType();
		if( MessageType.ISSUE_VALUE.equals(msgType) ) {

			// 課題票を開くアクション
			openIssueAction.setIssueKeys( msg.getFingerPrint(), msg.getIssueType() );
			manager.add(openIssueAction);
			
		}else if ( MessageType.CONFLICT_VALUE.equals(msgType) ) {
			
			// 課題票を開くアクション
			openIssueAction.setIssueKeys( msg.getFingerPrint(), msg.getIssueType() );
			manager.add(openIssueAction);
			
		}else if ( MessageType.USER_VALUE.equals(msgType) ) {
			
			// 添付ファイルを保存するアクション
			if( msg.getAttachFileName()!=null && !msg.getAttachFileName().equals("")) {
				saveAttachmentAction.setMessagePacket(msg);
				manager.add(saveAttachmentAction);
			}
		}
		
		// メッセージ送信画面を開くアクション
		replyMsgDlgAction.setFromUser(msg.getFromUser());
		replyMsgDlgAction.setReplyMessage(msg.getMessage());
		manager.add(replyMsgDlgAction);
		manager.add(new Separator());

		// メッセージを削除するアクション
		deleteMessageAction.setMessagePacket(msg);
		manager.add(deleteMessageAction);
		
		// Other plug-ins can contribute there actions here
		manager.add(new Separator(IWorkbenchActionConstants.MB_ADDITIONS));
	}


	/**
	 * ツールバーの設定です。
	 * 「フィルターボタン」「メッセージ作成ボタン」を表示します。
	 */
	private void fillLocalToolBar() {
		IActionBars bars = getViewSite().getActionBars();
		IToolBarManager toolbar = bars.getToolBarManager(); 
		toolbar.add(openMsgDlgAction);
		toolbar.add(openFilterDlgAction);
	}


	
	/**
	 * アクションクラスの生成。
	 * このビューから呼び出されるアクションクラスを生成します。
	 */
	private void makeActions() {
		
		// 課題票を開くアクション
		openIssueAction = new OpenIssueAction(OpenIssueAction.OPEN_EXIST_ISSUE);

		//「メッセージ作成」ダイアログを開くアクション
		openMsgDlgAction = new OpenMessageDlgAction(this);
		openMsgDlgAction.setText(Messages.getString("MessageListView.2")); //$NON-NLS-1$
		openMsgDlgAction.setToolTipText(Messages.getString("MessageListView.3"));  //$NON-NLS-1$
		
		//「メッセージ返信」ダイアログを開くアクション
		replyMsgDlgAction = new OpenMessageDlgAction(this);
		replyMsgDlgAction.setText(Messages.getString("MessageListView.4")); //$NON-NLS-1$

		//「フィルターダイアログ」をポップアップ表示するアクション
		openFilterDlgAction = new OpenMsgFilterDlgAction(this);
		
		//「メッセージ削除」をポップアップ表示するアクション
		deleteMessageAction = new DeleteMessageAction();

		// 添付ファイルを保存するアクション
		saveAttachmentAction = new DownloadMessageAttachmentAction() ;
		
		// カレントプロジェクトに以下の2つのアクションを加える
		CurrentProject project = CurrentProject.getInsance();
		project.addAction(openFilterDlgAction);
		project.addAction(openMsgDlgAction);
		
    	// カレントプロジェクトのオープン／クローズにより活性／非活性の表示を分ける
		if(project.isOpen()) {
			openFilterDlgAction.setEnabled(true);
			openMsgDlgAction.setEnabled(true);
		} else {
			openFilterDlgAction.setEnabled(false);
			openMsgDlgAction.setEnabled(false);
		}
	}


	/**
	 * Passing the focus request to the viewer's control.
	 */
	public void setFocus() {
		viewer.getControl().setFocus();
	}

	/**
	 * 右ペインを生成します。
	 * 右ペインにはメッセージ全文を表示するテキストを配置します。
	 * 
	 * @param		parent				親コンポジット
	 * @throws		Exception
	 */
	private void createRightPain(Composite parent) {
		// テキスト生成
		msgTxt = new Text(parent, SWT.MULTI | SWT.V_SCROLL | SWT.FLAT | SWT.READ_ONLY | SWT.WRAP);
		Display display = parent.getDisplay();
		msgTxt.setBackground(display.getSystemColor(SWT.COLOR_WHITE));
	}
	
	
	/**
	 * メッセージ一覧テーブルを生成します。
	 * 
	 * @param			parent				親コンポジットです。
	 */
	private void createMessageListTable(Composite parent) {

		// メッセージリストのテーブルを生成
		Table table= new Table(parent, SWT.SINGLE | SWT.H_SCROLL | SWT.V_SCROLL | SWT.FULL_SELECTION | SWT.BORDER);

		// レイアウト設定
		table.setLayoutData(new GridData(GridData.FILL_BOTH));
		TableLayout layout = new TableLayout();
		table.setLayout(layout);

		// 表示設定
		table.setHeaderVisible(true);
		table.setLinesVisible(true);
		
		
		// 「Delete」キーでメッセージ削除
		table.addKeyListener(new KeyAdapter(){

			/*
			 *  (非 Javadoc)
			 * @see org.eclipse.swt.events.KeyListener#keyPressed(org.eclipse.swt.events.KeyEvent)
			 */
			public void keyPressed(KeyEvent e) {
				
				// 「Delete」キー押下時
				if(e.character == SWT.DEL) {
					
					// 選択レコードを取得
					MessagePacket msg = getSelectionRecord();
					
					if(msg != null) {
						
						// 選択レコードを設定
						deleteMessageAction.setMessagePacket(msg);

						// メッセージ削除
						deleteMessageAction.run();	
						
					}
					
				}
				
			}
			
		});
		
		
		// メッセージ一覧テーブルからビューアインスタンスを取得
		viewer = new TableViewer(table);
		
		// リスナー取得
		SelectionListener headerListener = getColumnListener(viewer);
		
		// テーブルの1カラム目：アイコン
		TableColumn col1= new TableColumn(table, SWT.NONE, 0);
		col1.setText(""); 
		col1.setWidth(20);
		col1.setAlignment(SWT.LEFT);
		col1.addSelectionListener(headerListener);

		// テーブルの2カラム目：メッセージ
		TableColumn col2= new TableColumn(table, SWT.NONE, 1);
		col2.setText(Messages.getString("MessageListView.5"));  //$NON-NLS-1$
		col2.setWidth(300);
		col2.setAlignment(SWT.LEFT);
		col2.addSelectionListener(headerListener);
		
		// テーブルの3カラム目：送信者
		TableColumn col3= new TableColumn(table, SWT.NONE, 2);
		col3.setText(Messages.getString("MessageListView.6"));  //$NON-NLS-1$
		col3.setWidth(80);
		col3.setAlignment(SWT.LEFT);
		col3.addSelectionListener(headerListener);
		
		// テーブルの4カラム目：送信日時
		TableColumn col4= new TableColumn(table, SWT.NONE, 3);
		col4.setText(Messages.getString("MessageListView.7"));  //$NON-NLS-1$
		col4.setWidth(105);
		col4.setAlignment(SWT.LEFT);
		col4.addSelectionListener(headerListener);
		
		// ビューアにプロバイダを設定
		MessageListViewContentProvider provider = new MessageListViewContentProvider();
		viewer.setContentProvider(provider);
		viewer.setLabelProvider(new MessageListViewLabelProvider());
		
		// ソートのデフォルト設定（日付の降順）
		MessageListViewSorter sorter = new MessageListViewSorter(3);
		sorter.setReversed(true);
		viewer.setSorter(sorter);	
		
		// テーブルの初期データ（空）を設定
		viewer.setInput(new ArrayList());
		
		// カレントプロジェクトがある場合はそれに属するメッセージをテーブルに表示する
		CurrentProject project = CurrentProject.getInsance();
		if(project.isOpen()) {
			provider.refresh(null);
		}
		
	}

	
	/**
	 * このビューが持つメッセージテキストインスタンスを渡します。
	 * 
	 * @return		メッセージテキストインスタンス
	 */
	public Text getMsgTxt() {
		return msgTxt;
	}

	/*
	 *  (非 Javadoc)
	 * @see jp.valtech.bts.ui.IBtsViewPart#updateDescription(int)
	 */
	public void updateDescription(int filterListSize) {
		// 何もしない
		;
	}


	/**
	 * 当インスタンスで持つテーブルビューアインスタンスを返します。
	 * 
	 * @return			当インスタンスで持つテーブルビューアインスタンス
	 */
	public TableViewer getViewer() {
		return this.viewer;
	}
	
	
	/**
	 * ソータを設定するリスナーです。
	 * @param		tableViewer		テーブルビューア
	 * @return		セレクションアダプタ
	 */
	private SelectionListener getColumnListener(final TableViewer tableViewer){
		
		return new SelectionAdapter(){

			/*
			 *  (非 Javadoc)
			 * @see org.eclipse.swt.events.SelectionListener#widgetSelected(org.eclipse.swt.events.SelectionEvent)
			 */
			public void widgetSelected(SelectionEvent e) {
				
				// クリックしたカラム番号取得
				int column = tableViewer.getTable().indexOf((TableColumn)e.widget);
				
				// ソータの取得
				MessageListViewSorter oldSorter = (MessageListViewSorter) tableViewer.getSorter();
				
				// 連続して同じカラムをクリックしたときの処理
				if(oldSorter != null && column == oldSorter.getColumnNumber()){
					// ソート反転フラグの切換え
					oldSorter.setReversed(!oldSorter.isReversed());
					// テーブルの再表示
					tableViewer.refresh();
				} else {
					// ソータの設定
					tableViewer.setSorter(new MessageListViewSorter(column));
				}
			}
		};
	}
	
    /*
     *  (非 Javadoc)
     * @see org.eclipse.ui.IWorkbenchPart#dispose()
     */
    public void dispose() {
    	
    	// カレントプロジェクトに設定したアクションを削除する
		CurrentProject project = CurrentProject.getInsance();
		project.removeAction(openFilterDlgAction);
		project.removeAction(openMsgDlgAction);
    	
    	super.dispose();
    }
}

