/***********************************************************************
 * Copyright(C) 2006 Valtech Co.,Ltd.
 * All Rights Reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 ***********************************************************************/
package jp.valtech.bts.ui.action;

import java.util.Calendar;

import jp.valtech.bts.util.BTSUtility;

import org.eclipse.jface.action.Action;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CLabel;
import org.eclipse.swt.events.MouseAdapter;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.events.MouseTrackAdapter;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.graphics.FontData;
import org.eclipse.swt.graphics.RGB;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;

/**
 * カレンダーを開くアクションです。
 * 
 * @author		<A href="mailto:s_imai@valtech.jp">S.Imai</A>
 * @version	Ver.0.8
 */
public class OpenCalendarDlgAction extends Action {

	/** 選択した日付を表示するテキスト */
	private Text text;

	/** 現在のカレンダ */
	private Calendar today;

	/** 現在の年 */
	private int currentYear;

	/** 現在の月 */
	private int currentMonth;

	/** 1日の秒数（ミリ秒） */
	private int MSEC_ONEDAY = 1000 * 60 * 60 * 24;

	/**
	 * 初期設定
	 * 
	 * @param parent
	 */
	public OpenCalendarDlgAction(Text text) {
		super();
		this.text = text;
		// 現在の年月を設定
		moveToday();
	}

	/*
	 * (非 Javadoc)
	 * 
	 * @see org.eclipse.jface.action.Action#run()
	 */
	public void run() {
		CalendarDlg calendarDlg = new CalendarDlg(text.getShell());
		calendarDlg.open();
	}

	/**
	 * 現在の年月を取得します。
	 */
	private void moveToday() {
		today = Calendar.getInstance();
		currentYear = today.get(Calendar.YEAR);
		currentMonth = today.get(Calendar.MONTH) + 1;
	}

	/**
	 * カレンダーを表示するダイアログです。
	 * 
	 * @author <A href="mailto:s_imai@valtech.jp">$Author$</A>
	 * @version $Rev$
	 */
	public class CalendarDlg extends Dialog {

		/** 当ダイアログのコンポジット */
		private Composite composite;

		/** 親コンポジット */
		private Composite parentComposite;

		/** 当ダイアログで使用するフォント */
		private Font font = new Font(Display.getCurrent(), new FontData("", 10, SWT.NORMAL));

		/** 背景色を一時保存 */
		private Color tempBackColor;

		/** 前景色を一時保存 */
		private Color tempForeColor;

		/**
		 * 初期設定
		 * 
		 * @param parentShell シェル
		 */
		protected CalendarDlg(Shell parentShell) {
			super(parentShell);
		}

		/*
		 * (非 Javadoc)
		 * 
		 * @see org.eclipse.jface.dialogs.Dialog#createDialogArea(org.eclipse.swt.widgets.Composite)
		 */
		protected Control createDialogArea(Composite parent) {

			// 親コンポジット
			parentComposite = parent;

			// ダイアログ名の設定
			this.getShell().setText(Messages.getString("OpenCalendarDlgAction.0")); //$NON-NLS-1$
			this.getShell().setMinimumSize(175, 270);

			// コンポジット生成
			composite = (Composite) super.createDialogArea(parentComposite);

			// ダイアログの背景色の設定
			composite.setBackground(Display.getCurrent().getSystemColor(SWT.COLOR_WHITE));

			// グリッドレイアウト設定
			GridLayout layout = new GridLayout(7, true);
			layout.horizontalSpacing = 0;
			layout.verticalSpacing = 0;
			composite.setLayout(layout);

			// 年月ラベルの設定
			createCurrentMonth();

			// 月移動ラベルの設定
			createMoveMonth();

			// 曜日ラベルの設定
			createWeek();

			// 日付ラベルの設定
			createDay();

			// 閉じるラベル設定
			closeCalendar();

			return composite;

		}

		/*
		 * (非 Javadoc)
		 * 
		 * @see org.eclipse.jface.window.Window#createContents(org.eclipse.swt.widgets.Composite)
		 */
		protected Control createContents(Composite parent) {
			// create the top level composite for the dialog
			Composite composite = new Composite(parent, 0);
			GridLayout layout = new GridLayout();
			layout.marginHeight = 0;
			layout.marginWidth = 0;
			layout.verticalSpacing = 0;
			composite.setLayout(layout);
			composite.setLayoutData(new GridData(GridData.FILL_BOTH));
			applyDialogFont(composite);
			// initialize the dialog units
			initializeDialogUnits(composite);
			// create the dialog area and button bar
			dialogArea = createDialogArea(composite);
			
			// ButtonBarエリアの削除
			// buttonBar = createButtonBar(composite);
			
			return composite;
		}

		/**
		 * 現在の年月を表示します。
		 */
		private void createCurrentMonth() {

			// 現在の年月ラベル生成
			CLabel monthLabel = new CLabel(composite, SWT.CENTER);

			// 現在の年月ラベルのフォント設定
			monthLabel.setFont(new Font(Display.getCurrent(), new FontData("", 10, SWT.BOLD)));

			// 現在の年月ラベルの背景色設定
			monthLabel.setBackground(new Color(Display.getCurrent(), new RGB(140, 140, 255)));

			// 現在の年月ラベルの前景色設定
			monthLabel.setForeground(new Color(Display.getCurrent(), new RGB(255, 255, 255)));

			// 現在の年月ラベルのテキスト設定
			monthLabel.setText(currentYear + Messages.getString("OpenCalendarDlgAction.1") + currentMonth + Messages.getString("OpenCalendarDlgAction.2")); //$NON-NLS-1$ //$NON-NLS-2$

			// 現在の年月ラベルのレイアウト設定
			GridData data = new GridData(GridData.HORIZONTAL_ALIGN_FILL | GridData.VERTICAL_ALIGN_FILL);
			data.horizontalSpan = 7;
			monthLabel.setLayoutData(data);

		}

		/**
		 * 月遷移ラベルを設定します。
		 */
		private void createMoveMonth() {

			// 前月ラベル生成
			final CLabel lastMonthLabel = new CLabel(composite, SWT.CENTER | SWT.BORDER);

			// 前月ラベルのフォント設定
			lastMonthLabel.setFont(font);

			// 前月ラベルの背景色設定
			lastMonthLabel.setBackground(new Color(Display.getCurrent(), new RGB(245, 245, 255)));

			// 前月ラベルの前景色設定
			lastMonthLabel.setForeground(new Color(Display.getCurrent(), new RGB(0, 0, 255)));

			// 前月ラベルのテキスト設定
			lastMonthLabel.setText(Messages.getString("OpenCalendarDlgAction.3")); //$NON-NLS-1$

			// 前月ラベルのレイアウト設定
			GridData lastMonthData = new GridData(GridData.HORIZONTAL_ALIGN_FILL | GridData.VERTICAL_ALIGN_FILL);
			lastMonthData.horizontalSpan = 2;
			lastMonthLabel.setLayoutData(lastMonthData);

			// 前月ラベルにMouseTrackListenerを追加
			lastMonthLabel.addMouseTrackListener(new LabelMouseTrackAdapter(lastMonthLabel));

			// 前月ラベルにMouseListenerを追加
			lastMonthLabel.addMouseListener(new MouseAdapter() {

				/*
				 * (非 Javadoc)
				 * 
				 * @see org.eclipse.swt.events.MouseListener#mouseDown(org.eclipse.swt.events.MouseEvent)
				 */
				public void mouseDown(MouseEvent e) {
					// 前月ラベルをクリックしたとき
					moveMonth(-1);
				}

			});

			// 今月ラベル生成
			final CLabel nowMonthLabel = new CLabel(composite, SWT.CENTER | SWT.BORDER);

			// 今月ラベルのフォント設定
			nowMonthLabel.setFont(font);

			// 今月ラベルの背景色設定
			nowMonthLabel.setBackground(new Color(Display.getCurrent(), new RGB(245, 245, 255)));

			// 今月ラベルの前景色設定
			nowMonthLabel.setForeground(new Color(Display.getCurrent(), new RGB(0, 0, 255)));

			// 今月ラベルのテキスト設定
			nowMonthLabel.setText(Messages.getString("OpenCalendarDlgAction.4")); //$NON-NLS-1$

			// 今月ラベルのレイアウト設定
			GridData nowMonthData = new GridData(GridData.HORIZONTAL_ALIGN_FILL | GridData.VERTICAL_ALIGN_FILL);
			nowMonthData.horizontalSpan = 3;
			nowMonthLabel.setLayoutData(nowMonthData);

			// 今月ラベルにMouseTrackListenerを追加
			nowMonthLabel.addMouseTrackListener(new LabelMouseTrackAdapter(nowMonthLabel));

			// 今月ラベルにMouseListenerを追加
			nowMonthLabel.addMouseListener(new MouseAdapter() {

				/*
				 * (非 Javadoc)
				 * 
				 * @see org.eclipse.swt.events.MouseListener#mouseDown(org.eclipse.swt.events.MouseEvent)
				 */
				public void mouseDown(MouseEvent e) {
					// 今月ラベルをクリックしたとき
					moveToday();
					moveMonth(0);
				}

			});

			// 翌月ラベル生成
			final CLabel nextMonthLabel = new CLabel(composite, SWT.CENTER | SWT.BORDER);

			// 翌月ラベルのフォント設定
			nextMonthLabel.setFont(font);

			// 翌月ラベルの背景色設定
			nextMonthLabel.setBackground(new Color(Display.getCurrent(), new RGB(245, 245, 255)));

			// 翌月ラベルの前景色設定
			nextMonthLabel.setForeground(new Color(Display.getCurrent(), new RGB(0, 0, 255)));

			// 翌月ラベルのテキスト設定
			nextMonthLabel.setText(Messages.getString("OpenCalendarDlgAction.5")); //$NON-NLS-1$

			// 翌月ラベルのレイアウト設定（前月と同じ）
			nextMonthLabel.setLayoutData(lastMonthData);

			// 翌月ラベルにMouseTrackListenerを追加
			nextMonthLabel.addMouseTrackListener(new LabelMouseTrackAdapter(nextMonthLabel));

			// 翌月ラベルにMouseListenerを追加
			nextMonthLabel.addMouseListener(new MouseAdapter() {

				/*
				 * (非 Javadoc)
				 * 
				 * @see org.eclipse.swt.events.MouseListener#mouseDown(org.eclipse.swt.events.MouseEvent)
				 */
				public void mouseDown(MouseEvent e) {
					// 翌月ラベルをクリックしたとき
					moveMonth(1);
				}

			});

		}

		/**
		 * 曜日ラベルを表示します。
		 */
		private void createWeek() {

			// 日曜ラベルの設定
			CLabel sundayLabel = new CLabel(composite, SWT.CENTER | SWT.BORDER);
			sundayLabel.setFont(font);
			sundayLabel.setBackground(new Color(Display.getCurrent(), new RGB(225, 225, 225)));
			sundayLabel.setForeground(new Color(Display.getCurrent(), new RGB(255, 0, 0)));
			sundayLabel.setText(Messages.getString("OpenCalendarDlgAction.6")); //$NON-NLS-1$
			sundayLabel.setLayoutData(new GridData(GridData.HORIZONTAL_ALIGN_FILL | GridData.VERTICAL_ALIGN_FILL));

			// 月曜ラベルの設定
			CLabel mondayLabel = new CLabel(composite, SWT.CENTER | SWT.BORDER);
			mondayLabel.setFont(font);
			mondayLabel.setBackground(new Color(Display.getCurrent(), new RGB(225, 225, 225)));
			mondayLabel.setForeground(new Color(Display.getCurrent(), new RGB(0, 0, 0)));
			mondayLabel.setText(Messages.getString("OpenCalendarDlgAction.7")); //$NON-NLS-1$
			mondayLabel.setLayoutData(new GridData(GridData.HORIZONTAL_ALIGN_FILL | GridData.VERTICAL_ALIGN_FILL));

			// 火曜ラベルの設定
			CLabel tuesdayLabel = new CLabel(composite, SWT.CENTER | SWT.BORDER);
			tuesdayLabel.setFont(font);
			tuesdayLabel.setBackground(new Color(Display.getCurrent(), new RGB(225, 225, 225)));
			tuesdayLabel.setForeground(new Color(Display.getCurrent(), new RGB(0, 0, 0)));
			tuesdayLabel.setText(Messages.getString("OpenCalendarDlgAction.8")); //$NON-NLS-1$
			tuesdayLabel.setLayoutData(new GridData(GridData.HORIZONTAL_ALIGN_FILL | GridData.VERTICAL_ALIGN_FILL));

			// 水曜ラベルの設定
			CLabel wednesdayLabel = new CLabel(composite, SWT.CENTER | SWT.BORDER);
			wednesdayLabel.setFont(font);
			wednesdayLabel.setBackground(new Color(Display.getCurrent(), new RGB(225, 225, 225)));
			wednesdayLabel.setForeground(new Color(Display.getCurrent(), new RGB(0, 0, 0)));
			wednesdayLabel.setText(Messages.getString("OpenCalendarDlgAction.9")); //$NON-NLS-1$
			wednesdayLabel.setLayoutData(new GridData(GridData.HORIZONTAL_ALIGN_FILL | GridData.VERTICAL_ALIGN_FILL));

			// 木曜ラベルの設定
			CLabel thursdayLabel = new CLabel(composite, SWT.CENTER | SWT.BORDER);
			thursdayLabel.setFont(font);
			thursdayLabel.setBackground(new Color(Display.getCurrent(), new RGB(225, 225, 225)));
			thursdayLabel.setForeground(new Color(Display.getCurrent(), new RGB(0, 0, 0)));
			thursdayLabel.setText(Messages.getString("OpenCalendarDlgAction.10")); //$NON-NLS-1$
			thursdayLabel.setLayoutData(new GridData(GridData.HORIZONTAL_ALIGN_FILL | GridData.VERTICAL_ALIGN_FILL));

			// 金曜ラベルの設定
			CLabel fridayLabel = new CLabel(composite, SWT.CENTER | SWT.BORDER);
			fridayLabel.setFont(font);
			fridayLabel.setBackground(new Color(Display.getCurrent(), new RGB(225, 225, 225)));
			fridayLabel.setForeground(new Color(Display.getCurrent(), new RGB(0, 0, 0)));
			fridayLabel.setText(Messages.getString("OpenCalendarDlgAction.11")); //$NON-NLS-1$
			fridayLabel.setLayoutData(new GridData(GridData.HORIZONTAL_ALIGN_FILL | GridData.VERTICAL_ALIGN_FILL));

			// 土曜ラベルの設定
			CLabel saturdayLabel = new CLabel(composite, SWT.CENTER | SWT.BORDER);
			saturdayLabel.setFont(font);
			saturdayLabel.setBackground(new Color(Display.getCurrent(), new RGB(225, 225, 225)));
			saturdayLabel.setForeground(new Color(Display.getCurrent(), new RGB(0, 0, 255)));
			saturdayLabel.setText(Messages.getString("OpenCalendarDlgAction.12")); //$NON-NLS-1$
			saturdayLabel.setLayoutData(new GridData(GridData.HORIZONTAL_ALIGN_FILL | GridData.VERTICAL_ALIGN_FILL));

		}

		/**
		 * 日にちラベルの設定を行います。
		 */
		private void createDay() {

			// 表示開始日付を取得（日曜から表示する）
			final Calendar startDate = Calendar.getInstance();
			startDate.set(currentYear, currentMonth - 1, 1);
			startDate.setTimeInMillis(startDate.getTimeInMillis() - startDate.get(Calendar.DAY_OF_WEEK) * MSEC_ONEDAY + MSEC_ONEDAY);

			// 表示終了日付を取得（土曜まで表示する）
			Calendar endDate = Calendar.getInstance();
			endDate.set(currentYear, currentMonth, 1);
			endDate.setTimeInMillis(endDate.getTimeInMillis() - MSEC_ONEDAY);
			endDate.setTimeInMillis(endDate.getTimeInMillis() + (6 - endDate.get(Calendar.DAY_OF_WEEK)) * MSEC_ONEDAY + MSEC_ONEDAY);

			// 表示開始日付～表示終了日付のラベル設定
			for (; startDate.getTimeInMillis() <= endDate.getTimeInMillis();
			      startDate.setTimeInMillis(startDate.getTimeInMillis()+ MSEC_ONEDAY)) {

				// 日付ラベル生成
				final CLabel dayLabel = new CLabel(composite, SWT.CENTER | SWT.BORDER);

				// 日にちラベルのフォント設定
				dayLabel.setFont(font);

				// 日付ラベルのテキスト設定
				dayLabel.setText(" " + Integer.toString(startDate.get(Calendar.DAY_OF_MONTH)) + " ");

				// 日付ラベルのレイアウト設定
				GridData data = new GridData(GridData.HORIZONTAL_ALIGN_FILL | GridData.VERTICAL_ALIGN_FILL);
				data.heightHint = 25;
				dayLabel.setLayoutData(data);

				// 日付ラベルにMouseListenerを追加
				dayLabel.addMouseListener(new DayMouseAdapter(startDate.getTimeInMillis()));

				// 今日の日付とラベルの日付が同じ場合
				if (Calendar.getInstance().get(Calendar.YEAR) == startDate.get(Calendar.YEAR)
						&& Calendar.getInstance().get(Calendar.MONTH) == startDate.get(Calendar.MONTH)
						&& Calendar.getInstance().get(Calendar.DAY_OF_MONTH) == startDate.get(Calendar.DAY_OF_MONTH)) {

					// 今日の日付ラベルの背景色設定
					dayLabel.setBackground(new Color(Display.getCurrent(), new RGB(255, 255, 221)));

					// 今日の日付ラベルの前景色設定
					dayLabel.setForeground(new Color(Display.getCurrent(), new RGB(179, 165, 90)));

					// 今日の日付ラベルにMouseTrackListenerを追加
					dayLabel.addMouseTrackListener(new LabelMouseTrackAdapter(dayLabel));

				}

				// 日曜日のラベル設定
				else if (startDate.get(Calendar.DAY_OF_WEEK) == 1) {

					// 日曜日ラベルの背景色設定
					dayLabel.setBackground(new Color(Display.getCurrent(), new RGB(255, 235, 235)));

					// 日曜日ラベルの前景色設定
					if (currentMonth == startDate.get(Calendar.MONTH) + 1) {
						dayLabel.setForeground(new Color(Display.getCurrent(), new RGB(255, 0, 0)));
					} else {
						dayLabel.setForeground(new Color(Display.getCurrent(), new RGB(255, 180, 180)));
					}

					// 日曜日ラベルにMouseTrackListenerを追加
					dayLabel.addMouseTrackListener(new LabelMouseTrackAdapter(dayLabel));

				}

				// 土曜日のラベル設定
				else if (startDate.get(Calendar.DAY_OF_WEEK) == 7) {

					// 土曜日ラベルの背景色設定
					dayLabel.setBackground(new Color(Display.getCurrent(), new RGB(235, 235, 255)));

					// 土曜日ラベルの前景色設定
					if (currentMonth == startDate.get(Calendar.MONTH) + 1) {
						dayLabel.setForeground(new Color(Display.getCurrent(), new RGB(0, 0, 255)));
					} else {
						dayLabel.setForeground(new Color(Display.getCurrent(), new RGB(180, 180, 255)));
					}

					// 土曜日ラベルにMouseTrackListenerを追加
					dayLabel.addMouseTrackListener(new LabelMouseTrackAdapter(dayLabel));

				}

				// 月～金曜日のラベル設定
				else {

					// 月～金曜日ラベルの背景色設定
					dayLabel.setBackground(new Color(Display.getCurrent(), new RGB(255, 255, 255)));

					// 月～金曜日ラベルの前景色設定
					if (currentMonth == startDate.get(Calendar.MONTH) + 1) {
						dayLabel.setForeground(new Color(Display.getCurrent(), new RGB(0, 0, 0)));
					} else {
						dayLabel.setForeground(new Color(Display.getCurrent(), new RGB(180, 180, 180)));
					}

					// 月～金曜日ラベルにMouseTrackListenerを追加
					dayLabel.addMouseTrackListener(new LabelMouseTrackAdapter(dayLabel));

				}

			}

		}

		/**
		 * ラベルにマウスを合わせたときと外したときのリスナークラスです。
		 * 
		 * @author <A href="mailto:s_imai@valtech.jp">$Author$</A>
		 * @version $Rev$
		 */
		private class LabelMouseTrackAdapter extends MouseTrackAdapter {

			/** リスナーを追加するラベル */
			private CLabel label;

			/**
			 * 初期設定
			 * 
			 * @param label リスナーに追加するラベル
			 */
			private LabelMouseTrackAdapter(CLabel label) {
				this.label = label;
			}

			/*
			 * (非 Javadoc)
			 * 
			 * @see org.eclipse.swt.events.MouseTrackListener#mouseEnter(org.eclipse.swt.events.MouseEvent)
			 */
			public void mouseEnter(MouseEvent e) {
				// 背景色と前景色を一旦保存し、変更
				changeGround(label);
			}

			/*
			 * (非 Javadoc)
			 * 
			 * @see org.eclipse.swt.events.MouseTrackListener#mouseExit(org.eclipse.swt.events.MouseEvent)
			 */
			public void mouseExit(MouseEvent e) {
				// 背景色と前景色を元に戻す
				returnGround(label);
			}

		}

		/**
		 * ラベルにマウスを合わせたときに背景色と前景色を変えます。
		 * 
		 * @param label 色を変えるラベル
		 */
		private void changeGround(CLabel label) {

			// 変更前の背景色と前景色を保存
			tempBackColor = label.getBackground();
			tempForeColor = label.getForeground();

			// 背景色と前景色を変更
			label.setBackground(new Color(Display.getCurrent(), new RGB(235, 255, 235)));
			label.setForeground(new Color(Display.getCurrent(), new RGB(0, 255, 0)));

		}

		/**
		 * ラベルからマウスを外したときに背景色と前景色を元に戻します。
		 * 
		 * @param label 色を戻すラベル
		 */
		private void returnGround(CLabel label) {

			// 背景色と前景色を元に戻す
			label.setBackground(tempBackColor);
			label.setForeground(tempForeColor);

		}

		/**
		 * 日付ラベルをクリックしたときのリスナークラスです。 選択された日付をテキストフィールドに表示します。
		 * 
		 * @author <A href="mailto:s_imai@valtech.jp">$Author$</A>
		 * @version $Rev$
		 */
		private class DayMouseAdapter extends MouseAdapter {

			/** フォーマット済みの選択された日付 */
			private String selectDate;

			/**
			 * 選択された日付をフォーマットします。
			 * 
			 * @param date 選択された日付
			 */
			public DayMouseAdapter(long date) {
				selectDate = BTSUtility.formatDate(date, "yyyy/MM/dd");
			}

			/*
			 * (非 Javadoc)
			 * 
			 * @see org.eclipse.swt.events.MouseListener#mouseDown(org.eclipse.swt.events.MouseEvent)
			 */
			public void mouseDown(MouseEvent e) {
				// テキストフィールドに日付を表示
				text.setText(selectDate);
				// ダイアログを閉じる
				close();
			}

		}

		/**
		 * 閉じるラベルの設定を行います。
		 */
		private void closeCalendar() {

			// 閉じるラベルの生成
			final CLabel closeLabel = new CLabel(composite, SWT.CENTER | SWT.BORDER);

			// 閉じるラベルのフォント設定
			closeLabel.setFont(font);

			// 閉じるラベルの背景色設定
			closeLabel.setBackground(new Color(Display.getCurrent(), new RGB(245, 245, 255)));

			// 閉じるラベルの前景色設定
			closeLabel.setForeground(new Color(Display.getCurrent(), new RGB(0, 0, 255)));

			// 閉じるラベルのテキスト設定
			closeLabel.setText(Messages.getString("OpenCalendarDlgAction.13")); //$NON-NLS-1$

			// 閉じるラベルのレイアウト設定
			GridData data = new GridData(GridData.HORIZONTAL_ALIGN_FILL | GridData.VERTICAL_ALIGN_FILL);
			data.horizontalSpan = 7;
			closeLabel.setLayoutData(data);

			// 閉じるラベルにMouseTrackListenerを追加
			closeLabel.addMouseTrackListener(new LabelMouseTrackAdapter(closeLabel));

			// 閉じるラベルにMouseListenerを追加
			closeLabel.addMouseListener(new MouseAdapter() {

				/*
				 * (非 Javadoc)
				 * 
				 * @see org.eclipse.swt.events.MouseListener#mouseDown(org.eclipse.swt.events.MouseEvent)
				 */
				public void mouseDown(MouseEvent e) {
					// 閉じるラベルをクリックしたとき
					close();
				}

			});

		}

		/**
		 * 月遷移ラベルをクリックしたときの処理です。
		 * 
		 * @param adjust
		 */
		private void moveMonth(int adjust) {

			// カレンダー表示月を更新
			currentMonth += adjust;

			// カレンダー表示年を更新
			if (currentMonth > 12) {
				currentYear++;
				currentMonth = 1;
			} else if (currentMonth < 1) {
				currentYear--;
				currentMonth = 12;
			}

			// カレンダーコンポジットを一旦消去
			Control[] children = parentComposite.getChildren();
			for (int i = 0; i < children.length; i++) {
				children[i].dispose();
				children[i] = null;
			}
			
			// カレンダーコンポジットを生成
			createDialogArea(parentComposite);
			reflow();

		}
		
		/**
		 * カレンダーコンポジット以下を再描画する。
		 *
		 */
		private void reflow() {
			Composite c = parentComposite;
			while (c != null) {
				c.setRedraw(false);
				c = c.getParent();
			}
			c = parentComposite;
			while (c != null) {
				c.layout(true);
				c = c.getParent();
			}
			c = parentComposite;
			while (c != null) {
				c.setRedraw(true);
				c = c.getParent();
			}
		}

	}

}