/***********************************************************************
 * Copyright(C) 2006 Valtech Co.,Ltd.
 * All Rights Reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 ***********************************************************************/
package jp.valtech.bts.network.server;

import java.io.IOException;

import jp.valtech.bts.data.NetworkConfig;
import jp.valtech.bts.ui.BtsException;
import jp.valtech.bts.ui.BtsPlugin;
import jp.valtech.bts.util.Logging;

/**
 * サーバプロセス。
 * 当システムにてデータの送受信を行うための受信側プロセスです。
 * {@link MulticastServer マルチキャスト通信サーバプロセス}と{@link TCPServer TCP通信サーバプロセス}
 * を一括管理します。<br>
 * 	{@link #startServer(String)}でサーバプロセスの開始処理を行います。<br>
 * 	{@link #stopServer(String)}でサーバプロセスの処理処理を行います。<br>
 * 
 * @author		<A href="mailto:m_sugitou@valtech.jp">M.Sugito</A>
 * @version	Ver.0.8
 */
public class Server implements Logging {

	/** サーバプロセスインスタンス。唯一のインスタンスです。 */
	private static Server server;
	
	/** マルチキャスト通信サーバプロセス */
	private MulticastServer multicastServer = null;
	
	/** TCP通信サーバプロセス */
	private TCPServer tcpServer = null;
	
	/**
	 * 外部からは参照されません。
	 */
	private Server() {
		super();
	}
	

	/**
	 * サーバプロセスを取得します。
	 * 
	 * @return			サーバプロセス
	 */
	public static synchronized Server getServer() {
		if(server==null) {
			server = new Server();
		}
		return server;
	}
	
	
	
	/**
	 * サーバプロセス開始。
	 * <ul>
	 *   <li>{@link MulticastServer}を起動して、マルチキャスト通信受信待ち受け状態になります。
	 *   <li>{@link TCPServer}を起動して、TCP通信受信待ち受け状態になります。
	 * </ul>
	 */
	public void startServer() throws BtsException {

		// 既にサーバプロセスが起動されている場合は終了処理を行う。
		if( multicastServer != null ) {
			try {
				multicastServer.stop();
			} catch (Exception e) {
				;
			}
		}
		if( tcpServer != null) {
			try {
				tcpServer.stop();
			} catch (Exception e) {
				;
			}
		}
		
		
		// 設定ファイルを取得する
		NetworkConfig config = new NetworkConfig();

		// マルチキャスト通信サーバインスタンス起動
		try {
			this.multicastServer = new MulticastServer( config );
		} catch (IOException e) {
			String msg 
				= Messages.getString("Server.0") //$NON-NLS-1$
				+  Messages.getString("Server.1") + config.getMulticastPort() //$NON-NLS-1$
				+ Messages.getString("Server.2"); //$NON-NLS-1$
			logger.fatal(msg, e);
			BtsPlugin.getInstance().error(msg, e);
			
			// 例外をキャッチしたら終了
			stopServer();
			throw new BtsException(msg, e);
		}

		// TCP通信サーバインスタンス起動
		try {
			this.tcpServer = new TCPServer( config ) ;
		} catch (IOException e) {
			String msg 
				= Messages.getString("Server.3") //$NON-NLS-1$
				+  Messages.getString("Server.4") + config.getTCPPort() //$NON-NLS-1$
				+ Messages.getString("Server.5"); //$NON-NLS-1$
			// 例外をキャッチしたら終了
			stopServer();
			throw new BtsException(msg, e);
		}

		// Multicast通信待ちうけ処理をスレッドで開始する
		Thread multicastThread	= new Thread( multicastServer, "MulticastServer" );
		multicastThread.start();

		// TCP通信待ちうけ処理をスレッドを開始する
		Thread tcpThread	= new Thread( tcpServer, "TCPServer" );
		tcpThread.start();

		logger.info( Messages.getString("Server.6") ); //$NON-NLS-1$
	}
	

	/**
	 * サーバプロセス終了。
	 * <ul>
	 *   <li>{@link MulticastServer}を停止します。
	 *   <li>{@link TCPServer}を停止します。
	 * </ul>
	 */
	public void stopServer() {
		// マルチキャストサーバ停止
		try {
			if(multicastServer != null) {
				multicastServer.stop();
				multicastServer = null;
			}
		}catch (Exception e) {
			logger.fatal(e);
		}

		// TCPサーバ停止
		try {
			if(tcpServer != null) {
				tcpServer.stop();
				tcpServer = null;
			}
		}catch (Exception e) {
			logger.fatal(e);
		}
	}
}