/***********************************************************************
 * Copyright(C) 2006 Valtech Co.,Ltd.
 * All Rights Reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 ***********************************************************************/
package jp.valtech.bts.facade;

import java.util.ArrayList;
import java.util.List;

import jp.valtech.bts.connection.IssueDBConnection;
import jp.valtech.bts.dao.AttachmentDAO;
import jp.valtech.bts.dao.BtsDBException;
import jp.valtech.bts.dao.CommentHistoryDAO;
import jp.valtech.bts.dao.IssueDAO;
import jp.valtech.bts.dao.IssueHistoryDAO;
import jp.valtech.bts.data.Attachment;
import jp.valtech.bts.data.CommentHistory;
import jp.valtech.bts.data.Issue;
import jp.valtech.bts.data.IssueHistory;
import jp.valtech.bts.data.IssueType;
import jp.valtech.bts.ui.BtsPlugin;
import jp.valtech.bts.util.Logging;

/**
 * {@link jp.valtech.bts.dao.IssueDAO}用のFacadeクラスです。
 * 
 * @author		<A href="mailto:m_sugitou@valtech.jp">M.Sugito</A>
 * @version	Ver.0.8
 */
public class ModifyIssueFacade implements Logging {

	/**
	 * 何もしない。
	 */
	public ModifyIssueFacade() {
		;
	}

	/** 
	 * 新規追加された課題票をローカルDBに登録します。
	 * <pre>
	 * 　CASE:新規登録
	 * 　　　・登録時に追加れた添付ファイル情報の添付を、ファイルシステムとDBの両方に追加
	 * <pre>
	 * @param		issue		新規追加された課題票
	 */
	public void addNewDraftIssue(Issue issue)  throws Exception {

		//	DBコネクション取得
		IssueDBConnection dbcon = new IssueDBConnection();
		try {
			// 課題票を登録
			IssueDAO dao = new IssueDAO( dbcon );
			dao.addIssue( issue );

			// 新規追加添付ファイル
			AttachmentFacade.add(issue, dbcon);
			
			dbcon.commit();

		} catch (Exception e) {
			dbcon.rollback();

			// エラー情報の出力
			String msg = Messages.getString("ModifyIssueFacade.0") + issue.getIssueID(); //$NON-NLS-1$
			logger.fatal(msg, e);
			BtsPlugin.getInstance().error(msg, e);
			throw e;
			
		} finally {
			dbcon.close();
			dbcon = null;
		}
	}
	
	
	/** 
	 * RELEASE課題票を元にDRAFT課題票を生成します。
	 * <pre>
	 * 　CASE:RELEASE→DRAFT(DRAFTがまだ無い) 
	 * 　　　・更新時にそのまま残った添付ファイル情報は、クローンを作成してファイルシステムとDBの両方に登録
	 * 　　　・更新時に追加れた添付ファイル情報の添付を、ファイルシステムとDBの両方に追加
	 * <pre>
	 * @param		issue		新規追加された課題票
	 */
	public void addCloneIssue(Issue issue)  throws Exception {

		//	DBコネクション取得
		IssueDBConnection dbcon = new IssueDBConnection();
		try {
			// 課題票を登録
			IssueDAO dao = new IssueDAO( dbcon );
			dao.addIssue( issue );

			// RELEASE課題票で持っていたものはクローンを作成してDRAFT課題票に登録
			AttachmentFacade.clone(issue, dbcon);
			
			// 新規追加添付ファイル
			AttachmentFacade.add(issue, dbcon);
			
			dbcon.commit();

		} catch (Exception e) {
			dbcon.rollback();

			// エラー情報の出力
			String msg = Messages.getString("ModifyIssueFacade.1") + issue.getIssueID(); //$NON-NLS-1$
			logger.fatal(msg, e);
			BtsPlugin.getInstance().error(msg, e);
			throw e;
			
		} finally {
			dbcon.close();
			dbcon = null;
		}
	}

	
	/** 
	 * 指定の{@link Issue 課題票情報}をローカルＤＢに上書き保存します。
	 * <PRE>
	 * 　CASE:DRAFT → DRAFT 
	 * 　　　・更新時に削除された添付ファイル情報の添付を、ファイルシステムとDBの両方から削除
	 * 　　　・更新時に追加れた添付ファイル情報の添付を、ファイルシステムとDBの両方に追加
	 * </PRE>
	 * 
	 * @param		issue					課題票
	 * @param		deleteAttachmentList	課題票更新時に削除された課題票情報のリスト
	 */
	public void updateIssue(Issue issue, Attachment[] deleteList) throws Exception {

		IssueDBConnection dbcon = new IssueDBConnection();
		try {
			// 課題票を更新
			IssueDAO dao = new IssueDAO( dbcon );
			dao.modifyIssue( issue , IssueType.DRAFT_VALUE );
			
			//### CASE1:DRAFT → DRAFT ###

			// 削除された添付ファイル情報を削除
			AttachmentFacade.delete(deleteList, dbcon);

			// 新規追加添付ファイル
			AttachmentFacade.add(issue, dbcon);
				
			
			dbcon.commit();
		
		} catch (Exception e) {
			dbcon.rollback();

			// エラー情報の出力
			String msg = Messages.getString("ModifyIssueFacade.2") + issue.getIssueID(); //$NON-NLS-1$
			logger.fatal(msg, e);
			BtsPlugin.getInstance().error(msg, e);
			throw e;
		} finally {
			dbcon.close();
			dbcon = null;
		}
	}

	/** 
	 * 指定の{@link Issue 課題票情報}をローカルＤＢに上書き保存します。
	 * <PRE>
	 * 　CASE:RELEASE → DRAFT(DRAFTがすでにある)
	 * 　　　・DRAFTの当該課題票に属する全ての添付ファイルを削除
	 * 　　　・更新時にそのまま残った添付ファイル情報は、クローンを作成してファイルシステムとDBの両方に登録
	 * 　　　・更新時に追加れた添付ファイル情報の添付を、ファイルシステムとDBの両方に追加
	 * </PRE>
	 * @param		issue					課題票
	 * @param		deleteAttachmentList	課題票更新時に削除された課題票情報のリスト
	 * @param		typeUpdateBefore		課題票を更新する前の{@link jp.valtech.bts.data.IssueType 課題票種別}
	 */
	public void replaceIssue(Issue issue, Attachment[] deleteList) throws Exception {

		IssueDBConnection dbcon = new IssueDBConnection();
		try {
			// 課題票を更新
			IssueDAO dao = new IssueDAO( dbcon );
			dao.modifyIssue( issue , IssueType.DRAFT_VALUE );
			
			//### CASE2:RELEASE → DRAFT(DRAFTがすでにある) ###
			
			// DRAFT課題票に属する添付を一旦全消去
			AttachmentFacade.deleteAll(issue, IssueType.DRAFT_VALUE, dbcon);
			
			// RELEASE課題票で持っていたものはクローンを作成してDRAFT課題票に登録
			AttachmentFacade.clone(issue, dbcon);
			
			// 新規追加添付ファイル
			AttachmentFacade.add(issue, dbcon);

			
			dbcon.commit();
		
		} catch (Exception e) {
			dbcon.rollback();

			// エラー情報の出力
			String msg = Messages.getString("ModifyIssueFacade.3") + issue.getIssueID(); //$NON-NLS-1$
			logger.fatal(msg, e);
			BtsPlugin.getInstance().error(msg, e);
			throw e;
		} finally {
			dbcon.close();
			dbcon = null;
		}
	}

	/**
	 * 指定のFingerPrintを持つ課題票を削除します。
	 * 
	 * @param		fingerPrint		課題票のFingerPrint
	 * @param		type			課題票種別（'D'か'C'のみ有効）
	 */
	public void deleteByFingerPrint(String fingerPrint, String type) {
		
		if(IssueType.RELEASE_VALUE.equals(type)) {
			throw new IllegalArgumentException(Messages.getString("ModifyIssueFacade.4")); //$NON-NLS-1$
		}

		IssueDBConnection dbcon = new IssueDBConnection();
		try {
			// 課題票を削除
			IssueDAO issueDAO = new IssueDAO( dbcon );
			issueDAO.deleteByFingerPrint(fingerPrint, type);
			
			// 添付を削除
			AttachmentFacade.deleteAll(fingerPrint, type, dbcon);

			// 課題票履歴を削除
			IssueHistoryDAO historyDAO = new IssueHistoryDAO(dbcon);
			historyDAO.deleteByFingerPrint(fingerPrint, type);
			

			// 課題票コメント履歴を削除
			CommentHistoryDAO commentHistoryDAO = new CommentHistoryDAO(dbcon);
			commentHistoryDAO.deleteByFingerPrint(fingerPrint, type);
			
			
			dbcon.commit();
		
		} catch (Exception e) {
			dbcon.rollback();

			// エラー情報の出力
			String msg = Messages.getString("ModifyIssueFacade.5") + fingerPrint; //$NON-NLS-1$
			logger.fatal(msg, e);
			BtsPlugin.getInstance().error(msg, e);
		} finally {
			dbcon.close();
			dbcon = null;
		}
	}

	
	/**
	 * 指定のfingerPrintを持つ{@link Issue 課題票}を取得します。
	 * 
	 * @param			fingerPrint			指定のfingerPrint
	 * @param			type				課題票の種類（{@link jp.valtech.bts.data.IssueType}）
	 * @return			指定のfingerPrintに該当する課題票
	 */
	public Issue getByFingerPrint(String fingerPrint, String type) {
		
		IssueDBConnection dbcon = new IssueDBConnection();
		try {
			// 課題票情報を取得
			IssueDAO issueDAO = new IssueDAO( dbcon );
			Issue issue = issueDAO.getByFingerPrint( fingerPrint , type);
			if(issue==null) {
				return null;
			}
			
			//### 添付ファイル情報格納 ###

			// 課題票の添付ファイルを取得
			AttachmentDAO attachmentDAO = new AttachmentDAO( dbcon);
			ArrayList attachmentList = attachmentDAO.getByFingerPrint(fingerPrint, type);

			// 取得できなかった場合は空のArrayListを設定する
			if(attachmentList == null) {
				attachmentList = new ArrayList();
			}

			// 課題票に添付ファイルを格納
			issue.setAttachmentList(attachmentList);
			

			//### コメント履歴格納 ###
			
			// 以降、課題票種別がDRAFTの場合にはRELEASE版のコメントを参照するようにする。
			if(type.equals(IssueType.DRAFT_VALUE)) {
				type = IssueType.RELEASE_VALUE;
			}
			// 課題票のコメント取得
			CommentHistoryDAO commentHistoryDAO = new CommentHistoryDAO(dbcon);
			CommentHistory[] commentHistories = commentHistoryDAO.getByFingerPrint(fingerPrint, type);
			// 課題票にコメント情報を格納
			issue.setCommentHistories(commentHistories);

			//### 課題票履歴履歴格納 ###
			
			// 課題票の履歴取得
			IssueHistoryDAO issueHistoryDAO = new IssueHistoryDAO( dbcon );
			IssueHistory[] issueHistories =  issueHistoryDAO.getByFingerPrint( fingerPrint, type );
			// 課題票に履歴情報を格納
			issue.setIssueHistories(issueHistories);

			return issue; 
		}catch (Exception e) {
			String msg = Messages.getString("ModifyIssueFacade.6") + fingerPrint; //$NON-NLS-1$
			logger.fatal(msg, e);
			BtsPlugin.getInstance().error(msg, e);
			return null;
		} finally {
			dbcon.close();
			dbcon = null;
		}
	}
	
	
	/**
	 * 自分が更新した回数カラム付き課題票一覧（全件）を取得します。
	 * 課題票一覧情報はローカルDBから取得します。
	 * 
	 * @return			課題票一覧（全件）
	 */
	public List getIssueList() throws BtsDBException {
		
		// 戻り値
		List issueList = null;
		
		// DBコネクション取得
		IssueDBConnection dbcon = new IssueDBConnection();
		try {
			
			// 課題票一覧を取得
			IssueDAO dao = new IssueDAO( dbcon );
			issueList = dao.getIssueList();
			
		} finally {
			dbcon.close();
			dbcon = null;
		}
		return issueList;
	}
	

	/**
	 * 指定のtypeを持つ{@link Issue 課題票}を取得します。
	 * 
	 * @param			type				課題票の種類（{@link jp.valtech.bts.data.IssueType}）
	 * @return			指定のtypeに該当する課題票
	 */
	public List getByType(String type) throws BtsDBException {
		
		// 戻り値
		List issueList = null;
		
		// DBコネクション取得
		IssueDBConnection dbcon = new IssueDBConnection();
		try {
			
			// 課題票一覧を取得
			IssueDAO dao = new IssueDAO( dbcon );
			issueList = dao.getByType(type);
			
		} finally {
			dbcon.close();
			dbcon = null;
		}
		return issueList;
	}
	
	
	/**
	 * 担当者一覧(重複なし)を取得します。
	 * 
	 * @return			担当者一覧(重複なし)
	 */
	public String[] getAssigned() throws BtsDBException {
		
		// 戻り値
		String[] assigned;
		
		// DBコネクション取得
		IssueDBConnection dbcon = new IssueDBConnection();
		try {
			
			// 担当者一覧を取得
			IssueDAO dao = new IssueDAO( dbcon );
			assigned = dao.getAssigned();
			
		} finally {
			dbcon.close();
			dbcon = null;
		}
		return assigned;
	}
	
	
	/**
	 * 課題票種別がRの担当者一覧(重複なし)を取得します。
	 * 
	 * @return			種別がRの担当者一覧(重複なし)
	 */
	public String[] getAssignedTypeR() throws BtsDBException {
		
		// 戻り値
		String[] assigned;
		
		// DBコネクション取得
		IssueDBConnection dbcon = new IssueDBConnection();
		try {
			
			// 担当者一覧を取得
			IssueDAO dao = new IssueDAO( dbcon );
			assigned = dao.getAssignedTypeR();
			
		} finally {
			dbcon.close();
			dbcon = null;
		}
		return assigned;
	}
	
	
	/**
	 * 課題票種別がRのカテゴリ一覧(重複なし)を取得します。
	 * 
	 * @return			種別がRのカテゴリ一覧(重複なし)
	 */
	public String[] getCategoryTypeR() throws BtsDBException {
		
		// 戻り値
		String[] category;
		
		// DBコネクション取得
		IssueDBConnection dbcon = new IssueDBConnection();
		try {
			
			// 担当者一覧を取得
			IssueDAO dao = new IssueDAO( dbcon );
			category = dao.getCategoryTypeR();
			
		} finally {
			dbcon.close();
			dbcon = null;
		}
		return category;
	}
	
	
	/**
	 * 優先度一覧(重複なし)を取得します。
	 * 
	 * @return			優先度一覧(重複なし)
	 */
	public String[] getPriority() throws BtsDBException {
		
		// 戻り値
		String[] priority;
		
		// DBコネクション取得
		IssueDBConnection dbcon = new IssueDBConnection();
		try {
			
			// 担当者一覧を取得
			IssueDAO dao = new IssueDAO( dbcon );
			priority = dao.getPriority();
			
		} finally {
			dbcon.close();
			dbcon = null;
		}
		return priority;
	}
	
}
