/***********************************************************************
 * Copyright(C) 2006 Valtech Co.,Ltd.
 * All Rights Reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 ***********************************************************************/
package jp.valtech.bts.dao;

import java.util.ArrayList;
import java.util.List;

import jp.valtech.bts.connection.DBConnection;
import jp.valtech.bts.data.Vote;

import org.apache.commons.dbutils.QueryRunner;
import org.apache.commons.dbutils.ResultSetHandler;
import org.apache.commons.dbutils.handlers.BeanListHandler;
import org.apache.commons.dbutils.handlers.ScalarHandler;

/**
 * <dl>
 * <dt><b>投票用のDAO</b></dt>
 * <dd> </dd>
 * <dt><b>使い方</b></dt>
 * <dd> </dd>
 * </dl>
 * 
 * @author		<A href="mailto:s_imai@valtech.jp">S.Imai</A>
 * @version	Ver.0.8
 */
public class VoteDAO extends DataBaseAccessor {

	/** テーブル作成SQL */
	private static final String CREATE_TABLE 
			= "CREATE TABLE VOTE" 
			+ " ( FINGERPRINT CHAR(40)" 
			+ " , VOTEUSER 	  VARCHAR(30)"
			+ " , VOTECNT 	  INTEGER DEFAULT 0" 
			+ " ) ;";

	/** テーブル削除SQL */
	private static final String DROP_TABLE 
	= "DROP TABLE VOTE CASCADE;";

	/** データ登録SQL */
	private static final String INSERT_SQL 
			= "INSERT INTO VOTE" 
			+ " ( FINGERPRINT"
			+ " , VOTEUSER" 
			+ " , VOTECNT" 
			+ " )"
			+ " VALUES (?, ?, ?)";

	/** データ削除SQL */
	private static final String DELETE_SQL 
			= "DELETE FROM VOTE WHERE FINGERPRINT=?";

	/** データを全件取得するSQL */
	private static final String SELECT_ALL 
			= "SELECT" 
			+ "   FINGERPRINT" 
			+ " , VOTEUSER" 
			+ " , VOTECNT" 
			+ " FROM VOTE";

	/** フィンガープリントに対応するデータを取得するSQL */
	private static final String SELECT_BY_FINGERPRINT 
			= SELECT_ALL + " WHERE FINGERPRINT=?";

	/** 投票数を1増やすSQL */
	private static final String UPDATE_VOTECNT 
			= "UPDATE VOTE" 
			+ "   SET VOTECNT=VOTECNT+1" 
			+ " WHERE FINGERPRINT=? AND VOTEUSER=?";

	/** 投票総数を取得するSQL */
	private static final String VOTECNT_SUM 
			= "SELECT SUM ( VOTECNT ) FROM VOTE WHERE FINGERPRINT=?";

	/**
	 * <DL>
	 * <DT><B>コンストラクタ</B>
	 * <DD></DD>
	 * </DL>
	 * 
	 * @param			connection			コネクション
	 */
	public VoteDAO(DBConnection connection) {
		super(connection);
	}

	/**
	 * テーブルを生成します。
	 * 
	 * @throws			BtsDBException
	 */
	public void createTable() throws BtsDBException {
		try {
			// テーブル生成用SQL実行
			new QueryRunner().update(this.con, CREATE_TABLE);

		} catch (Exception e) {
			throw new BtsDBException(e);
		}
	}

	/**
	 * テーブルを削除します。
	 * 
	 * @throws			BtsDBException
	 */
	public void dropTable() throws BtsDBException {
		try {
			// テーブル削除用SQL実行
			new QueryRunner().update(this.con, DROP_TABLE);
		} catch (Exception e) {
			throw new BtsDBException(e);
		}
	}

	/**
	 * <DL>
	 * <DT><B>データを登録します。</B>
	 * </DL>
	 * <DD></DD>
	 * </DL>
	 * 
	 * @param			vote			投票データ
	 * @throws			BtsDBException
	 */
	public void insert(Vote vote) throws BtsDBException {
		try {
			List param = new ArrayList();
			param.add(vote.getFingerPrint());
			param.add(vote.getVoteUser());
			param.add(new Integer(vote.getVoteCnt()));

			new QueryRunner().update(this.con, INSERT_SQL, param.toArray());
		} catch (Exception e) {
			throw new BtsDBException(e);
		}
	}

	/**
	 * <DL>
	 * <DT><B>データを削除します。</B>
	 * </DL>
	 * <DD></DD>
	 * </DL>
	 * 
	 * @param			vote			投票データ
	 * @throws 		BtsDBException
	 */
	public void deleteByFingerPrint(String fingerPrint) throws BtsDBException {
		try {
			new QueryRunner().update(this.con, DELETE_SQL, fingerPrint);
		} catch (Exception e) {
			throw new BtsDBException(e);
		}
	}

	/**
	 * <DL>
	 * <DT><B>指定されたフィンガープリントに属する投票データ全件を取得します。</B>
	 * </DL>
	 * <DD></DD>
	 * </DL>
	 * 
	 * @param			fingerPrint			フィンガープリント
	 * @return			指定されたフィンガープリントに属する投票データ全件
	 * @throws			BtsDBException
	 */
	public Vote[] get(String fingerPrint) throws BtsDBException {
		Vote[] votes = new Vote[] {};

		try {
			ResultSetHandler rsh = new BeanListHandler(Vote.class);

			List result = (List) new QueryRunner().query(this.con, SELECT_BY_FINGERPRINT, fingerPrint, rsh);

			votes = (Vote[]) result.toArray(votes);

		} catch (Exception e) {
			throw new BtsDBException(e);
		}

		return votes;
	}

	/**
	 * <DL>
	 * <DT><B>指定された投票データの投票数を＋１します。</B>
	 * </DL>
	 * <DD></DD>
	 * </DL>
	 * 
	 * @param			vote			投票データ
	 * @throws 		BtsDBException
	 */
	public int cntPlus(Vote vote) throws BtsDBException {

		try {
			
			List param = new ArrayList();
			param.add(vote.getFingerPrint());
			param.add(vote.getVoteUser());
			
			return new QueryRunner().update(this.con, UPDATE_VOTECNT, param.toArray());
			
		} catch (Exception e) {
			throw new BtsDBException(e);
		}

	}

	/**
	 * <DL>
	 * <DT><B>指定されたフィンガープリントに属する課題票の投票総数を返します。</B>
	 * </DL>
	 * <DD></DD>
	 * </DL>
	 * 
	 * @param			fingerPrint			フィンガープリント
	 * @return			投票総数
	 * @throws 		BtsDBException
	 */
	public int getSum(String fingerPrint) throws BtsDBException {

		try {
			
			ResultSetHandler rsh = new ScalarHandler(1);

			Object result = new QueryRunner().query(this.con, VOTECNT_SUM, fingerPrint, rsh);
			
			if (result == null) {
				return 0;
			}
			
			return Integer.parseInt(result.toString());

		} catch (Exception e) {
			throw new BtsDBException(e);
		}

	}

}
