/***********************************************************************
 * Copyright(C) 2006 Valtech Co.,Ltd.
 * All Rights Reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 ***********************************************************************/
package jp.valtech.bts.command.server;

import jp.valtech.bts.data.Issue;
import jp.valtech.bts.data.IssueType;
import jp.valtech.bts.data.NetworkConfig;
import jp.valtech.bts.facade.ReceiveGarbageIssueFacade;
import jp.valtech.bts.facade.ReceiveReleaseIssueFacade;
import jp.valtech.bts.facade.VoteDAOFacade;
import jp.valtech.bts.network.Request;
import jp.valtech.bts.network.Response;
import jp.valtech.bts.network.command.TCPServerCommand;
import jp.valtech.bts.ui.BtsPlugin;


/**
 * <dl>
 *   <dt><b>クライアントから課題票の送信要求処理を行います。</b></dt>
 *   <dd><ul>
 * 	   <li>他のクライアントからの課題票送信リクエストを受信し、このクラスが呼び出されると、送信された課題票を受け取ります。受信できる課題票はひとつのみです。
 * 	   <li>課題票を受け取ると、レスポンスを作成します。レスポンスには受信が成功したかどうかを設定するだけです。
 *     <li>エラーが発生した場合には、えらー情報をレスポンスメッセージに含めます。
 *   </ul></dd>
 * </dl>
 * 
 * @see			jp.valtech.bts.command.client.SendIssue
 * @see			jp.valtech.bts.network.command.TCPServerCommand
 * @author		<A href="mailto:m_sugitou@valtech.jp">M.Sugito</A>
 * @version	Ver.0.8
 */
public class SendIssueServer extends TCPServerCommand {

	/**
	 * なにもしない。
	 */
	public SendIssueServer() {
		super();
	}

	/**
	 * {@link jp.valtech.bts.network.Request リクエストオブジェクト}から課題票を取得します。
	 * <br>
	 * 取得すべき情報は以下のものがあります。<br>
	 * 　・課題票本体<br>
	 * 　・課題票履歴情報<br>
	 * 　・送信元のユーザ名<br>
	 * 
	 * これらの全ての受信を確認すると{@link jp.valtech.bts.network.Response レスポンスオブジェクト}に
	 * {@link Response#OK}を設定して送信元に返信します。<br>
	 * 受信した課題票情報のローカルDBへの登録は返信後に行います。DBへの登録処理結果は送信元には通知しません。
	 * 
	 * @param		request			リクエストオブジェクト
	 * @param		response		レスポンスオブジェクト
	 * @param		myconfig		ローカル端末のネットワーク情報
	 */
	public void execute(Request request, Response response, NetworkConfig myconfig){

		// 課題票取得
		Issue receiveIssue= (Issue)request.getParameterObject( "issue" );
		
		// 送信者取得
		String fromUser = request.getParameterString("fromUser");

		// 受信オブジェクトのチェック。チェック結果を送信元に返信します。
		// 後で行うDBへの登録処理の結果は送信元には送りません。
		if( receiveIssue != null) {
			
			// "Response#OK"を送信元に返信
			response.setCode(Response.OK);
			reply(response);
		
		} else {
			// "Response#ERROR"を送信元に返信
			response.setCode(Response.ERROR);
			response.setMessage("NullPointerException : request data is null");
			reply(response);
			return;
		}
		
		// 受信した課題票情報をDBに登録する
		String fromAddr = getRequestClient().getHostAddress();
		logger.debug("FromUser    :" + fromUser + "(" + fromAddr + ")"); //$NON-NLS-1$
		logger.debug("RequestIssue:" + receiveIssue.getTitle()); //$NON-NLS-1$
		
		try {
			
			if(IssueType.RELEASE_VALUE.equals(receiveIssue.getType())) {
				// 課題票情報を保存
				ReceiveReleaseIssueFacade facade = new ReceiveReleaseIssueFacade();
				facade.saveReceivedIssue(receiveIssue);

				// 更新メッセージを生成・保存
				facade.saveReceiveMessage(fromUser, fromAddr);
				
				// 投票情報更新
				VoteDAOFacade voteDAOFacade = new VoteDAOFacade();
				voteDAOFacade.cleanInsert(receiveIssue.getVotes());
			}
			
			else if(IssueType.GARBAGE_VALUE.equals(receiveIssue.getType())) {

				// ごみ箱更新を保存
				ReceiveGarbageIssueFacade facade = new ReceiveGarbageIssueFacade();
				facade.changeGarbageIssue(receiveIssue);
				
				// 更新メッセージを生成・保存
				facade.saveReceiveMessage(fromUser, fromAddr);
			}
			
		} catch (Exception e) {
			String msg = "failed 'send issue request'" ;
			logger.fatal(msg, e);
			BtsPlugin.getInstance().error(msg, e);
		}
	}

}
