/***********************************************************************
 * Copyright(C) 2006 Valtech Co.,Ltd.
 * All Rights Reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 ***********************************************************************/
package jp.valtech.bts.util;

import java.security.GeneralSecurityException;
import java.security.InvalidKeyException;
import java.security.Key;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;

import javax.crypto.Cipher;
import javax.crypto.SecretKey;
import javax.crypto.SecretKeyFactory;
import javax.crypto.spec.DESKeySpec;

import jp.valtech.bts.ui.BtsPlugin;

/**
 * <dl><dt><b>汎用暗号化・復号化・ハッシュ取得ライブラリ:</b></dt>
 * 	<dd>
 * 	JCEの機能を利用して暗号化を行うラッパークラスです。
 * 	</dd>
 * </dl>
 *
 * @author		<A href="mailto:iinuma@valtech.jp">K.Iinuma</A>
 * @version	Ver.0.8
 */
public class CipherUtility {

	private static final char[] HEX = { '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'A', 'B', 'C', 'D', 'E', 'F' };

	// 鍵(指定されない場合にはこのキーを使う)
//	private static final byte[] defaultKey = {
//		(byte)0x0e ,(byte)0x9b ,(byte)0xb9 ,(byte)0xda ,(byte)0x79 ,(byte)0xd3 ,(byte)0xdc ,(byte)0xf1
//	};

	/** 暗号化アルゴリズム : AES */
	public final static String KEY_ALGORITHM_TYPE_AES		= "AES";
	/** 暗号化アルゴリズム : DES */
	public final static String KEY_ALGORITHM_TYPE_DES		= "DES";
	/** 暗号化アルゴリズム : SHA1 */
	public final static String KEY_ALGORITHM_TYPE_SHA1	= "SHA1";

	/** 処理モード : 暗号化 */
	public final static byte ENCRYPT_MODE = 1;
	/** 処理モード : 復号化 */
	public final static byte DECRYPT_MODE = 2;

	//-------------------- DES で 暗号化

	/**
	 * 指定されたバイナリデータを、引数のキーを使用して暗号化します。
	 * 暗号化には、3DESが使用されます。
	 * @param target						対象とするバイナリデータ
	 * @param key							キー
	 * @return								暗号化されたバイナリ
	 * @throws InvalidKeyException			正しくないキーが利用された場合
	 * @throws IllegalStateException		正しくない状態
	 * @throws GeneralSecurityException	セキュリティ例外
	 */
	public static byte[] encrypt( byte[] target , String key ) throws InvalidKeyException, IllegalStateException, GeneralSecurityException{
		return crypt( ENCRYPT_MODE , target , key.getBytes() , KEY_ALGORITHM_TYPE_DES );
	}

	/**
	 * @param target						対象とするバイナリデータ
	 * @param key							キー
	 * @return								復号化されたバイナリ
	 * @throws InvalidKeyException			正しくないキーが利用された場合
	 * @throws IllegalStateException		正しくない状態
	 * @throws GeneralSecurityException	セキュリティ例外
	 */
	public static byte[] decrypt( byte[] target , String key ) throws InvalidKeyException, GeneralSecurityException, IllegalStateException{
		return crypt( DECRYPT_MODE , target , key.getBytes() , KEY_ALGORITHM_TYPE_DES );
	}

	/**
	 * @param mode							処理モード
	 * @param target						対象とするバイナリデータ
	 * @param pkey							暗号化キー
	 * @param algorithm						アルゴリズム
	 * @return								結果バイナリデータ
	 * @throws InvalidKeyException			正しくないキーが利用された場合
	 * @throws IllegalStateException		正しくない状態
	 * @throws GeneralSecurityException	セキュリティ例外
	 */
	private static byte[] crypt( byte mode , byte[] target , byte[] pkey , String algorithm ) throws InvalidKeyException, GeneralSecurityException, IllegalStateException{

		byte[] result = null;
		
		pkey = hash( pkey ).getBytes();

		Key key = createDESCryptKey( pkey , algorithm );

		Cipher cipher = Cipher.getInstance( algorithm );
		cipher.init( mode , key );

		result = cipher.doFinal( target );

		return result;
	}


	private static SecretKey createDESCryptKey( byte[] b , String algorithm ) throws InvalidKeyException, GeneralSecurityException {
		DESKeySpec  keySpec = new DESKeySpec( b );
		SecretKeyFactory keyFactory
				 = SecretKeyFactory.getInstance( algorithm );
		return keyFactory.generateSecret(keySpec);
	}


	/**
	 * @param target		対象とするバイナリデータ
	 * @return				指定されたアルゴリズムで取得した引数targetのハッシュ値
	 */
	//-------------------- SHA で 暗号化
	public static String hash( String target ) {
		String result = null;
		try {
			result = hash( target , KEY_ALGORITHM_TYPE_SHA1 );
		} catch (NoSuchAlgorithmException e) {
			BtsPlugin.getInstance().errorlog(e);
		}
		return result;
	}

	/**
	 * @param target						対象とするバイナリデータ
	 * @param algorithm						アルゴリズム
	 * @return								指定されたアルゴリズムで取得した引数targetのハッシュ値
	 * @throws NoSuchAlgorithmException	アルゴリズムが見つからない場合

	 */
	private static String hash( String target , String algorithm ) throws NoSuchAlgorithmException{
		return hash( target.getBytes() , algorithm );
	}

	/**
	 * @param target		対象とするバイナリデータ
	 * @return				指定されたアルゴリズムで取得した引数targetのハッシュ値
	 */
	public static String hash( byte[] target ){
		String result = null;
		try {
			result =  hash( target , KEY_ALGORITHM_TYPE_SHA1 );
		} catch (NoSuchAlgorithmException e) {
			BtsPlugin.getInstance().errorlog(e);
		}
		return result;
	}

	/**
	 * @param target						対象とするバイナリデータ
	 * @param algorithm						アルゴリズム
	 * @return								指定されたアルゴリズムで取得した引数targetのハッシュ値
	 * @throws NoSuchAlgorithmException	アルゴリズムが見つからない場合
	 */
	private static String hash( byte[] target , String algorithm ) throws NoSuchAlgorithmException{

		MessageDigest md = MessageDigest.getInstance( algorithm );

		md.update( target );
			
		byte[] b = md.digest();

		StringBuffer sb = new StringBuffer();
		for (int i = 0; i < b.length; i++) {
			sb.append(HEX[byte2int(b[i]) / 16]);
			sb.append(HEX[byte2int(b[i]) % 16]);
		}

		return sb.toString();

	}

	private static int byte2int(byte b) {
		if(b < 0) {
			return b + 0x100;
		}
		return b;
	}


}
