/***********************************************************************
 * Copyright(C) 2006 Valtech Co.,Ltd.
 * All Rights Reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 ***********************************************************************/
package jp.valtech.bts.ui.navigator;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import jp.valtech.bts.dao.BtsDBException;
import jp.valtech.bts.data.CurrentProject;
import jp.valtech.bts.data.Issue;
import jp.valtech.bts.data.IssueType;
import jp.valtech.bts.facade.ModifyIssueFacade;
import jp.valtech.bts.ui.BtsPlugin;
import jp.valtech.bts.ui.action.OpenCalendarDlgAction;
import jp.valtech.bts.util.BTSUtility;
import jp.valtech.bts.util.Logging;

import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.CheckboxTableViewer;
import org.eclipse.jface.viewers.ColumnWeightData;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.TableLayout;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.window.IShellProvider;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.forms.events.HyperlinkAdapter;
import org.eclipse.ui.forms.events.HyperlinkEvent;
import org.eclipse.ui.forms.widgets.ImageHyperlink;

/**
 * ファイル出力設定用のダイアログです。
 * {@link jp.valtech.bts.ui.navigator.NavigatorView ナビゲータービュー}から呼ばれるダイアログです。
 * 
 * @author		<A href="mailto:m_sugitou@valtech.jp">M.Sugito</A>
 * @version	Ver.0.8
 */
public class IssueClassifiedExportDialog extends Dialog implements Logging{

    /** 「ヘッダ文字列」用テキストボックス */
	Text headerTxt;
    /** 「フッタ文字列」用テキストボックス */
	Text footerTxt;

    /** 「課題票一覧」用チェックボックス */
    Button issueListRdi;
	/** 「担当者別カウント」用チェックボックス */
    Button cntAssignedRdi;
	/** 「ステータス別カウント」用チェックボックス */
    Button cntStatusRdi;
	/** 「バグ累計」用チェックボックス */
    Button totalBugRdi;
	/** 「1日単位」用チェックボックス */
    Button unitDayRdi;
	/** 「1週間単位」用チェックボックス */
    Button unitWeekRdi;
	/** 「1ヶ月単位」用チェックボックス */
    Button unitMonthRdi;
    
	/** 「ファイル形式：HTML」用ラジオボタン */
    Button fileHTMRdi;
	/** 「ファイル形式：PDF」用ラジオボタン */
    private Button filePDFRdi;
	/** 「ファイル形式：XML」用ラジオボタン */
    Button fileXMLRdi;
	/** 「ファイル形式：CSV」用ラジオボタン */
    private Button fileCSVRdi;
    
    /** 「起票日(from)」用テキストボックス */
    private Text frmDateTxt;
    /** 「起票日(to)用テキストボックス */
    private Text toDateTxt;
    /**「 カテゴリ一覧テーブル」用テーブルビューア */
    private CheckboxTableViewer categoryViewer;
    /** 「内緒のバグを含めない」用チェックボックス */
    private Button secretChk;
    /** 「保留中の課題を含めない」用チェックボックス */
    private Button reserveChk;

	/** カテゴリ一覧 */
	private String[] category;


	/**
	 * 親クラスにシェルを設定します。
	 * 
	 * @param			parentShell			シェル
	 */
	public IssueClassifiedExportDialog(Shell parentShell) {
		super(parentShell);		
	}

	/**
	 * 親クラスにプロバイダを設定します。
	 * 
	 * @param			parentShell
	 */
	public IssueClassifiedExportDialog(IShellProvider parentShell) {
		super(parentShell);
	}

    /**
     * 当ダイアログの設定します。タイトルを設定します。
     * 
     * @param			newShell			該ダイアログのシェル
     */
    protected void configureShell(Shell newShell) {
        super.configureShell(newShell);
        newShell.setText(Messages.getString("IssueClassifiedExportDialog.0"));  //$NON-NLS-1$
    }



    /**
     * フィルターパターン編集用のダイアログを生成します。
     * 
     * @param			parent					親コンポジット
     * @return			当ダイアログのコントロールインスタンス
     * @see				Dialog#createDialogArea(org.eclipse.swt.widgets.Composite)
     */
    protected Control createDialogArea(Composite parent) {
        Composite composite = (Composite) super.createDialogArea(parent);
        try {
        	
        	// 「ファイル出力形式」設定のUIを生成
        	createFileFormatArea(composite);
        		
            // 「出力帳票」設定のUIを生成
            createOutputChooseArea(composite);
        		
            // 「出力条件」設定のUIを生成
            createOutputConditionArea(composite);
        	
        	// 「ヘッダ」「フッタ」設定のUIを生成
            createHeaderArea(composite);

            // セパレータ生成
            createSeparatorLine(composite);
            
        }catch (Exception e) {
			String msg = Messages.getString("IssueClassifiedExportDialog.29"); //$NON-NLS-1$
			logger.fatal(msg, e);
			BtsPlugin.getInstance().error(msg, e);
		}
        return composite;
    }
    
    /**
     * 「出力項目選択」のUIを生成します。
     * 親コンポジットに対して「出力項目選択」のUIを生成します。<br>
     * 
     * @param			parent						親コンポジット
     * @throws			Exception
     */
    private void createFileFormatArea(Composite parent) throws Exception {

    	// このエリアのコンポジットを生成。
        Composite thisArea = new Composite(parent, SWT.NONE);

        // レイアウトを設定。
        thisArea.setLayout(new GridLayout());
        thisArea.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
        
        // 「出力ファイル形式」グループの生成
    	Group group = new Group(thisArea, SWT.NONE);
        group.setText(Messages.getString("IssueClassifiedExportDialog.1")); //$NON-NLS-1$

        // 「出力ファイル形式」のレイアウト設定
    	GridLayout layout = new GridLayout();
        layout.numColumns = 4;
    	group.setLayout(layout);
    	group.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));

    	// 「HTML」のラジオボタン
        fileHTMRdi = new Button(group, SWT.RADIO);
        fileHTMRdi.setText("HTML");

    	// 「XML」のラジオボタン
        fileXMLRdi = new Button(group, SWT.RADIO);
        fileXMLRdi.setText("XML");

    	// 「PDF」のラジオボタン
        filePDFRdi = new Button(group, SWT.RADIO);
        filePDFRdi.setText("PDF");
        filePDFRdi.setEnabled(false);

    	// 「CSV」のラジオボタン
        fileCSVRdi = new Button(group, SWT.RADIO);
        fileCSVRdi.setText("CSV");
        fileCSVRdi.setEnabled(false);

    }
    
    
    /**
     * 「出力項目選択」のUIを生成します。
     * 親コンポジットに対して「出力項目選択」のUIを生成します。<br>
     * 
     * @param			parent						親コンポジット
     * @throws			Exception
     */
    private void createOutputChooseArea(Composite parent) throws Exception {

    	// このエリアのコンポジットを生成。
        Composite thisArea = new Composite(parent, SWT.NONE);

    	// レイアウトを設定。
        thisArea.setLayout(new GridLayout());
        thisArea.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
        
        // 「出力ファイル形式」グループの生成
    	Group group = new Group(thisArea, SWT.NONE);
        group.setText(Messages.getString("IssueClassifiedExportDialog.2")); //$NON-NLS-1$

        // 「出力ファイル形式」のレイアウト設定
    	group.setLayout(new GridLayout());
    	group.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));


        // 「課題票一覧」用チェックボックス
        issueListRdi = new Button(group, SWT.RADIO);
    	issueListRdi.setText(Messages.getString("IssueClassifiedExportDialog.3")); //$NON-NLS-1$
    	
    	// 「課題票一覧」用チェックボックスにセレクションリスナ追加
    	issueListRdi.addSelectionListener(new SelectionAdapter() {

    		/*
    		 *  (非 Javadoc)
    		 * @see org.eclipse.swt.events.SelectionListener#widgetSelected(org.eclipse.swt.events.SelectionEvent)
    		 */
			public void widgetSelected(SelectionEvent e) {
				
				// 出力項目のいずれかにチェックがされている場合は「ファイル出力」ボタンを非活性
				if(issueListRdi.getSelection() || cntAssignedRdi.getSelection()
						|| cntStatusRdi.getSelection() ||totalBugRdi.getSelection()) {
					
			        getButton(IDialogConstants.OK_ID).setEnabled(true);
					
				} else {
					
					getButton(IDialogConstants.OK_ID).setEnabled(false);
					
				}
				
			}
    		
    	});

    	// 「担当者別カウント」用チェックボックス
        cntAssignedRdi = new Button(group, SWT.RADIO);
    	cntAssignedRdi.setText(Messages.getString("IssueClassifiedExportDialog.4")); //$NON-NLS-1$
    	
    	// 「担当者カウント」用チェックボックスにセレクションリスナを追加
    	cntAssignedRdi.addSelectionListener(new SelectionAdapter() {

    		/*
    		 *  (非 Javadoc)
    		 * @see org.eclipse.swt.events.SelectionListener#widgetSelected(org.eclipse.swt.events.SelectionEvent)
    		 */
			public void widgetSelected(SelectionEvent e) {
				
				// 出力項目のいずれかにチェックがされている場合は「ファイル出力」ボタンを非活性
				if(issueListRdi.getSelection() || cntAssignedRdi.getSelection()
						|| cntStatusRdi.getSelection() ||totalBugRdi.getSelection()) {
					
			        getButton(IDialogConstants.OK_ID).setEnabled(true);
					
				} else {
					
					getButton(IDialogConstants.OK_ID).setEnabled(false);
					
				}
				
			}
    		
    	});

    	// 「ステータス別カウント」用チェックボックス
        cntStatusRdi = new Button(group, SWT.RADIO);
    	cntStatusRdi.setText(Messages.getString("IssueClassifiedExportDialog.5")); //$NON-NLS-1$
    	cntStatusRdi.addSelectionListener(new SelectionAdapter() {

    		/*
    		 *  (非 Javadoc)
    		 * @see org.eclipse.swt.events.SelectionListener#widgetSelected(org.eclipse.swt.events.SelectionEvent)
    		 */
			public void widgetSelected(SelectionEvent e) {
				
				// 出力項目のいずれかにチェックがされている場合は「ファイル出力」ボタンを非活性
				if(issueListRdi.getSelection() || cntAssignedRdi.getSelection()
						|| cntStatusRdi.getSelection() ||totalBugRdi.getSelection()) {
					
			        getButton(IDialogConstants.OK_ID).setEnabled(true);
					
				} else {
					
					getButton(IDialogConstants.OK_ID).setEnabled(false);
					
				}
				
			}
    		
    	});

    	// 「バグ累計」用チェックボックス
        totalBugRdi = new Button(group, SWT.RADIO);
    	totalBugRdi.setText(Messages.getString("IssueClassifiedExportDialog.6")); //$NON-NLS-1$
    	
    	// 「バグ累計」用チェックボックスにセレクションリスナ設定
    	totalBugRdi.addSelectionListener(new SelectionAdapter() {
    		
    		/*
    		 *  (非 Javadoc)
    		 * @see org.eclipse.swt.events.SelectionListener#widgetSelected(org.eclipse.swt.events.SelectionEvent)
    		 */
			public void widgetSelected(SelectionEvent e) {
				
				// 出力項目のいずれかにチェックがされている場合は「ファイル出力」ボタンを非活性
				if(issueListRdi.getSelection() || cntAssignedRdi.getSelection()
						|| cntStatusRdi.getSelection() ||totalBugRdi.getSelection()) {
					
			        getButton(IDialogConstants.OK_ID).setEnabled(true);
					
				} else {
					
					getButton(IDialogConstants.OK_ID).setEnabled(false);
					
				}
				
				// 「バグ累計」用チェックボックスがチェックされている場合
				if(totalBugRdi.getSelection()) {
					
					// 出力単位用ラジオボタンを活性化
					unitDayRdi.setEnabled(true);
					unitWeekRdi.setEnabled(true);
					unitMonthRdi.setEnabled(true);
					
				} else {
					
					// 出力単位用ラジオボタンを非活性化
					unitDayRdi.setEnabled(false);
					unitWeekRdi.setEnabled(false);
					unitMonthRdi.setEnabled(false);
					
				}
				
			}
    		
    	});

    	// 出力単位コンポジット生成
    	Composite unit = new Composite(group, SWT.NONE);
    	
    	// レイアウト設定
    	unit.setLayout(new GridLayout());
    	unit.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
    	
    	// 左インデント設定
        GridData indent = new GridData(GridData.FILL_BOTH);
        indent.horizontalIndent = 15;

        // 「1日単位」用ラジオボタン
        unitDayRdi = new Button(unit, SWT.RADIO);
        unitDayRdi.setLayoutData(indent);
        unitDayRdi.setText(Messages.getString("IssueClassifiedExportDialog.7")); //$NON-NLS-1$
        unitDayRdi.setSelection(true);
        unitDayRdi.setEnabled(false);

        // 「1週間単位」用ラジオボタン
    	unitWeekRdi = new Button(unit, SWT.RADIO);
        unitWeekRdi.setLayoutData(indent);
    	unitWeekRdi.setText(Messages.getString("IssueClassifiedExportDialog.8")); //$NON-NLS-1$
    	unitWeekRdi.setEnabled(false);

        // 「1ヶ月単位」用ラジオボタン
        unitMonthRdi = new Button(unit, SWT.RADIO);
        unitMonthRdi.setLayoutData(indent);
    	unitMonthRdi.setText(Messages.getString("IssueClassifiedExportDialog.9")); //$NON-NLS-1$
    	unitMonthRdi.setEnabled(false);
        
    }
    
    
    /**
     * 「出力する条件」のUIを生成します。
     * 親コンポジットに対して「出力する条件」のUIを生成します。<br>
     * 
     * @param 			parent						親コンポジット
     * @throws Exception
     */
    private void createOutputConditionArea(Composite parent) throws Exception {
    	
    	// このエリアのコンポジットを生成。
    	Composite thisArea = new Composite(parent, SWT.NONE);
    	
    	// レイアウトを設定。
        thisArea.setLayout(new GridLayout());
        thisArea.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
        
        // 「出力する条件」グループの生成
    	Group group = new Group(thisArea, SWT.NONE);
        group.setText(Messages.getString("IssueClassifiedExportDialog.10")); //$NON-NLS-1$
        
        // 「出力する条件」のレイアウト設定
    	GridLayout layout = new GridLayout();
        layout.numColumns = 7;
        layout.verticalSpacing = 10;
    	group.setLayout(layout);
    	group.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
    	
        // 「FromDate」ラベル
        Label label = new Label(group, SWT.NONE);
        label.setText(Messages.getString("IssueClassifiedExportDialog.11")); //$NON-NLS-1$

        // 「FromDate」テキストボックス
        frmDateTxt = new Text(group,SWT.BORDER);
		GridData gridData = new GridData(GridData.FILL_HORIZONTAL); 
        gridData.widthHint = 80;
        frmDateTxt.setLayoutData(gridData);
        
        // 「FromDate」カレンダーボタン
		ImageHyperlink calFrom = new ImageHyperlink(group, SWT.NULL);
		calFrom.setImage(BtsPlugin.getInstance().getImage("calendar.gif"));
		calFrom.setBackground(Display.getDefault().getSystemColor(SWT.COLOR_WIDGET_BACKGROUND));
		calFrom.setToolTipText(Messages.getString("IssueClassifiedExportDialog.12")); //$NON-NLS-1$
		calFrom.addHyperlinkListener(new HyperlinkAdapter() {
			public void linkActivated(HyperlinkEvent e) {
				// 日付入力用のカレンダーを開きます。
				OpenCalendarDlgAction action = new OpenCalendarDlgAction(frmDateTxt);
				action.run();
			}
		});

        // 「ToDate」ラベル
        label = new Label(group, SWT.NONE);
        label.setText(Messages.getString("IssueClassifiedExportDialog.13")); //$NON-NLS-1$
        
        //「ToDate」テキストボックス
        toDateTxt = new Text(group,SWT.BORDER);
		toDateTxt.setLayoutData(gridData);
		
        // 「ToDate」カレンダーボタン
		ImageHyperlink calTo = new ImageHyperlink(group, SWT.NULL);
		calTo.setImage(BtsPlugin.getInstance().getImage("calendar.gif"));
		calTo.setBackground(Display.getDefault().getSystemColor(SWT.COLOR_WIDGET_BACKGROUND));
		calTo.setToolTipText(Messages.getString("IssueClassifiedExportDialog.14")); //$NON-NLS-1$
		calTo.addHyperlinkListener(new HyperlinkAdapter() {
			public void linkActivated(HyperlinkEvent e) {
				// 日付入力用のカレンダーを開きます。
				OpenCalendarDlgAction action = new OpenCalendarDlgAction(toDateTxt);
				action.run();
			}
		});		

        // 書式用ラベル
        label = new Label(group, SWT.NONE);
        label.setText("(YYYY/MM/DD)");


        // 「カテゴリ」テーブルの表示設定
        Table table = new Table(group
        		, SWT.CHECK | SWT.H_SCROLL | SWT.V_SCROLL | SWT.MULTI | SWT.FULL_SELECTION | SWT.BORDER);
        table.setLinesVisible(true);
        table.setHeaderVisible(true);

        // テーブルレイアウトの設定
        TableLayout tableLayout = new TableLayout();
        table.setLayout(tableLayout);
        
        // カラムの設定
        tableLayout.addColumnData(new ColumnWeightData(350, true));
        TableColumn tc = new TableColumn(table, SWT.NONE, 0);
        tc.setText(Messages.getString("IssueClassifiedExportDialog.15")); //$NON-NLS-1$

        // テーブルの横幅・高さ・マージンの調整
        GridData tableGridData = new GridData(GridData.FILL_HORIZONTAL);
        tableGridData.heightHint = 45;
        tableGridData.widthHint = 180;
        tableGridData.horizontalSpan = 7;

        // テーブルのビューアオブジェクト設定
        categoryViewer = new CheckboxTableViewer(table);
        categoryViewer.getControl().setLayoutData(tableGridData);
        
        // テーブルにプロバイダを設定
        categoryViewer.setContentProvider(getCategoryTableContentProvider());
        categoryViewer.setLabelProvider(getCategoryTableLabelProvider());

        // 初期値を設定
		CurrentProject project = CurrentProject.getInsance();
		if(project.isOpen()) {
			ModifyIssueFacade issueModifyFacade = new ModifyIssueFacade(); 
			category = issueModifyFacade.getCategoryTypeR();
			categoryViewer.setInput(category);
		}
        
        // デフォルト設定
        categoryViewer.setAllChecked(true);
        
        
        // 「内緒のバグを含めない」チェックボックス
        secretChk = new Button(group, SWT.CHECK);
        secretChk.setText(Messages.getString("IssueClassifiedExportDialog.16")); //$NON-NLS-1$
        secretChk.setSelection(false);
        GridData data = new GridData(GridData.FILL_HORIZONTAL);
        data.horizontalSpan = 7;
        secretChk.setLayoutData(data);

        // 「保留中の課題を含めない」チェックボックス
        reserveChk = new Button(group, SWT.CHECK);
        reserveChk.setText(Messages.getString("IssueClassifiedExportDialog.17")); //$NON-NLS-1$
        reserveChk.setSelection(false);
        reserveChk.setLayoutData(data);
    	
    }
    
    
    /**
     * 「カテゴリテーブル一覧」のコンテントプロバイダを返します。
     * 
     * @return			「カテゴリテーブル一覧」のコンテントプロバイダ
     */
    private IStructuredContentProvider getCategoryTableContentProvider() {
        return new IStructuredContentProvider() {
            public Object[] getElements(Object inputElement) {
            	return category;
            }
            public void dispose() {
            }
            public void inputChanged(Viewer viewer, Object oldInput,
                    Object newInput) {
            }
        };
    }

    /**
     * {@link CategoryTableLabelProvider 「カテゴリテーブル一覧」のコンテントプロバイダ}を返します。
     * 
     * @return			{@link CategoryTableLabelProvider}
     */
    private ILabelProvider getCategoryTableLabelProvider() {
        return new CategoryTableLabelProvider();
    }

    
    /**
     * カテゴリテーブル用のラベルプロバイダです。
     * 
     */
    private class CategoryTableLabelProvider extends LabelProvider implements ITableLabelProvider {

		/* (non-Javadoc)
		 * @see org.eclipse.jface.viewers.ITableLabelProvider#getColumnImage(java.lang.Object, int)
		 */
		public Image getColumnImage(Object element, int columnIndex) {
		    return null;
		}
		
		/* (non-Javadoc)
		 * @see org.eclipse.jface.viewers.ITableLabelProvider#getColumnText(java.lang.Object, int)
		 */
		public String getColumnText(Object element, int columnIndex) {
			if(columnIndex == 0) {
				return element.toString();
			}
			return "";
		}
    }
    
    
    /**
     * 「ヘッダ」「フッタ」のUIを生成。
     * 親コンポジットに「ヘッダ」「フッタ」設定のUIを生成します。
     * 
     * @param			parent					親コンポジット
     * @throws			Exception
     */
    private void createHeaderArea(Composite parent) throws Exception {
    	
    	// このエリアのコンポジットを生成。
    	Composite thisArea = new Composite(parent, SWT.NONE);

    	// レイアウトを設定。
    	GridLayout layout = new GridLayout();
    	layout.numColumns = 2;
    	thisArea.setLayout(layout);
        thisArea.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));

        // テキストボックスのサイズ指定
        GridData gridData = new GridData(GridData.FILL); 
        gridData.widthHint = 220;

        // 「ヘッダ」ラベル
        Label label = new Label(thisArea, SWT.NONE);
        label.setText(Messages.getString("IssueClassifiedExportDialog.18")); //$NON-NLS-1$

        // 「ヘッダ」テキストボックス
        headerTxt = new Text(thisArea, SWT.BORDER);
        headerTxt.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
        headerTxt.setLayoutData(gridData);

    
        // 「フッタ」ラベル
        label = new Label(thisArea, SWT.NONE);
        label.setText(Messages.getString("IssueClassifiedExportDialog.19")); //$NON-NLS-1$

        // 「フッタ」テキストボックス
        footerTxt = new Text(thisArea, SWT.BORDER);
        footerTxt.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
        footerTxt.setLayoutData(gridData);
    }

    /**
     * セパレータラインを生成します。
     * 親コンポジットに対してセパレータラインを生成します。
     * 
     * @param			parent						親コンポジット
     */
    private void createSeparatorLine(Composite parent) {
        // 親コンポーネントにラベルを生成
        Label separator = new Label(parent, SWT.HORIZONTAL | SWT.SEPARATOR);
        
        // レイアウトを設定。グリッドレイアウトを使う。
        GridData gd = new GridData(GridData.FILL_HORIZONTAL);
        gd.horizontalSpan = 1;
        separator.setLayoutData(gd);
    }

    /**
     * ボタンバーにボタンを生成します。
     * 左側のボタンのラベルを「ファイル出力」にするためにオーバーライドしています。
     * 
     * @param			parent						親コンポジット
     */
    protected void createButtonsForButtonBar(Composite parent) {
    	
        createButton(parent, IDialogConstants.OK_ID, Messages.getString("IssueClassifiedExportDialog.20"), true); //$NON-NLS-1$
        	
        // 「ファイル出力」ボタンを非活性
        getButton(IDialogConstants.OK_ID).setEnabled(false);

        createButton(parent, IDialogConstants.CANCEL_ID,
                IDialogConstants.CANCEL_LABEL, false);
        
    }

    /**
     * 「ファイル出力」ボタンを押したときの処理です。
     * エラーチェックを行い、エラーが無い場合はファイルに出力します。
     */
	protected void okPressed() {
		
		// 出力条件に合う課題票リスト
		List issueList = null;
			
		// 入力された起票日取得
		String frmDateString = frmDateTxt.getText().trim();
		String toDateString = toDateTxt.getText().trim();
			
		// 起票日が入力されているか判定
		if(!"".equals(frmDateString) || !"".equals(toDateString)) {
				
			// エラーチェック
			if(!BTSUtility.dateTextError(frmDateString, toDateString)) {
				// エラーあり
				return;
			}
				
		}
			
		// 出力条件に合う課題票リスト取得
		issueList = addCondition();
			
		// プロジェクトが開いていない場合
		if(issueList == null) {
			MessageDialog.openError(this.getShell(), Messages.getString("IssueClassifiedExportDialog.21"), Messages.getString("IssueClassifiedExportDialog.22")); //$NON-NLS-1$ //$NON-NLS-2$
			return;
		}
			
		// 出力対象の課題票が無い場合
		if(issueList.size() == 0) {
			MessageDialog.openError(this.getShell(), Messages.getString("IssueClassifiedExportDialog.23"), Messages.getString("IssueClassifiedExportDialog.24")); //$NON-NLS-1$ //$NON-NLS-2$
			return;
		}
		
		
		// ファイル保存ダイアログ生成
		FileDialog fileDialog = new FileDialog(this.getShell(), SWT.SAVE);

		// 拡張子
		String extension;
		
		// 出力形式がHTMLの場合
		if(fileHTMRdi.getSelection()) {
			
			// 拡張子設定
			extension = ".html";		
			String[] extensions = { "*.html", "*.*" };
			fileDialog.setFilterExtensions(extensions);
			
			// 「ファイルの種類」設定
			String[] filterNames = { Messages.getString("IssueClassifiedExportDialog.25"), Messages.getString("IssueClassifiedExportDialog.26") }; //$NON-NLS-1$ //$NON-NLS-2$
			fileDialog.setFilterNames(filterNames);
			
		}
		
		// 出力形式がXMLの場合
		else {
			
			// 拡張子設定
			extension = ".xml";
			String[] extensions = { "*.xml", "*.*" };
			fileDialog.setFilterExtensions(extensions);
			
			// 「ファイルの種類」設定
			String[] filterNames = { Messages.getString("IssueClassifiedExportDialog.27"), Messages.getString("IssueClassifiedExportDialog.28") }; //$NON-NLS-1$ //$NON-NLS-2$
			fileDialog.setFilterNames(filterNames);
			
		}
		
		// ファイル名のデフォルト設定
		if(issueListRdi.getSelection()) {			
			fileDialog.setFileName("issue" + extension);        // 課題票一覧			
		}
		else if(cntAssignedRdi.getSelection()) {			
			fileDialog.setFileName("assign" + extension);		// 担当者別課題票件数			
		}
		else if(cntStatusRdi.getSelection()) {			
			fileDialog.setFileName("status" + extension);		// ステータス別課題票件数
		}
		else {			
			fileDialog.setFileName("bug" + extension);			// バグ累計情報
		}
		
		
		// ファイル保存ダイアログを開き、ファイルパスを取得
		String fileName = fileDialog.open();
		
		// フォルダパス取得
		String filterPath = fileDialog.getFilterPath();

		// ファイルパスがnullの場合は処理を抜けます
		if(fileName == null) {
			return;
		}
		
		// ファイルに出力
		IssueClassifiedExport issueExport = new IssueClassifiedExport(this, issueList, fileName, filterPath);
		issueExport.exportRun();
		
		super.okPressed();
		
	}
	
	/**
	 * 出力条件に合う課題票リストを取得します。
	 * 種別がRの課題票リストをDBから取得し、条件に合わない課題票をリストから削除します。
	 * データベース接続エラーの場合はnullを返します。
	 * 
	 * @return			出力条件に合う課題票リスト
	 */
	private List addCondition() {
		
		// 課題票リスト(種別がRのみ)
		List allList = null;
		
		// 課題票一覧(種別がRのみ)を取得
		ModifyIssueFacade issueModifyFacade = new ModifyIssueFacade();
		try {
			allList = issueModifyFacade.getByType(IssueType.RELEASE_VALUE);
		} catch (BtsDBException e1) {
			return null;
		}
		
		// 出力条件に合う課題票リスト
		List newList = new ArrayList(allList);
		
		for(int i=0; i<allList.size(); i++) {
			
			// 課題票取得
			Issue issue = (Issue) allList.get(i);
			
			// 起票日テキストが両方とも空ではない場合
			if(!"".equals(frmDateTxt.getText().trim()) && !"".equals(toDateTxt.getText().trim())) {
				
				// 入力された起票日取得
				Date frmDate = BTSUtility.createDate(frmDateTxt.getText().trim());
				Date toDate = new Date((long)(BTSUtility.createDate(toDateTxt.getText().trim()).getTime() + 86400000L));

				// 範囲外の課題票をリストから削除
				if(issue.getCreateDate().before(frmDate) || issue.getCreateDate().after(toDate)) {
					newList.remove(issue);
				}
				
			}
			
			// 起票日(from)のみ入力時
			else if(!"".equals(frmDateTxt.getText().trim()) && "".equals(toDateTxt.getText().trim())) {
				
				// 入力された起票日取得
				Date frmDate = BTSUtility.createDate(frmDateTxt.getText().trim());
				
				// 範囲外の課題票をリストから削除
				if(issue.getCreateDate().before(frmDate)) {
					newList.remove(issue);
				}
				
			}
			
			// 起票日(to)のみ入力時
			else if("".equals(frmDateTxt.getText().trim()) && !"".equals(toDateTxt.getText().trim())) {
				
				// 入力された起票日取得
				Date toDate = new Date((long)(BTSUtility.createDate(toDateTxt.getText().trim()).getTime() + 86400000L));
				
				// 範囲外の課題票をリストから削除
				if(issue.getCreateDate().after(toDate)) {
					newList.remove(issue);
				}
				
			}
			
			for(int j=0; j<category.length; j++) {
				
				// 課題票のカテゴリチェックが外れている場合はリストから削除
				if (!categoryViewer.getChecked(category[j]) && category[j].equals(issue.getCategory())) {
					newList.remove(issue);
				}
				
			}
			
			// 「内緒のバグを含めない」がチェック、かつ内緒のバグの課題票はリストから削除
			if (secretChk.getSelection() && issue.getSecretBug().booleanValue()) {
				newList.remove(issue);
			}
			
			// 「保留中の課題票を含めない」がチェック、かつ保留中の課題票はリストから削除
			if (reserveChk.getSelection() && issue.getSuspend().booleanValue()) {
				newList.remove(issue);
			}
			
		}
		
		return newList;
		
	}
    

    /*
     * ＜参考＞
     * ブックマークビュー
     * 　org.eclipse.ui.ide
     *  　├org.eclipse.ui.views.markers.internal.DialogBookmarkFilter
     *  　└org.eclipse.ui.views.markers.internal.DialogMarkerFilter
     * 
     */
    
}
