/***********************************************************************
 * Copyright(C) 2006 Valtech Co.,Ltd.
 * All Rights Reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 ***********************************************************************/
package jp.valtech.bts.ui.navigator;

import java.awt.Color;
import java.awt.Font;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.math.BigDecimal;
import java.net.URL;
import java.sql.Timestamp;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;

import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.stream.StreamResult;
import javax.xml.transform.stream.StreamSource;

import jp.valtech.bts.dao.BtsDBException;
import jp.valtech.bts.data.CurrentProject;
import jp.valtech.bts.data.Issue;
import jp.valtech.bts.data.IssueHistory;
import jp.valtech.bts.data.IssueStatus;
import jp.valtech.bts.data.OutputAssigned;
import jp.valtech.bts.data.OutputBugChart;
import jp.valtech.bts.data.OutputIssues;
import jp.valtech.bts.data.OutputStatus;
import jp.valtech.bts.facade.IssueHistoryFacade;
import jp.valtech.bts.facade.ModifyIssueFacade;
import jp.valtech.bts.ui.BtsPlugin;
import jp.valtech.bts.util.BTSUtility;

import org.jfree.chart.ChartFactory;
import org.jfree.chart.ChartUtilities;
import org.jfree.chart.JFreeChart;
import org.jfree.chart.axis.DateAxis;
import org.jfree.chart.axis.DateTickUnit;
import org.jfree.chart.axis.NumberAxis;
import org.jfree.chart.axis.TickUnits;
import org.jfree.chart.plot.XYPlot;
import org.jfree.chart.title.LegendTitle;
import org.jfree.chart.title.TextTitle;
import org.jfree.data.time.Day;
import org.jfree.data.time.Month;
import org.jfree.data.time.RegularTimePeriod;
import org.jfree.data.time.TimeSeries;
import org.jfree.data.time.TimeSeriesCollection;
import org.jfree.data.time.Week;
import org.jfree.ui.HorizontalAlignment;
import org.jfree.ui.RectangleEdge;

/**
 * 帳票をファイルに出力します。
 * 
 * @author		<A href="mailto:s_imai@valtech.jp">S.Imai</A>
 * @version	Ver.0.8
 */
public class IssueClassifiedExport {
	
	/** 帳票出力ダイアログ */
	private IssueClassifiedExportDialog dialog;
	
	/** 出力対象の課題票リスト */
	private List issueList;
	
    /** 帳票出力先のファイルパス */
    private String fileName = null;
    
    /** 帳票出力先のフォルダパス */
    private String filterPath;
    
	/** 担当者一覧 */
	private String[] assigned;
	
	/** 最小日付タイムスタンプ */
	private Timestamp minTimestamp;
	
	/** 最大日付タイムスタンプ */
	private Timestamp maxTimestamp;
	
	/**  発生バグ累計カウント */
	private int[] updateOpenTotal;
	
	/**  解決バグ累計カウント */
	private int[] updateCloseTotal;

    
    /**
     * 初期設定を行います。
     * 
     * @param		dialog			課題票種類別出力ダイアログ
     * @param 		issueList		出力対象の課題票
     * @param 		fileName		出力先パス
     * @param 		filterPath		出力先フォルダパス
     */
	public IssueClassifiedExport(IssueClassifiedExportDialog dialog, List issueList, String fileName, String filterPath) {
		this.dialog = dialog;
		this.issueList = issueList;
		this.fileName = fileName;
		this.filterPath = filterPath;
	}
	
	
	/**
	 * 指定された種類の帳票を出力します。
	 */
	public void exportRun() {
		
		// 「課題票一覧」がチェックされている場合
		if(dialog.issueListRdi.getSelection()) {

			// OutputIssuesオブジェクトに値をセットし、指定された形式でファイルに出力
			outputFile(setIssues());
			
		}
		
		// 「担当者別課題票件数」がチェックされている場合
		else if(dialog.cntAssignedRdi.getSelection()) {

			// OutputAssignedオブジェクトに値をセットし、指定された形式でファイルに出力
			outputFile(setAssigned());
			
		}
		
		// 「ステータス別課題票件数」がチェックされている場合
		else if(dialog.cntStatusRdi.getSelection()) {
			
			// OutputStatusオブジェクトに値をセットし、指定された形式でファイルに出力
			outputFile(setStatus());
			
		}
		
		// 「バグ累計情報」がチェックされている場合
		else if(dialog.totalBugRdi.getSelection()) {
			
			// OutputBugChartオブジェクトに値をセットし、指定された形式でファイルに出力
			outputFile(setBugChart());
			
		}
		
	}
	
	/**
	 * オブジェクトを指定された形式でファイルに出力します。
	 * Transformer#transform(Source, Result)を用いる際に、
	 * StreamSourceとStreamResultの入出力元ストリームは明示的にクローズする必要があります。
	 * 
	 * @param		object		出力するオブジェクト
	 */
	private void outputFile(Object object) {
		
		try {
			
			// XMLファイルに出力する場合
			if (dialog.fileXMLRdi.getSelection()) {

				// オブジェクトをXMLファイルに出力
				BTSUtility.serialize(new File(fileName), object);

			}
			
			// HTMLファイルに出力する場合
			else if (dialog.fileHTMRdi.getSelection()) {
				
				// cssファイルを出力
				BtsPlugin.getInstance().copy("doc/output.css", filterPath + "/output.css");
				
				// オブジェクトをXMLファイルに出力
				BTSUtility.serialize(new File(filterPath + "/tmp.xml"), object);

				// TransformerFactoryインスタンスを取得
				TransformerFactory factory = TransformerFactory.newInstance();
				
				// XSLファイルのURL
				URL url = null;
				
				// 課題票一覧出力の場合
				if(dialog.issueListRdi.getSelection()) {
					
					// 完了期限切れアイコンをコピー
					BtsPlugin.getInstance().copy("icons/error1.gif", filterPath + "/error1.gif");			
					
					// 完了期限が迫っているアイコンをコピー
					BtsPlugin.getInstance().copy("icons/warning1.gif", filterPath + "/warning1.gif");
					
					url = BtsPlugin.getInstance().getURL("doc/outputIssues.xsl");
					
				}
				
				// 担当者別課題票件数出力の場合
				else if(dialog.cntAssignedRdi.getSelection()) {
					
					url = BtsPlugin.getInstance().getURL("doc/outputAssigned.xsl");
					
				}
				
				// ステータス別課題票件数出力の場合
				else if(dialog.cntStatusRdi.getSelection()) {
					
					url = BtsPlugin.getInstance().getURL("doc/outputStatus.xsl");
					
				}
				
				// バグ累計情報出力の場合
				else if(dialog.totalBugRdi.getSelection()) {
					
					url = BtsPlugin.getInstance().getURL("doc/outputBugChart.xsl");
					
				}

				// XSLファイルからtranceformerを取得
				InputStream inputStream = url.openStream();
				Transformer transformer = factory.newTransformer(new StreamSource(inputStream));
				
				// 出力するエンコーディングを設定
				transformer.setOutputProperty("encoding", "UTF-8");
				
				// XSLTを用いてXMLファイルをHTMLファイルに変換して出力
				InputStream in = new FileInputStream(filterPath + "/tmp.xml");
				OutputStream out = new FileOutputStream(fileName);
				transformer.transform(new StreamSource(in), new StreamResult(out));
				
				// クローズ
				inputStream.close();
				in.close();
				out.close();
				
			}
			
		} catch (Exception e) {
			BtsPlugin.getInstance().errorlog(e);
		}	
		
	}
	
	/**
	 * 課題票一覧出力用の情報をOutputIssuesオブジェクトにセットします。
	 * 
	 * @return		出力するオブジェクト配列
	 */
	private OutputIssues[] setIssues() {
		
		// OutputIssuesオブジェクトを格納する配列生成
		OutputIssues[] outputIssue = new OutputIssues[issueList.size()];
		
		// 現在開いているプロジェクト設定情報取得
		ProjectConfig config = CurrentProject.getInsance().getProjectConfig();
		
		// 完了期限警告残り日数
		String limit = null;
		
		// 完了期限が迫っているアイコンを表示する設定である場合
		if(config.isUntilDeadLineCheck()) {
			
			// 完了期限警告日数取得
			limit = config.getUntilDeadLine();

		}
		// 完了期限が迫っているアイコンを表示しない設定である場合
		else {
			
			limit = "-";
			
		}

		for(int i=0; i<issueList.size(); i++) {

			// 課題票取得
			Issue issue = (Issue)issueList.get(i);
			
			// 完了期限の種別を取得 (0:余裕がある　1:迫っている　2:切れている)
			int limitType = BTSUtility.judgeLimit(issue);
			

			// OutputIssueオブジェクトに現在の課題票の情報をセット
			outputIssue[i] = new OutputIssues();
			outputIssue[i].setDeadline(Integer.toString(limitType));									// 種別
			outputIssue[i].setIssueID(issue.getDisplayIssueID());										// 課題票ID
			outputIssue[i].setTitle(issue.getTitle());													// タイトル
			outputIssue[i].setCategory(issue.getCategory());											// カテゴリ
			outputIssue[i].setPriority(issue.getPriority());											// 優先度
			outputIssue[i].setStatusValue(issue.getStatus());											// ステータス(value)
			outputIssue[i].setStatus(IssueStatus.getEnumFromValue(issue.getStatus()).getDescription());	// ステータス
			outputIssue[i].setAssigned(issue.getAssigned());											// 担当者
			outputIssue[i].setCreateDate(BTSUtility.formatDate(issue.getCreateDate()));					// 起票日
			
			// OutputIssueオブジェクトにヘッダフッタ情報をセット
			outputIssue[i].setHeader(dialog.headerTxt.getText().trim());								// ヘッダ
			outputIssue[i].setFooter(dialog.footerTxt.getText().trim());								// フッタ	
			outputIssue[i].setNowDate(BTSUtility.formatDate(new Date()));								// 作成日時
			outputIssue[i].setLimit(limit);																// 完了期限までの残り日数
			outputIssue[i].setIssueNum(Integer.toString(issueList.size()));								// 件数
			
		}
		
		return outputIssue;
		
	}

	/**
	 * 担当者別課題票件数出力用の情報をOutputAssignedオブジェクトにセットします。
	 * 
	 * @return		出力するオブジェクト配列
	 */
	private OutputAssigned[] setAssigned() {
		
		// 担当者個人の「担当者割り当て(未対応)」カウンタ
		int assignedCnt = 0;
		// 課題票全体の「担当者割り当て(未対応)」カウンタ
		int assignedSum = 0;
		
		// 担当者個人の「対応済み」カウンタ
		int resolvedCnt = 0;
		// 課題票全体の「対応済み」カウンタ
		int resolvedSum = 0;
		
		// 担当者個人の「確認済み」カウンタ
		int verifiedCnt = 0;
		// 課題票全体の「確認済み」カウンタ
		int verifiedSum = 0;
		
		// 担当者個人の「完了」カウンタ
		int closedCnt = 0;
		// 課題票全体の「完了」カウンタ
		int closedSum = 0;
		
		// 担当者別課題票数カウンタ
		int sum = 0;
		// 全体の課題票数カウンタ
		int allSum = 0;
		
		
		// 担当者一覧(重複なし)取得
		try {
			
			CurrentProject project = CurrentProject.getInsance();
			if(project.isOpen()) {
				ModifyIssueFacade issueModifyFacade = new ModifyIssueFacade();
				assigned = issueModifyFacade.getAssignedTypeR();
			}
			
		} catch (BtsDBException e) {
			BtsPlugin.getInstance().errorlog(e);
		}
		
		// OutputAssignedオブジェクトを格納する配列生成
		OutputAssigned[] outputAssigned = new OutputAssigned[assigned.length + 1];
		
		// 担当者ごとに課題票件数をカウントします
		for(int i=0; i<assigned.length; i++) {
			
			// カウントクリア
			assignedCnt = 0;
			resolvedCnt = 0;
			verifiedCnt = 0;
			closedCnt = 0;
			sum = 0;		
			
			for(int j=0; j<issueList.size(); j++) {
				
				// 課題票取得
				Issue issue = (Issue) issueList.get(j);
				
				// ステータス取得
				String status = issue.getStatus();
				
				// 自分が担当している課題票の場合
				if(assigned[i].equals(issue.getAssigned())) {
					
					// ステータスが「担当者割り当て」の場合はカウント+1
					if(IssueStatus.ASSIGNED_VALUE.equals(status)) {
						assignedCnt++;
					}
					
					// ステータスが「対応済み」の場合はカウント+1
					else if(IssueStatus.RESOLVED_VALUE.equals(status)) {
						resolvedCnt++;
					}
					
					// ステータスが「確認済み」の場合はカウント+1
					else if(IssueStatus.VERIFIED_VALUE.equals(status)) {
						verifiedCnt++;
					}
					
					// ステータスが「完了」の場合はカウント+1
					else if(IssueStatus.CLOSED_VALUE.equals(status)) {
						closedCnt++;
					}
					
				}
				
			}
			
			// 担当者別課題票数を計算
			sum = assignedCnt + resolvedCnt + verifiedCnt + closedCnt;
			
			// OutputAssignedオブジェクトに担当者個人の課題票件数をセット
			outputAssigned[i+1] = new OutputAssigned();
			outputAssigned[i+1].setAssigned(assigned[i]);						// 担当者
			outputAssigned[i+1].setAssignedCnt(Integer.toString(assignedCnt));	// 未対応
			outputAssigned[i+1].setResolvedCnt(Integer.toString(resolvedCnt));	// 対応済み
			outputAssigned[i+1].setVerifiedCnt(Integer.toString(verifiedCnt));	// 確認済み
			outputAssigned[i+1].setClosedCnt(Integer.toString(closedCnt));		// 完了
			outputAssigned[i+1].setSum(Integer.toString(sum));					// 担当者別合計
			
			// OutputAssignedオブジェクトに担当者個人の未対応率をセット
			try{
				outputAssigned[i+1].setAssignedRate(rateCalculate(assignedCnt, sum));				
			} catch(Exception e) {
				outputAssigned[i+1].setAssignedRate("0.0");	
			}
			
			// 合計カウンタに個人のカウントを加算
			assignedSum += assignedCnt;
			resolvedSum += resolvedCnt;
			verifiedSum += verifiedCnt;
			closedSum += closedCnt;
			allSum += sum;

		}
		
		// OutputAssignedオブジェクトに全体の課題票件数をセット
		outputAssigned[0] = new OutputAssigned();
		outputAssigned[0].setAssignedSum(Integer.toString(assignedSum));				// 未対応合計
		outputAssigned[0].setResolvedSum(Integer.toString(resolvedSum));				// 対応済み合計
		outputAssigned[0].setVerifiedSum(Integer.toString(verifiedSum));				// 確認済み合計
		outputAssigned[0].setClosedSum(Integer.toString(closedSum));					// 完了合計
		outputAssigned[0].setAllSum(Integer.toString(allSum));							// 合計
		outputAssigned[0].setOpenedCnt(Integer.toString(issueList.size() - allSum));	// 担当者未割り当て数
		
		// OutputAssignedオブジェクトに未対応率をセット
		try{
			outputAssigned[0].setAssignedRateAvg(rateCalculate(assignedSum, allSum));
		} catch(Exception e) {
			outputAssigned[0].setAssignedRateAvg("0.0");
		}
		
		// OutputAssignedオブジェクトにヘッダフッタ情報をセット
		outputAssigned[0].setHeader(dialog.headerTxt.getText().trim());		// ヘッダ
		outputAssigned[0].setFooter(dialog.footerTxt.getText().trim());		// フッタ
		outputAssigned[0].setNowDate(BTSUtility.formatDate(new Date()));	// 作成日時
		
		return outputAssigned;
		
	}
	
	/**
	 * ステータス別課題票件数出力用の情報をOutputStatusオブジェクトにセットします。
	 * 
	 * @return		出力するオブジェクト
	 */
    private OutputStatus setStatus() {
    	
    	// 「起票」カウンタ
    	int openedCnt = 0;
    	// 完了期限切れ「起票」カウンタ
    	int openedOverCnt = 0;
    	
    	// 「担当者割り当て」カウンタ
    	int assignedCnt = 0;
    	// 完了期限切れ「担当者割り当て」カウンタ
    	int assignedOverCnt = 0;
    	
    	// 「対応済み」カウンタ
    	int resolvedCnt = 0;
    	// 完了期限切れ「対応済み」カウンタ
    	int resolvedOverCnt = 0;
    	
    	// 「確認済み」カウンタ
    	int verifiedCnt = 0;
    	// 完了期限切れ「確認済み」カウンタ
    	int verifiedOverCnt = 0;
    	
    	// 「完了」カウンタ
    	int closedCnt = 0;
    
    	
    	for(int i=0; i<issueList.size(); i++) {
    		
    		// 課題票取得
    		Issue issue = (Issue) issueList.get(i);
    		
    		// 完了期限が切れているか判定
    		boolean judge = deadLineOver(issue);
    		
    		// ステータス取得
    		String status = issue.getStatus();
    		
    		// ステータスが「起票」の場合
    		if(IssueStatus.OPENED_VALUE.equals(status)) {
    			
    			// 「起票」カウント+1
    			openedCnt++;
    			
    			// 完了期限が切れている場合
    			if(judge) {
    				openedOverCnt++;
    			}
    			
    		}
    		
    		// ステータスが「担当者割り当て」の場合
    		else if(IssueStatus.ASSIGNED_VALUE.equals(status)) {
    			
    			// 「担当者割り当て」カウント+1
    			assignedCnt++;
    			
    			// 完了期限が切れている場合
    			if(judge) {
    				assignedOverCnt++;
    			}
    			
    		}
    		
    		// ステータスが「対応済み」の場合
    		else if(IssueStatus.RESOLVED_VALUE.equals(status)) {
    			
    			// 「対応済み」カウント+1
    			resolvedCnt++;
    			
    			// 完了期限が切れている場合
    			if(judge) {
    				resolvedOverCnt++;
    			}
    			
    		}
    		
    		// ステータスが「確認済み」の場合
    		else if(IssueStatus.VERIFIED_VALUE.equals(status)) {
    			
    			// 「確認済み」カウント+1
    			verifiedCnt++;
    			
    			// 完了期限が切れている場合
    			if(judge) {
    				verifiedOverCnt++;
    			}
    			
    		}
    		
    		// ステータスが「完了」の場合
    		else {
    			
    			// 「完了」カウント+1
    			closedCnt++;
    			
    		}
    		
    	}
    	
    	// 未完了の課題票件数を計算
    	int notClosedSum = openedCnt + assignedCnt + resolvedCnt + verifiedCnt;
    	
    	// 完了期限切れの未完了課題票の件数を計算
    	int notClosedOverSum = openedOverCnt + assignedOverCnt + resolvedOverCnt + verifiedOverCnt;
    	
    	// 合計件数を計算
    	int statusSum = notClosedSum + closedCnt;
    	
    	
		// OutputStatusオブジェクトにステータス別課題票件数をセット
    	OutputStatus outputStatus = new OutputStatus();
    	outputStatus.setOpenedCnt(Integer.toString(openedCnt));					// 起票カウント
    	outputStatus.setOpenedOverCnt(Integer.toString(openedOverCnt));			// 期限切れ起票カウント
    	outputStatus.setAssignedCnt(Integer.toString(assignedCnt));				// 担当者割り当てカウント
    	outputStatus.setAssignedOverCnt(Integer.toString(assignedOverCnt));		// 期限切れ担当者割り当てカウント
    	outputStatus.setResolvedCnt(Integer.toString(resolvedCnt));				// 対応済みカウント
    	outputStatus.setResolvedOverCnt(Integer.toString(resolvedOverCnt));		// 期限切れ対応済みカウント
    	outputStatus.setVerifiedCnt(Integer.toString(verifiedCnt));				// 確認済みカウント
    	outputStatus.setVerifiedOverCnt(Integer.toString(verifiedOverCnt));		// 期限切れ確認済みカウント
    	outputStatus.setClosedCnt(Integer.toString(closedCnt));					// 完了カウント
    	
    	// OutputStatusオブジェクトに合計件数をセット
    	outputStatus.setNotClosedSum(Integer.toString(notClosedSum));			// 未完了合計
    	outputStatus.setNotClosedOverSum(Integer.toString(notClosedOverSum));	// 期限切れ未完了合計
    	outputStatus.setStatusSum(Integer.toString(statusSum));					// ステータス合計
    	
    	// OutputStatusオブジェクトにヘッダフッタ情報をセット
    	outputStatus.setHeader(dialog.headerTxt.getText().trim());				// ヘッダ
    	outputStatus.setFooter(dialog.footerTxt.getText().trim());				// フッタ
    	outputStatus.setNowDate(BTSUtility.formatDate(new Date()));				// 作成日時
    	
    	return outputStatus;
    	
    }
    
	/**
	 * バグ累計出力用の情報をOutputBugChartオブジェクトにセットします。
	 * 
	 * @return		出力するオブジェクト配列
	 */
    private OutputBugChart[] setBugChart() {
    	
		// 発生バグと解決バグをカウント
		countBug();
    	
		// カウントをもとにグラフをPNGで出力
		createChart();
		
		// OutputBugChartオブジェクトを格納する配列生成
		OutputBugChart[] outputBugCharts = new OutputBugChart[updateOpenTotal.length];
		
		// 最小日付カレンダー生成
    	Calendar calendar = Calendar.getInstance();
    	calendar.setTimeInMillis(minTimestamp.getTime());

		for (int i = 0; i < updateOpenTotal.length; i++) {
			
			// OutputBugChartオブジェクトにバグ累計カウントをセット
			outputBugCharts[i] = new OutputBugChart();
			outputBugCharts[i].setOpenedBug(Integer.toString(updateOpenTotal[i]));							// 発生バグ
			outputBugCharts[i].setClosedBug(Integer.toString(updateCloseTotal[i]));							// 解決バグ
			outputBugCharts[i].setRemainBug(Integer.toString(updateOpenTotal[i] - updateCloseTotal[i]));	// 残存バグ
			
			// OutputBugChartオブジェクトにヘッダフッタ情報をセット
			outputBugCharts[i].setHeader(dialog.headerTxt.getText().trim());								// ヘッダ
			outputBugCharts[i].setFooter(dialog.footerTxt.getText().trim());								// フッタ
			outputBugCharts[i].setNowDate(BTSUtility.formatDate(new Date()));								// 作成日時
			
			// 月毎表示の場合
			if(dialog.unitMonthRdi.getSelection()) {
				
				outputBugCharts[i].setDate(BTSUtility.formatDate(calendar.getTime(), "yyyy/M"));			// 日付
				outputBugCharts[i].setUnit(Messages.getString("IssueClassifiedExport.6"));													// 出力単位 //$NON-NLS-1$
				
				// 月数+1
				calendar.add(Calendar.MONTH, 1);
				
			}
			// 週毎表示の場合
			else if(dialog.unitWeekRdi.getSelection()) {
				
				// カレンダを週の最初の日付(日曜日)に設定
				calendar.set(Calendar.DAY_OF_WEEK, 1);
				
				outputBugCharts[i].setDate(BTSUtility.formatDate(calendar.getTime(), "M/d"));				// 日付
				outputBugCharts[i].setUnit(Messages.getString("IssueClassifiedExport.7"));													// 出力単位 //$NON-NLS-1$

				// 週数+1
				calendar.add(Calendar.WEEK_OF_YEAR, 1);
				
			}
			// 日毎表示の場合
			else {
				
				outputBugCharts[i].setDate(BTSUtility.formatDate(calendar.getTime(), "M/d"));				// 日付
				outputBugCharts[i].setUnit(Messages.getString("IssueClassifiedExport.8"));													// 出力単位 //$NON-NLS-1$

				// 日数+1
				calendar.add(Calendar.DAY_OF_MONTH, 1);
				
			}
			
		}

		return outputBugCharts;
    	
    }
    
    
	/**
	 * 発生バグと解決バグをカウントします。
	 *
	 * @param		issueList		課題票リスト
	 */
	private void countBug() {
		
		// 最小日付の初期値を設定
		minTimestamp = new Timestamp(9999999999999L);
		
		// 最大日付の初期値を設定
		maxTimestamp = new Timestamp(0L);
		
		// グラフを生成するための更新履歴リスト取得
		List bugChartList = getIssueHistoryForChart();
		
		// 最小日付をカレンダにセット
    	Calendar calendarMin = Calendar.getInstance();
    	calendarMin.setTimeInMillis(minTimestamp.getTime());
    	
    	// 最大日付をカレンダにセット
    	Calendar calendarMax = Calendar.getInstance();
    	calendarMax.setTimeInMillis(maxTimestamp.getTime());
 
    	
    	// 最小日付の年を取得
    	int minYear = calendarMin.get(Calendar.YEAR);
    	// 最大日付の年を取得
    	int maxYear = calendarMax.get(Calendar.YEAR);
    	
    	// 最小日付の月
    	int minMonth = 0;
    	// 最大日付の月
    	int maxMonth = 0;
    	
    	// 最小日付の週
    	int minWeek = 0;
    	// 最大日付の週
    	int maxWeek = 0;
    	
    	// 最小日付の日
    	int minDay = 0;
    	// 最大日付の日
    	int maxDay = 0;
    	
    	
		// バグ累計カウント用配列の要素数
		int argsNumMax;
			
		// 月毎表示の場合
		if(dialog.unitMonthRdi.getSelection()) {
			
	    	// 月の最小値と最大値を取得
	    	minMonth = calendarMin.get(Calendar.MONTH) +1;
	    	maxMonth = calendarMax.get(Calendar.MONTH) +1;
	    	
	    	// バグ累計カウント用の配列要素数を計算
	    	argsNumMax = (maxYear - minYear) * 12 + maxMonth - minMonth + 1;
			
		}
		// 週毎表示の場合
		else if(dialog.unitWeekRdi.getSelection()) {
			
			// 週の最小値と最大値を取得
	    	minWeek = calendarMin.get(Calendar.WEEK_OF_YEAR);
	    	maxWeek = calendarMax.get(Calendar.WEEK_OF_YEAR);

	    	// バグ累計カウント用の配列要素数を計算
	    	argsNumMax = (maxYear - minYear) * 52 + maxWeek - minWeek + 1;
			
		}
		// 日毎表示の場合
		else {
			
			// 日の最小値と最大値を取得
	    	minDay = calendarMin.get(Calendar.DAY_OF_YEAR);
	    	maxDay = calendarMax.get(Calendar.DAY_OF_YEAR);
	    	
	    	// バグ累計カウント用の配列要素数を計算
	    	argsNumMax = (maxYear - minYear) * 365 + maxDay - minDay + 1;

		}

		// 発生バグ累計カウント用の配列生成
		updateOpenTotal = new int[argsNumMax];
		
		// 解決バグ累計カウント用の配列生成
		updateCloseTotal = new int[argsNumMax];
		
		// カレンダインスタンス生成
		Calendar calendar = Calendar.getInstance();
		
		for(int i=0; i<bugChartList.size(); i++) {
			
			// 更新タイムスタンプ（long）取得
			long updateDateLong = ((IssueHistory)bugChartList.get(i)).getUpdateDateTimestamp();
			
			// カレンダに更新タイムスタンプをセット
			calendar.setTimeInMillis(updateDateLong);
			
			// 配列の要素番号
			int argsNum;
			
			// 更新日の年を取得
			int updateYear = calendar.get(Calendar.YEAR);
			
			// 月毎表示の場合
			if(dialog.unitMonthRdi.getSelection()) {
				
				// 配列の要素番号を計算
				argsNum = (updateYear - minYear) * 12 + calendar.get(Calendar.MONTH)+1 - minMonth;
				
			}
			// 週毎表示の場合
			else if(dialog.unitWeekRdi.getSelection()) {
				
				// 配列の要素番号を計算
				argsNum = (updateYear - minYear) * 52 + calendar.get(Calendar.WEEK_OF_YEAR) - minWeek;
				
			}
			// 日毎表示の場合
			else {
				
				// 配列の要素番号を計算
				argsNum = (updateYear - minYear) * 365 + calendar.get(Calendar.DAY_OF_YEAR) - minDay;
				
			}
			
			// 「起票」の場合
			if(IssueHistory.ISSUE_INITIAL_REGIST.equals(((IssueHistory)bugChartList.get(i)).getUpdateAttribute())){
				
				// 発生バグカウント＋１
				while(argsNum < updateOpenTotal.length) {
					updateOpenTotal[argsNum] += 1;
					argsNum++;
				}

			}
			// 「完了（閉じる）」の場合
			else {
				
				// 解決バグカウント＋１
				while(argsNum < updateOpenTotal.length) {
					updateCloseTotal[argsNum] += 1;
					argsNum++;
				}
				
			}
			
		}
		
	}
	
	/**
	 * カウントをもとにグラフをPNGで出力します。
	 */
	private void createChart() {
	
		// 時間軸クラス
		Class klass;
		
		// 月毎表示の場合
		if(dialog.unitMonthRdi.getSelection()) {
			klass = Month.class;
		}
		// 週毎表示の場合
		else if(dialog.unitWeekRdi.getSelection()) {
			klass = Week.class;
		}
		// 日毎表示の場合
		else {
			klass = Day.class;
		}
		
		// タイムシリーズ生成
		TimeSeries openTimeSeries = new TimeSeries(Messages.getString("IssueClassifiedExport.0"), klass); //$NON-NLS-1$
		TimeSeries closeTimeSeries = new TimeSeries(Messages.getString("IssueClassifiedExport.1"), klass); //$NON-NLS-1$
		TimeSeries remainderTimeSeries = new TimeSeries(Messages.getString("IssueClassifiedExport.2"), klass); //$NON-NLS-1$
		
		// カレンダに最小日付をセット
    	Calendar calendar = Calendar.getInstance();
    	calendar.setTimeInMillis(minTimestamp.getTime());
    	
		for(int i=0; i<updateOpenTotal.length; i++) {
			
			// タイムピリオド
			RegularTimePeriod timePeriod;

			// 月毎表示の場合
			if(dialog.unitMonthRdi.getSelection()) {
				
				// 月タイムピリオド生成
				timePeriod = new Month(calendar.get(Calendar.MONTH)+1, calendar.get(Calendar.YEAR));
				
				// 月数+1
				calendar.add(Calendar.MONTH, 1);
				
			}
			// 週毎表示の場合
			else if(dialog.unitWeekRdi.getSelection()) {

				// 週タイムピリオド生成
				timePeriod = new Week(calendar.get(Calendar.WEEK_OF_YEAR), calendar.get(Calendar.YEAR));

				// 週数+1
				calendar.add(Calendar.WEEK_OF_YEAR, 1);
				
			}
			// 日毎表示の場合
			else {
				
				// 日タイムピリオド生成
				timePeriod = new Day(calendar.get(Calendar.DAY_OF_MONTH), calendar.get(Calendar.MONTH)+1, calendar.get(Calendar.YEAR));
				
				// 日数+1
				calendar.add(Calendar.DAY_OF_MONTH, 1);
				
			}
			
			// 発生バグ累計に値をセット
			openTimeSeries.add(timePeriod, updateOpenTotal[i]);

			// 解決バグ累計に値をセット
			closeTimeSeries.add(timePeriod, updateCloseTotal[i]);

			// 残存バグ累計に値をセット
			remainderTimeSeries.add(timePeriod, updateOpenTotal[i] - updateCloseTotal[i]);

		}
		
		// データセットに値をセット
		TimeSeriesCollection dataset = new TimeSeriesCollection();
		dataset.addSeries(openTimeSeries);
		dataset.addSeries(closeTimeSeries);
		dataset.addSeries(remainderTimeSeries);

		// JFreeChartオブジェクトの生成
		JFreeChart chart = ChartFactory.createTimeSeriesChart(null, null, Messages.getString("IssueClassifiedExport.3"), dataset, true, true, false); //$NON-NLS-1$
		
		// グラフ外枠の背景色を設定(#EFEEFE)
		chart.setBackgroundPaint(new Color(239, 238, 254));
		
		// グラフタイトルを設定
		TextTitle textTitle = new TextTitle(Messages.getString("IssueClassifiedExport.4")); //$NON-NLS-1$
		textTitle.setFont(new Font("SansSerif", 1, 16));
		textTitle.setHorizontalAlignment(HorizontalAlignment.CENTER);
		chart.addSubtitle(textTitle);
		
		// 横軸の目盛り線を非表示
		XYPlot xyplot = (XYPlot) chart.getPlot();
		xyplot.setDomainGridlinesVisible(false);

		// 日付設定取得
		DateAxis dateaxis = (DateAxis) xyplot.getDomainAxis();

		TickUnits tickUnits = new TickUnits();
		
		// 1週間単位の場合
		if(dialog.unitWeekRdi.getSelection()) {
			
			// 横軸の表示設定を追加
			tickUnits.add(new DateTickUnit(DateTickUnit.DAY, 7, new SimpleDateFormat("M/d")));
			
		}
		
		// 1日単位の場合
		else if(dialog.unitDayRdi.getSelection()) {
			
			// 横軸の表示設定を追加
			tickUnits.add(new DateTickUnit(DateTickUnit.DAY, 1, new SimpleDateFormat("M/d")));
			tickUnits.add(new DateTickUnit(DateTickUnit.DAY, 7, new SimpleDateFormat("M/d")));

		}
		
		// 横軸の表示設定を追加
		tickUnits.add(new DateTickUnit(DateTickUnit.MONTH, 1, new SimpleDateFormat("yyyy/M")));
		tickUnits.add(new DateTickUnit(DateTickUnit.MONTH, 3, new SimpleDateFormat("yyyy/M")));
		tickUnits.add(new DateTickUnit(DateTickUnit.MONTH, 6, new SimpleDateFormat("yyyy/M")));
		
		// 表示設定をセット
		dateaxis.setStandardTickUnits(tickUnits);
		
		// 表示設定を自動的に選択する
		dateaxis.setAutoTickUnitSelection(true);
		
		// 縦軸を整数単位で表示
		NumberAxis axis = (NumberAxis) xyplot.getRangeAxis();
		axis.setStandardTickUnits(NumberAxis.createIntegerTickUnits());
		
		// カテゴリを右端に設定
		LegendTitle legendtitle = (LegendTitle) chart.getSubtitle(0);
		legendtitle.setPosition(RectangleEdge.RIGHT);
		
		// ファイル生成
		File outputFile = new File(filterPath + "/bug.png");
		try {
			// PNGで出力
			ChartUtilities.saveChartAsPNG(outputFile, chart, 1000, 400);
		} catch (IOException e) {
			BtsPlugin.getInstance().errorlog(e);
		}
		
	}
	
	/**
	 * 「起票」または「完了(閉じる)」の更新履歴リストを取得します。
	 * また、更新日の最大と最小を求めます。
	 * 
	 * @return		「起票」または「完了(閉じる)」の更新履歴リスト
	 */
	private List getIssueHistoryForChart() {	

		// リスト生成
		List issueHistorys = new ArrayList();

		IssueHistoryFacade facade = new IssueHistoryFacade();

		for(int i=0; i<issueList.size(); i++) {
			
			// フィルタを通過した課題票を取得
			Issue issue = ((Issue)issueList.get(i));
			
			// フィルタを通過した課題票の更新履歴を取得
			IssueHistory[] filterIssueHistories 
				= facade.getByFingerPrint(issue.getFingerPrint(), issue.getType());

			for(int j=0; j<filterIssueHistories.length; j++) {
				
				// 更新した属性名取得
				String updateAttribute = filterIssueHistories[j].getUpdateAttribute();

				// 更新後の属性取得
				String updateAfter = filterIssueHistories[j].getUpdateAfter();
				
				// 「起票」または「完了(閉じる)」の更新履歴を抽出
				if (IssueHistory.ISSUE_INITIAL_REGIST.equals(updateAttribute) || 
						(Messages.getString("IssueClassifiedExport.5").equals(updateAttribute) && 
								IssueStatus.CLOSED.getDescription().equals(updateAfter))) { //$NON-NLS-1$

					// リストに追加
					issueHistorys.add(filterIssueHistories[j]);

					// 更新タイムスタンプ取得
					Timestamp updateDateTimestamp 
							= new Timestamp(filterIssueHistories[j].getUpdateDateTimestamp());

					// 最小日付タイムスタンプ更新
					if (minTimestamp.compareTo(updateDateTimestamp) > 0) {
						minTimestamp = updateDateTimestamp;
					}

					// 最大日付タイムスタンプ更新
					if (maxTimestamp.compareTo(updateDateTimestamp) < 0) {
						maxTimestamp = updateDateTimestamp;
					}

				}
				
			}
			
		}
		
		facade.close();
		return issueHistorys;
	}
	
	/**
	 * 未対応率を計算します。
	 * 計算結果は小数点以下第２位を四捨五入します。
	 * 
	 * @param		x		未対応数
	 * @param		y		合計
	 * @return				未対応率(String)
	 * @throws		Exception
	 */
	private String rateCalculate(int x, int y) throws Exception {
		
		// 未対応率計算
		double rate = (double)x * 100 / (double)y;
		
		// BigDecimal生成
		BigDecimal bd = new BigDecimal(String.valueOf(rate));
		
		// 未対応率を小数点以下第２位で四捨五入
		rate = bd.setScale(1, BigDecimal.ROUND_HALF_UP).doubleValue();
		
		// Stringに変換して返します
		return Double.toString(rate);
		
	}
    
    /**
     * 指定された課題票の完了期限が切れているかどうか判定します。
     * 期限が切れている場合はtrueを返します。
     * 
     * @param		issue		課題票
     * @return		判定
     */
    private boolean deadLineOver(Issue issue) {
    	
		// 現在の日付タイムスタンプ(long)取得
		long nowDate = new Date().getTime();

		// 現在の日付をカレンダにセット
		Calendar calendarNow = Calendar.getInstance();
		calendarNow.setTimeInMillis(nowDate);
		
		// 完了期限が設定されている場合
		if(issue.getDeadline() != null) {
			
			// 完了期限タイムスタンプ(long)取得
			long deadLine = issue.getDeadlineTimestamp();
			
			// 完了期限をカレンダにセット
			Calendar calendarDead = Calendar.getInstance();
			calendarDead.setTimeInMillis(deadLine);
			
			// 完了期限が切れている場合
			if(calendarNow.get(Calendar.DAY_OF_YEAR) > calendarDead.get(Calendar.DAY_OF_YEAR)) {
				return true;
			}
			
		}
		
		// 完了期限が切れていない場合（完了期限が設定されていない場合含む）
		return false;
    	
    }
}
