/***********************************************************************
 * Copyright(C) 2006 Valtech Co.,Ltd.
 * All Rights Reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 ***********************************************************************/
package jp.valtech.bts.ui.msglist;

import jp.valtech.bts.data.MessagePacket;

import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerSorter;



/**
 * {@link MessageListView メッセージ一覧テーブル}のソータクラスです。
 * 
 * @author		<A href="mailto:s_imai@valtech.jp">S.Imai</A>
 * @version	Ver.0.8
 */
public class MessageListViewSorter extends ViewerSorter {

	/** ソート反転フラグ */
	private boolean reversed = false;
	
	/** カラム番号 */
	private int columnNumber;
	
	/**
	 * コンストラクタ
	 * @param		columnNumber		カラム番号
	 */
	public MessageListViewSorter(int columnNumber){
		this.columnNumber = columnNumber;
	}
	
	/*
	 *  (非 Javadoc)
	 * @see org.eclipse.jface.viewers.ViewerSorter#compare(org.eclipse.jface.viewers.Viewer, java.lang.Object, java.lang.Object)
	 */
	public int compare(Viewer viewer, Object e1, Object e2) {
		
		// 比較結果
		int result = 0;
		
		// MessagePacketオブジェクトの比較を行う
		result = compareColumnValue((MessagePacket)e1, (MessagePacket)e2);
		
		// ソート反転処理
		if (reversed){
			result = -result;
		}
		
		return result;
	}
	
	/**
	 * 各カラムごとにデータの比較を行います。
	 * @param		e1 		最初のMessagePacketオブジェクト
	 * @param		e2 		次のMessagePacketオブジェクト
	 * @return		比較結果
	 */
	int compareColumnValue(MessagePacket e1, MessagePacket e2) {
		
		switch (columnNumber) {
		
			// 1カラム目：メッセージ種別
			case 0 :
				return compareTo(e1.getMessageType(), e2.getMessageType());
				
			// 2カラム目：メッセージ
			case 1 :
				return compareTo(e1.getMessage(), e2.getMessage());
				
			// 3カラム目：送信者
			case 2 :
				return compareTo(e1.getFromUser(), e2.getFromUser());
				
			// 4カラム目：送信日時
			case 3 :
				return compareTo(e1.getSent(), e2.getSent());
				
			default:
				return 0;
			
		}
		
	}
	
	
	/**
	 * 指定された２つのオブジェクトの順序を比較します。
	 * 比較対照のオブジェクトはかならず{@link Comparable}を実装している必要があります。
	 * 
	 * @param		obj1			比較対象のオブジェクト
	 * @param		obj2			比較対象のオブジェクト
	 * @return		負の数：obj1の方が小さい。
	 * 				ゼロ　：obj1,obj2両方等しい。
	 * 				正の数：obj2の方が小さい。
	 */
	private int compareTo(Object obj1, Object obj2) {
		if(obj1 == null && obj2 == null) {
			return 0;
		} else if(obj1 != null && obj2 == null) {
			return 1;
		} else if(obj1 == null && obj2 != null) {
			return -1;
		} else {
			return ((Comparable)obj1).compareTo((Comparable)obj2);
		}
	}

	/**
	 * ソート反転フラグを返します。
	 * @return		ソート反転フラグ
	 */
	public boolean isReversed() {
		return reversed;
	}

	/**
	 * ソート反転フラグを設定します。
	 * @param		reversed		ソート反転フラグ
	 */
	public void setReversed(boolean reversed) {
		this.reversed = reversed;
	}

	/**
	 * カラム番号を返します。
	 * @return		カラム番号
	 */
	public int getColumnNumber() {
		return columnNumber;
	}
	
}
