/***********************************************************************
 * Copyright(C) 2006 Valtech Co.,Ltd.
 * All Rights Reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 ***********************************************************************/
package jp.valtech.bts.ui.msglist;

import java.text.SimpleDateFormat;

import jp.valtech.bts.data.MessagePacket;
import jp.valtech.bts.data.MessageType;
import jp.valtech.bts.ui.BtsPlugin;

import org.eclipse.jface.viewers.IColorProvider;
import org.eclipse.jface.viewers.IFontProvider;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.graphics.FontData;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Display;

/**
 * {@link MessageListView メッセージ一覧ビュー}のプロバイダクラスです。
 * 
 * @author		<A href="mailto:m_sugitou@valtech.jp">M.Sugito</A>
 * @version	Ver.0.8
 */
public class MessageListViewLabelProvider extends LabelProvider implements ITableLabelProvider, IColorProvider, IFontProvider {

	/** 表示で使う色：赤 */
	private static final Color COLOR_RED = Display.getDefault().getSystemColor(SWT.COLOR_RED);
	/** 表示で使う色：青 */
	private static final Color COLOR_BLUE = new Color(Display.getCurrent(), 0, 0, 204);
	/** フォント：ボールド */
	private static final Font FONT_BOLD = new Font(Display.getDefault(), new FontData("", 9, SWT.BOLD));
	
	/** アイコン：課題票送受信（未読） */
	private static final Image IMG_ISSUE		= BtsPlugin.getInstance().getImage("msg_issue.gif");
	/** アイコン：課題票送受信（既読） */
	private static final Image IMG_ISSUE_OPEN	= BtsPlugin.getInstance().getImage("msg_issue_opn.gif");
	/** アイコン：ユーザメッセージ送受信（未読・添付無し） */
	private static final Image IMG_USER			= BtsPlugin.getInstance().getImage("msg_user.gif");
	/** アイコン：ユーザメッセージ送受信（既読・添付無し） */
	private static final Image IMG_USER_OPEN	= BtsPlugin.getInstance().getImage("msg_user_opn.gif");
	/** アイコン：ユーザメッセージ送受信（未読・添付有り） */
	private static final Image IMG_USER_ATACH	= BtsPlugin.getInstance().getImage("msg_user_atch.gif");
	/** アイコン：ユーザメッセージ送受信（未読・添付無し） */
	private static final Image IMG_USER_ATACHOP	= BtsPlugin.getInstance().getImage("msg_user_atch_opn.gif");
	/** アイコン：同期メッセージ */
	private static final Image IMG_SYNC			= BtsPlugin.getInstance().getImage("sync1.gif");
	
	/** フォーマッタ */
	private static final SimpleDateFormat FORMATTER = new SimpleDateFormat("yyyy/MM/dd HH:mm");
	/** 改行文字 */
	private static final String RETURN_STR = "\n";
	
	
	/*
	 *  (非 Javadoc)
	 * @see org.eclipse.jface.viewers.ITableLabelProvider#getColumnText(java.lang.Object, int)
	 */
	public String getColumnText(Object obj, int index) {
		// キャストする。
		MessagePacket messagePacket = (MessagePacket)obj;
		 
		// カラムにより処理を沸ける 
		switch (index) {
		
			// 2カラム目：メッセージ
			case 1:
				// メッセージ取得
				String msg = messagePacket.getMessage();
				
				// 一行目だけを取得して返す。
				if(msg != null && msg.indexOf(RETURN_STR) != -1) {
					return msg.substring(0, msg.indexOf(RETURN_STR));
				}
				return msg;
				
			// 3カラム目：送信者
			case 2:
				return messagePacket.getFromUser();
			
			// 4カラム目：送信日時
			case 3:
				return FORMATTER.format(messagePacket.getSent());
		}
		return "";
	}
	
	/*
	 *  (非 Javadoc)
	 * @see org.eclipse.jface.viewers.ITableLabelProvider#getColumnImage(java.lang.Object, int)
	 */
	public Image getColumnImage(Object obj, int index) {

		// キャストする。
		MessagePacket msg = (MessagePacket)obj;

		// ０カラム目のみアイコン表示
		switch (index) {
		
			// 1カラム目：メッセージの種類によりアイコンを分ける
			case 0:
				if( MessageType.SYNC_VALUE.equals(msg.getMessageType()) ) {
					
					// 同期メッセージのアイコン
					return IMG_SYNC;
					
				} else if( MessageType.CONFLICT_VALUE.equals(msg.getMessageType()) ) {
					// 課題票送受信のアイコン。
					if(!msg.isOpend().booleanValue()) {
						return IMG_ISSUE;
					}
					return IMG_ISSUE_OPEN;

				} else if( MessageType.ISSUE_VALUE.equals(msg.getMessageType()) ) {
					// 課題票送受信のアイコン。
					if(!msg.isOpend().booleanValue()) {
						return IMG_ISSUE;
					}
					return IMG_ISSUE_OPEN;
				} else if( MessageType.USER_VALUE.equals(msg.getMessageType()) ) {

					// ユーザ送信のアイコン
					if( msg.getAttachFileName() ==null || "".equals(msg.getAttachFileName())) {
						// 添付無しのアイコン
						if(!msg.isOpend().booleanValue()) {
							return IMG_USER;
						}
						return IMG_USER_OPEN;
					
					} else {
						// 添付有りのアイコン
						if(!msg.isOpend().booleanValue()) {
							return IMG_USER_ATACH;
						}
						return IMG_USER_ATACHOP;
					}
				}
		}
		return null;
	}

	
	/* (非 Javadoc)
	 * @see org.eclipse.jface.viewers.IColorProvider#getForeground(java.lang.Object)
	 */
	public Color getForeground(Object element) {

		// キャストする。
		MessagePacket messagePacket = (MessagePacket)element;
		
		// 競合メッセージの場合は赤表示にする
		if( MessageType.CONFLICT_VALUE.equals(messagePacket.getMessageType()) ) {
			return COLOR_RED;
		} else if( MessageType.USER_VALUE.equals(messagePacket.getMessageType()) ) {
			return COLOR_BLUE;
		}
		return null;
	}

	
	/* (非 Javadoc)
	 * @see org.eclipse.jface.viewers.IColorProvider#getBackground(java.lang.Object)
	 */
	public Color getBackground(Object element) {
		return null;
	}

	
	/* (非 Javadoc)
	 * @see org.eclipse.jface.viewers.IFontProvider#getFont(java.lang.Object)
	 */
	public Font getFont(Object element) {
		
		// キャストする。
		MessagePacket messagePacket = (MessagePacket)element;
		
		// メッセージ未読は太字フォントで表示
		if(!messagePacket.getOpened().booleanValue()) {
			return FONT_BOLD;
		}
		return null;
	}
}


