/***********************************************************************
 * Copyright(C) 2006 Valtech Co.,Ltd.
 * All Rights Reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 ***********************************************************************/
package jp.valtech.bts.ui.msglist;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;

import jp.valtech.bts.data.MessagePacket;
import jp.valtech.bts.data.MessageType;
import jp.valtech.bts.ui.BtsPlugin;

import org.eclipse.jface.dialogs.IDialogSettings;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerFilter;

/**
 * {@link MessageListView メッセージ一覧テーブル}のフィルタークラスです。
 * 
 * @author		<A href="mailto:s_imai@valtech.jp">S.Imai</A>
 * @version	Ver.0.8
 */
public class MessageListViewFilter {

	/** メッセージ一覧テーブルのビューアインスタンス */
	private TableViewer viewer;
	
	/** 現在のフィルター情報 */
	IDialogSettings section
		= BtsPlugin.getInstance().getDialogSettings().getSection(MessageFilterManager.SECTION_NAME);

	
	public MessageListViewFilter(TableViewer viewer) {
		this.viewer = viewer;
	}
	
	
	/**
	 * メッセージフィルタを追加します。
	 */
	public void addMessageFilters() {

		
		// 最初にフィルタをリセットする
		viewer.resetFilters();
		
		// 「課題票の更新通知メッセージを表示する」がチェックされていない場合
		if (!section.getBoolean(MessageFilterManager.KEY_SYSTEM)) {
			// システムメッセージフィルタを実行
			viewer.addFilter(new SystemFilter());
		}

		// 「任意に送受信したメッセージを表示する」がチェックされていない場合
		if (!section.getBoolean(MessageFilterManager.KEY_USER)) {
			// ユーザメッセージフィルタを実行
			viewer.addFilter(new UserFilter());
		}
		
		// メッセージテキストが空ではない場合
		if (!"".equals(section.get(MessageFilterManager.KEY_MESSAGE))) {
			// メッセージフィルタを実行
			viewer.addFilter(new MessageFilter());
		}
		
		// 送信日テキストが両方とも空ではない場合
		if (!"".equals(section.get(MessageFilterManager.KEY_FROM)) && !"".equals(section.get(MessageFilterManager.KEY_TO))) {
			// 送信日(from)フィルタ実行
			viewer.addFilter(new DateFromFilter());
			// 送信日(to)フィルタ実行
			viewer.addFilter(new DateToFilter());
		} else if (!"".equals(section.get(MessageFilterManager.KEY_FROM)) && "".equals(section.get(MessageFilterManager.KEY_TO))) {
			// 送信日(from)のみ入力時　課題票(from)フィルタ実行
			viewer.addFilter(new DateFromFilter());
		} else if ("".equals(section.get(MessageFilterManager.KEY_FROM)) && !"".equals(section.get(MessageFilterManager.KEY_TO))) {
			// 送信日(to)のみ入力時　課題票(to)フィルタ実行
			viewer.addFilter(new DateToFilter());
		}
		
		// 送信者チェックが1つ以上外れている場合
		if(!"".equals(section.get(MessageFilterManager.KEY_SENDER))){
			// チェックが外れている送信者取得
			String[] senderNames = section.get(MessageFilterManager.KEY_SENDER).split("\\|");
			for(int i=0; i<senderNames.length; i++) {
				// 送信者フィルタを実行
				viewer.addFilter(new SenderFilter(senderNames[i]));
			}
		}


	}
	
	
	/**
	 * システムメッセージフィルタを設定します。
	 * 
	 * @author		<A href="mailto:s_imai@valtech.jp">$Author$</A>
	 * @version		$Rev$
	 */
	private class SystemFilter extends ViewerFilter {
		
		/* (非 Javadoc)
		 * @see org.eclipse.jface.viewers.ViewerFilter#select(org.eclipse.jface.viewers.Viewer, java.lang.Object, java.lang.Object)
		 */
		public boolean select(Viewer viewer, Object parentElement, Object element) {

			// メッセージタイプ取得
			String messageType = ((MessagePacket)element).getMessageType();
			
			// ユーザメッセージの場合
			if( MessageType.USER_VALUE.equals(messageType) ) {
				return true;
			}
			
			return false;
			
		}	
		
	}
	
	/**
	 * ユーザメッセージフィルタを設定します。
	 * 
	 * @author		<A href="mailto:s_imai@valtech.jp">$Author$</A>
	 * @version		$Rev$
	 */
	private class UserFilter extends ViewerFilter {
		
		/* (非 Javadoc)
		 * @see org.eclipse.jface.viewers.ViewerFilter#select(org.eclipse.jface.viewers.Viewer, java.lang.Object, java.lang.Object)
		 */
		public boolean select(Viewer viewer, Object parentElement, Object element) {

			// メッセージタイプ取得
			String messageType = ((MessagePacket)element).getMessageType();
			
			// ユーザメッセージの場合
			if( MessageType.USER_VALUE.equals(messageType) ) {
				return false;
			}
			
			return true;
			
		}
		
	}
	
	/**
	 * メッセージ検索フィルタ設定します。
	 * 
	 * @author		<A href="mailto:s_imai@valtech.jp">$Author$</A>
	 * @version		$Rev$
	 */
	private class MessageFilter extends ViewerFilter {
		
		/* (非 Javadoc)
		 * @see org.eclipse.jface.viewers.ViewerFilter#select(org.eclipse.jface.viewers.Viewer, java.lang.Object, java.lang.Object)
		 */
		public boolean select(Viewer viewer, Object parentElement, Object element) {
			
			// 「メッセージ」に指定された文字列が含まれている位置を調べる
			int index = ((MessagePacket)element).getMessage().indexOf(section.get(MessageFilterManager.KEY_MESSAGE));

			// 「含む」が選択されている場合
			if (Messages.getString("MessageListViewFilter.0").equals(section.get(MessageFilterManager.KEY_MESSAGECOMBO))) { //$NON-NLS-1$
				// メッセージに指定された文字列が含まれる場合は表示する
				return index != -1;
			
			} else {
				// 「含まない」が選択されている場合
				// メッセージに指定された文字列が含まれない場合は表示する
				return index == -1;
			}
		}
	}
	
	/**
	 * 送信日(from)フィルタを設定します。
	 * 送信日(from)に入力された日付より後に起票された課題票のみ表示します。
	 * 
	 * @author		<A href="mailto:s_imai@valtech.jp">$Author$</A>
	 * @version		$Rev$
	 */
	private class DateFromFilter extends ViewerFilter {
		
		/* (非 Javadoc)
		 * @see org.eclipse.jface.viewers.ViewerFilter#select(org.eclipse.jface.viewers.Viewer, java.lang.Object, java.lang.Object)
		 */
		public boolean select(Viewer viewer, Object parentElement, Object element) {
			// フォーマット設定
			SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy/MM/dd");
			Date fromDate = null;

			try {
				// 入力された送信日(from)のタイムスタンプ生成
				fromDate = new Date(dateFormat.parse(section.get(MessageFilterManager.KEY_FROM)).getTime());
			} catch (ParseException e) {
				BtsPlugin.getInstance().errorlog(e);
			}
			// 送信日(from)に入力された日付より後に起票された課題票のみ表示
			return ((MessagePacket)element).getSent().after(fromDate);
		}
	}
	
	
	/**
	 * 送信日フィルタを設定します。
	 * 送信日(to)に入力された日付より前に入力された課題票のみ表示します。
	 * 
	 * @author		<A href="mailto:s_imai@valtech.jp">$Author$</A>
	 * @version		$Rev$
	 */
	private class DateToFilter extends ViewerFilter {
		
		/* (非 Javadoc)
		 * @see org.eclipse.jface.viewers.ViewerFilter#select(org.eclipse.jface.viewers.Viewer, java.lang.Object, java.lang.Object)
		 */
		public boolean select(Viewer viewer, Object parentElement, Object element) {
			// フォーマット設定
			SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy/MM/dd");
			Date toDate = null;
			
			try {
				// 入力された送信日(to)のタイムスタンプ生成（toDateは+1日）
				toDate = new Date(dateFormat.parse(section.get(MessageFilterManager.KEY_TO)).getTime()+86400000);
			} catch (ParseException e) {
				BtsPlugin.getInstance().errorlog(e);
			}

			// 送信日(to)に入力された日付より前に入力された課題票のみ表示
			return ((MessagePacket)element).getSent().before(toDate);

		}
		
	}
	
	
	/**
	 * 送信者フィルタを設定します。
	 * 
	 * @author		<A href="mailto:s_imai@valtech.jp">$Author$</A>
	 * @version		$Rev$
	 */
	private class SenderFilter extends ViewerFilter {
		
		/** チェックが外れている送信者 */
		private String senderName;
		
		/**
		 * チェックが外れている送信者を更新
		 * @param senderName
		 */
		private SenderFilter(String senderName){
			this.senderName = senderName;
		}
		
		/* (非 Javadoc)
		 * @see org.eclipse.jface.viewers.ViewerFilter#select(org.eclipse.jface.viewers.Viewer, java.lang.Object, java.lang.Object)
		 */
		public boolean select(Viewer viewer, Object parentElement, Object element) {
			// チェックが外れている送信者は表示しない
			return !senderName.equals(((MessagePacket)element).getFromUser());
		}
	}

}
