/***********************************************************************
 * Copyright(C) 2006 Valtech Co.,Ltd.
 * All Rights Reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 ***********************************************************************/
package jp.valtech.bts.ui.msglist;


import jp.valtech.bts.data.CurrentProject;
import jp.valtech.bts.facade.MessagePacketFacade;
import jp.valtech.bts.ui.BtsPlugin;
import jp.valtech.bts.ui.action.OpenCalendarDlgAction;
import jp.valtech.bts.util.BTSUtility;
import jp.valtech.bts.util.Logging;

import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IDialogSettings;
import org.eclipse.jface.viewers.CheckboxTableViewer;
import org.eclipse.jface.viewers.ColumnWeightData;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.TableLayout;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.TraverseEvent;
import org.eclipse.swt.events.TraverseListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.forms.events.HyperlinkAdapter;
import org.eclipse.ui.forms.events.HyperlinkEvent;
import org.eclipse.ui.forms.widgets.ImageHyperlink;


/**
 * フィルターパターン入力用のダイアログです。
 * {@link jp.valtech.bts.ui.msglist.MessageListView メッセージ一覧ビュー}から呼ばれるダイアログです。
 * 
 * @author		<A href="mailto:m_sugitou@valtech.jp">M.Sugito</A>
 * @version	Ver.0.8
 */
public class MessageFilterDialog extends Dialog implements Logging {

    /** 「From日付」用テキストボックス */
    private Text frmDateTxt;
    /** 「To日付」用テキストボックス */
	private Text toDateTxt;
    /** 「メッセージ」用コンボ */
    private Combo messageCmb;
    /** 「メッセージ」用テキストボックス */
    private Text messageTxt;
    /** 「課題票の更新通知メッセージを表示する」用チェックボックス */
    private Button msgSystemChk;
    /** 「任意に送受信したメッセージを表示する」用チェックボックス */
    private Button msgUserChk;
    /**「 送信者一覧テーブル」用テーブルビューア */
    private CheckboxTableViewer senderViewer;
	
	/** 旧メッセージフィルタ情報 */
	private MessageFilter oldConfig;

	/** メッセージ一覧テーブルのビューアインスタンス */
	private TableViewer viewer;
	
	/** 送信者一覧 */
	private String[] fromUser;
	
	
	/**
	 * フィルター設定のダイアログを生成します。
	 * 
	 * @param			parent		呼び出し元のビューオブジェ
	 */
	public MessageFilterDialog(MessageListView parent) {
		super(parent.getSite().getShell());
		
		// ビューアを設定します。
		this.viewer = parent.getViewer();
		
		// フィルターの設定情報取得
		IDialogSettings settings = 
			BtsPlugin.getInstance().getDialogSettings().getSection(MessageFilterManager.SECTION_NAME);
		if(settings != null) {
			this.oldConfig = MessageFilterManager.getMessageFilter();
		}
	}
	

	/**
     * 当ダイアログの設定します。タイトルを設定します。
     * 
     * @param			newShell			当該ダイアログのシェル
     */
    protected void configureShell(Shell newShell) {
        super.configureShell(newShell);
        newShell.setText(Messages.getString("MessageFilterDialog.0"));  //$NON-NLS-1$
    }



    /**
     * フィルターパターン編集用のダイアログを生成します。
     * 
     * @param			parent					親コンポジット
     * @return			当ダイアログのコントロールインスタンス
     * @see				Dialog#createDialogArea(org.eclipse.swt.widgets.Composite)
     */
    protected Control createDialogArea(Composite parent) {
        Composite composite = (Composite) super.createDialogArea(parent);
        try {
            // 「メッセージ種別」設定のUIを生成
            createStatusArea(composite);

        	// 「メッセージ」文字列指定のUIを生成
            createContainsArea(composite);

            // 「送信日」設定のUIを生成
        	createDateArea(composite);
        	
            // 「送信者」設定のUIを生成
            createAssignedArea(composite);
            
            // セパレータ生成
            createSeparatorLine(composite);
            
            // 設定をロード
            if(oldConfig != null) {
                loadFilterSettings();
            }
            
        }catch (Exception e) {
			String msg = Messages.getString("MessageFilterDialog.1"); //$NON-NLS-1$
			logger.fatal(msg, e);
			BtsPlugin.getInstance().error(msg, e);
		}
        return composite;
    }
    
    

    
    /**
     * 送信日指定のUIを生成。
     * 親コンポジットに送信日指定のUIを生成します。
     * 
     * @param			parent					親コンポジット
     * @throws			Exception
     */
    private void createDateArea(Composite parent) throws Exception {

    	// この部品のコンポジット生成
    	Composite thisArea = new Composite(parent, SWT.NONE);

    	// レイアウト設定。７列表示のグリッドレイアウトを使う
        GridLayout gridLayout = new GridLayout();
        gridLayout.numColumns = 7;
        thisArea.setLayout(gridLayout);
    	
        // 「FromDate」ラベル
        Label label = new Label(thisArea, SWT.NONE);
        label.setText(Messages.getString("MessageFilterDialog.2")); //$NON-NLS-1$

        // 「FromDate」テキストボックス
        frmDateTxt = new Text(thisArea,SWT.BORDER);
		GridData gridData = new GridData(GridData.FILL); 
        gridData.widthHint = 80;
        frmDateTxt.setLayoutData(gridData);
        
        // 「FromDate」カレンダーボタン
		ImageHyperlink calFrom = new ImageHyperlink(thisArea, SWT.NULL);
		calFrom.setImage(BtsPlugin.getInstance().getImage("calendar.gif"));
		calFrom.setBackground(Display.getDefault().getSystemColor(SWT.COLOR_WIDGET_BACKGROUND));
		calFrom.setToolTipText(Messages.getString("MessageFilterDialog.3")); //$NON-NLS-1$
		calFrom.addHyperlinkListener(new HyperlinkAdapter() {
			public void linkActivated(HyperlinkEvent e) {
				// 日付入力用のカレンダーを開きます。
				OpenCalendarDlgAction action = new OpenCalendarDlgAction(frmDateTxt);
				action.run();
			}
		});

        // 「ToDate」ラベル
        label = new Label(thisArea, SWT.NONE);
        label.setText(Messages.getString("MessageFilterDialog.4")); //$NON-NLS-1$
        
        //「ToDate」テキストボックス
        toDateTxt = new Text(thisArea,SWT.BORDER);
		toDateTxt.setLayoutData(gridData);
		
        // 「ToDate」カレンダーボタン
		ImageHyperlink calTo = new ImageHyperlink(thisArea, SWT.NULL);
		calTo.setImage(BtsPlugin.getInstance().getImage("calendar.gif"));
		calTo.setBackground(Display.getDefault().getSystemColor(SWT.COLOR_WIDGET_BACKGROUND));
		calTo.setToolTipText(Messages.getString("MessageFilterDialog.5")); //$NON-NLS-1$
		calTo.addHyperlinkListener(new HyperlinkAdapter() {
			public void linkActivated(HyperlinkEvent e) {
				// 日付入力用のカレンダーを開きます。
				OpenCalendarDlgAction action = new OpenCalendarDlgAction(toDateTxt);
				action.run();
			}
		});	
        
        // 書式用ラベル
        label = new Label(thisArea, SWT.NONE);
        label.setText("(YYYY/MM/DD)");

    }
    
    
    /**
     * 検索用文字列指定のUIを生成。
     * 「メッセージ」に含む（または含まない）文字列を設定するUIを生成します。
     * 
     * @param			parent					親コンポジット
     * @throws			Exception
     */
    private void createContainsArea(Composite parent) throws Exception {
        
    	// この部品のコンポジット生成
    	Composite thisArea = new Composite(parent, SWT.NONE);

    	// レイアウト設定。3列のグリッドレイアウトにする。
    	GridLayout gridLayout = new GridLayout();
        gridLayout.numColumns = 3;
        thisArea.setLayout(gridLayout);
        thisArea.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
    	
        // 「メッセージ」ラベル
        Label label = new Label(thisArea, SWT.NONE);
        label.setText(Messages.getString("MessageFilterDialog.6")); //$NON-NLS-1$

        // 「メッセージ」コンボ
        messageCmb = new Combo(thisArea, SWT.READ_ONLY);
        messageCmb.add(Messages.getString("MessageFilterDialog.7")); //$NON-NLS-1$
        messageCmb.add(Messages.getString("MessageFilterDialog.8")); //$NON-NLS-1$
//        combo.addSelectionListener(selectionListener);
        messageCmb.addTraverseListener(new TraverseListener() {
            public void keyTraversed(TraverseEvent e) {
                if (e.detail == SWT.TRAVERSE_ESCAPE
                        || e.detail == SWT.TRAVERSE_RETURN) {
                    e.doit = false;
                }
            }
        });
        // デフォルト設定
        messageCmb.select(messageCmb.indexOf(Messages.getString("MessageFilterDialog.9"))); //$NON-NLS-1$

        // 「メッセージ」テキストボックス
        messageTxt = new Text(thisArea, SWT.SINGLE | SWT.BORDER);
        messageTxt.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
//        titleContains.addModifyListener(new ModifyListener() {
//            public void modifyText(ModifyEvent e) {
//                DialogBookmarkFilter.this.markDirty();
//            }
//        });

    }


    /**
     * 「送信者」テーブルの表示。
     * 登録されているメッセージに存在するユーザをテーブルに表示する。
     * 
     * @param		parent				親コンポジット
     */
    private void createAssignedArea(Composite parent) throws Exception {
        
        // このパーツのレイアウト設定。グリッドレイアウトを使う
        Composite thisArea = new Composite(parent, SWT.NONE);
        thisArea.setLayout(new GridLayout());

        // テーブルタイトルの設定
        Label label = new Label(thisArea, SWT.NONE);
        label.setText(Messages.getString("MessageFilterDialog.10"));  //$NON-NLS-1$

        // テーブルの表示設定
        Table table = new Table(thisArea
        		, SWT.CHECK | SWT.H_SCROLL | SWT.V_SCROLL | SWT.MULTI | SWT.FULL_SELECTION | SWT.BORDER);
        table.setLinesVisible(true);
        table.setHeaderVisible(true);

        // テーブルレイアウトの設定
        TableLayout tableLayout = new TableLayout();
        table.setLayout(tableLayout);
        
        // 1カラム目（送信者名）の設定
        tableLayout.addColumnData(new ColumnWeightData(350, true));
        TableColumn tc = new TableColumn(table, SWT.NONE, 0);
        tc.setText(Messages.getString("MessageFilterDialog.11"));  //$NON-NLS-1$

        // テーブルのビューアオブジェクト設定
        senderViewer = new CheckboxTableViewer(table);

        // テーブルの横幅・高さの調整
        GridData gridData = new GridData(GridData.FILL_HORIZONTAL);
        gridData.heightHint = 60;
        gridData.widthHint = 350;
        senderViewer.getControl().setLayoutData(gridData);
        
        // テーブルにプロバイダを設定
        senderViewer.setContentProvider(getAssignedTableContentProvider());
        senderViewer.setLabelProvider(getAssignedTableLabelProvider());

        // リスナーの設定
//        typesViewer.addCheckStateListener(checkStateListener);
        
        // 初期値を設定
		CurrentProject project = CurrentProject.getInsance();
		if(project.isOpen()) {
			MessagePacketFacade messagePacketFacade = new MessagePacketFacade(); 
			fromUser = messagePacketFacade.getFromUser();
			senderViewer.setInput(fromUser);
		}
        
        // デフォルト設定
        senderViewer.setAllChecked(true);
    }
    
    
    /**
     * 「送信者一覧テーブル」のコンテントプロバイダを返します。
     * 
     * @return			「送信者一覧テーブル」のラベルプロバイダ
     * @throws			Exception
     */
    private IStructuredContentProvider getAssignedTableContentProvider() throws Exception {
        return new IStructuredContentProvider() {
        	/*
        	 *  (非 Javadoc)
        	 * @see org.eclipse.jface.viewers.IStructuredContentProvider#getElements(java.lang.Object)
        	 */
            public Object[] getElements(Object inputElement) {
            	return fromUser;
            }

            /*
             *  (非 Javadoc)
             * @see org.eclipse.jface.viewers.IContentProvider#dispose()
             */
            public void dispose() {
            }

            /*
             *  (非 Javadoc)
             * @see org.eclipse.jface.viewers.IContentProvider#inputChanged(org.eclipse.jface.viewers.Viewer, java.lang.Object, java.lang.Object)
             */
            public void inputChanged(Viewer viewer, Object oldInput,
                    Object newInput) {
            }
        };
    }


    /**
     * {@link AssignedTableLabelProvider 「送信者一覧テーブル」のラベルプロバイダ}を返します。
     * 
     * @return			「送信者一覧テーブル」のラベルプロバイダ
     */
    private ILabelProvider getAssignedTableLabelProvider() {
        return new AssignedTableLabelProvider();
    }

    /**
     * 「送信者一覧テーブル」のラベルプロバイダです。 
     */
    private class AssignedTableLabelProvider extends LabelProvider implements ITableLabelProvider {

		/* (non-Javadoc)
		 * @see org.eclipse.jface.viewers.ITableLabelProvider#getColumnImage(java.lang.Object, int)
		 */
		public Image getColumnImage(Object element, int columnIndex) {
		    return null;
		}
		
		/* (non-Javadoc)
		 * @see org.eclipse.jface.viewers.ITableLabelProvider#getColumnText(java.lang.Object, int)
		 */
		public String getColumnText(Object element, int columnIndex) {
			switch (columnIndex) {
				case 0:
					return element.toString();
			}
			return "";
		}
    }

    
    /**
     * 「表示メッセージ種別の選択」のUIを生成します。
     * 
     * @param			parent						親コンポジット
     * @throws			Exception
     */
    private void createStatusArea(Composite parent) throws Exception {
        
        // このエリアのコンポジット生成。
        Composite thisArea = new Composite(parent, SWT.NONE);

        // グリッドレイアウトを使う。
        thisArea.setLayout(new GridLayout());
        thisArea.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));


        // 「課題票の更新通知メッセージを表示する」チェックボックス
        msgSystemChk = new Button(thisArea, SWT.CHECK);
        msgSystemChk.setText(Messages.getString("MessageFilterDialog.12")); //$NON-NLS-1$
        // デフォルト設定
        msgSystemChk.setSelection(true);

        // 「任意に送受信したメッセージを表示する」チェックボックス
        msgUserChk = new Button(thisArea, SWT.CHECK);
        msgUserChk.setText(Messages.getString("MessageFilterDialog.13")); //$NON-NLS-1$
        // デフォルト設定
        msgUserChk.setSelection(true);

    }
    
    /**
     * セパレータラインを生成します。
     * 親コンポジットに対してセパレータラインを生成します。
     * 
     * @param			parent						親コンポジット
     */
    protected void createSeparatorLine(Composite parent) {
        // 親コンポーネントにラベルを生成
        Label separator = new Label(parent, SWT.HORIZONTAL | SWT.SEPARATOR);
        
        // レイアウトを設定。グリッドレイアウトを使う。
        GridData gd = new GridData(GridData.FILL_HORIZONTAL);
        gd.horizontalSpan = 1;
        separator.setLayoutData(gd);
    }

    
	/* (非 Javadoc)
	 * @see org.eclipse.jface.dialogs.Dialog#okPressed()
	 */
	protected void okPressed() {
		
		// 入力された起票日取得
		String frmDateString = frmDateTxt.getText().trim();
		String toDateString = toDateTxt.getText().trim();
		
		// 起票日が入力されているか判定
		if(!"".equals(frmDateString) || !"".equals(toDateString)) {
			
			// エラーチェック
			if(!BTSUtility.dateTextError(frmDateString, toDateString)) {
				// エラーあり
				return;
			}
			
		}
		
		// エラーなし
		save();
		super.okPressed();
		
		// フィルタを実行
		MessageListViewFilter filter = new MessageListViewFilter(viewer); 
		filter.addMessageFilters();
	}
	
	
	/**
	 * フィルタの設定をXMLファイルに保存します。
	 */
	private void save() {
		// メッセージフィルタ情報を生成
		MessageFilter newConfig = new MessageFilter();

		// 入力項目をメッセージフィルタ情報にセット
		newConfig.setMsgSystemChk(msgSystemChk.getSelection());
		newConfig.setMsgUserChk(msgUserChk.getSelection());
		newConfig.setMessageCmb(messageCmb.getText());
		newConfig.setMessage(messageTxt.getText().trim());
		newConfig.setFrmDate(frmDateTxt.getText().trim());
		newConfig.setToDate(toDateTxt.getText().trim());
		
		// チェックが付いていない送信者を「|」区切りで保存
		StringBuffer notsenderChk = new StringBuffer();
		for(int i=0; i<((String[])senderViewer.getInput()).length; i++) {
			// チェックが付いていない送信者名を文字列に追加
			if(senderViewer.getChecked(senderViewer.getElementAt(i)) == false) {
				notsenderChk.append(senderViewer.getElementAt(i) + "|");
			}
		}
		// 最後尾の「|」を削除
		if(notsenderChk.length() != 0) {
			notsenderChk.deleteCharAt(notsenderChk.length()-1);
		}
		newConfig.setSenderChk(new String(notsenderChk));
		
		// 旧メッセージフィルタ情報の有無を判定
		if (oldConfig == null) {
			// 旧メッセージフィルタ情報がなければ新規作成
			MessageFilterManager.save(newConfig);
		} else {
			// 旧メッセージフィルタ情報があれば編集
			MessageFilterManager.modify(oldConfig, newConfig);
		}
	}
	
	
	/**
	 * フィルタ設定をロードします。
	 *
	 */
	private void loadFilterSettings() {
        msgSystemChk.setSelection(oldConfig.isMsgSystemChk());
        msgUserChk.setSelection(oldConfig.isMsgUserChk());
        messageCmb.setText(oldConfig.getMessageCmb());
        messageTxt.setText(oldConfig.getMessage());
        frmDateTxt.setText(oldConfig.getFrmDate());
        toDateTxt.setText(oldConfig.getToDate());
        // チェックが付いていない送信者の文字列を配列に変換
        String[] tmp = oldConfig.getSenderChk().split("\\|");        	
        for(int i=0; i<tmp.length; i++) {
        	senderViewer.setChecked(tmp[i], false);
        }
        	
	}
	
  
    
}
