/***********************************************************************
 * Copyright(C) 2006 Valtech Co.,Ltd.
 * All Rights Reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 ***********************************************************************/
package jp.valtech.bts.ui.editor;


import java.io.File;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import jp.valtech.bts.dao.BtsDBException;
import jp.valtech.bts.data.Attachment;
import jp.valtech.bts.data.CurrentProject;
import jp.valtech.bts.data.Issue;
import jp.valtech.bts.data.IssueStatus;
import jp.valtech.bts.data.IssueType;
import jp.valtech.bts.facade.VoteDAOFacade;
import jp.valtech.bts.ui.BtsPlugin;
import jp.valtech.bts.ui.action.DownloadIssueAttachmentAction;
import jp.valtech.bts.ui.action.OpenCalendarDlgAction;
import jp.valtech.bts.ui.action.OpenDetailsExportDlgAction;
import jp.valtech.bts.ui.action.OpenProjectConfigDlgAction;
import jp.valtech.bts.ui.action.OpenVoteDlgAction;
import jp.valtech.bts.ui.action.SaveIssueAction;
import jp.valtech.bts.ui.action.SendReleaseIssueAction;
import jp.valtech.bts.ui.navigator.ProjectConfig;
import jp.valtech.bts.util.BTSUtility;
import jp.valtech.bts.util.IssueUtility;
import jp.valtech.bts.util.Logging;

import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.ColumnWeightData;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.TableLayout;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CLabel;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Link;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IWorkbenchActionConstants;
import org.eclipse.ui.forms.events.HyperlinkAdapter;
import org.eclipse.ui.forms.events.HyperlinkEvent;
import org.eclipse.ui.forms.widgets.ImageHyperlink;
import org.eclipse.ui.forms.widgets.ScrolledForm;
import org.eclipse.ui.forms.widgets.Section;

/**
 * 課題票エディタの「概要」ページ。
 * 
 * @author		<A href="mailto:m_sugitou@valtech.jp">M.Sugito</A>
 * @version	Ver.0.8
 */
public class OverviewPart extends AbstractIssueEditorPart implements Logging {

	/** このページのタブに設定する文字列 */
	public static final String TAB_TITLE=Messages.getString("OverviewPart.0"); //$NON-NLS-1$
	
	private static final Image IMAGE_DEADLINE_WORNING  = BtsPlugin.getInstance().getImage("warning1.gif");
	private static final Image IMAGE_DEADLINE_ERROR  = BtsPlugin.getInstance().getImage("error1.gif");

	/** 部品のサイズ：ラベルの幅（小） */
	private static final int WITH_LABEL_S = 55;
	/** 部品のサイズ：ラベルの幅（中） */
	private static final int WITH_LABEL_L = 250;
	/** 部品のサイズ：テキストボックスの幅（小） */
	private static final int WITH_TEXT_S = 70;
	/** 部品のサイズ：テキストボックスの幅（中） */
	private static final int WITH_TEXT_M = 120;
	/** 部品のサイズ：ボタンの幅（中） */
	private static final int WIDTH_BUTTON_M = 60;
	/** 部品のサイズ：ペインの幅 */
	private static final int WIDTH_PAIN = 430;
	/** 部品のサイズ：テキストボックスの幅（大） */
	private static final int WIDTH_TEXT_L = 300;

	/** 部品のサイズ：テキストボックスの高さ（大） */
	private static final int HEIGHT_TEXT_L = 170;
	/** 部品のサイズ：ボタンの高さ（小） */
	private static final int HEIGHT_BUTTON_S = 20;


	/** 基本レイアウトに使う：ラベルのサイズを定義(小) */
    private GridData labelSizeS;
	/** 基本レイアウトに使う：ラベルのサイズを定義(中) */
    private GridData labelSizeL;
	/** 基本レイアウトに使う：ラベルのサイズを定義(中) */
    private GridData labelSizeMB;
	/** 基本レイアウトに使う：テキストのサイズを定義(小) */
    private GridData textSizeS;
	/** 基本レイアウトに使う：テキストのサイズを定義(中) */
    private GridData textSizeM;
	/** 基本レイアウトに使う：テキストのサイズを定義(大) */
    private GridData textSizeL;
	/** 基本レイアウトに使う：テキストのサイズを定義(大大) */
    private GridData textSizeLL;
    /** 基本レイアウトに使う：コンボのサイズを定義 */
    private GridData comboSizeM;
    /** 基本レイアウトに使う：ボタンの高さを定義 */
    private GridData buttonSizeSS;
    /** 基本レイアウトに使う：ボタンの高さを定義 */
    private GridData buttonSizeM;
    
	
    /** このページの部品：「起票者」ラベル*/
    private Label createUserLbl;
    /** このページの部品：「起票日時」ラベル*/
    private Label createDateLbl;
    /** このページの部品：「更新日時」ラベル*/
    private Label updateDateLbl;
    /** このページの部品：「更新日時」ラベル*/
    private Label versionLbl;
    /** このページの部品：「課題票ID」テキストボックス*/
    private CLabel issueIdLbl;
	/** このページの部品：「タイトル」テキストボックス */
    private Text titleTxt;
	/** このページの部品：「詳細」テキストボックス */
    private Text descTxt;
	/** このページの部品：「カテゴリ」コンボボックス */
    private Combo categoryCmb;
	/** このページの部品：「優先度」コンボボックス */
    private Combo priorityCmb;
	/** このページの部品：「担当者名」コンボボックス */
    private Combo userNameCmb;
	/** このページの部品：「担当者名」ラベル */
    private Label userNameLbl;
	/** このページの部品：「ステータス：完了」ラジオボタン */
	private List statusRdiList = new ArrayList();
	private Button stClosedRdi;
	/** このページの部品：「完了期限」テキストボックス */
    private Text deadLineTxt;
	/** このページの部品：「内緒のバグ」チェックボックス */
    private Button secretChk;
	/** このページの部品：「保留」チェックボックス */
    private Button suspendChk;
	/** このページの部品：「完了期限」のラベル(アイコン表示) */
    private Label limitLabelImg;
	/** このページの部品：「完了期限」のラベル */
    private Label limitLabel;
    
    /**「添付ファイル一覧テーブル」用テーブルビューア */
    public TableViewer attachViewer;
	/** このページのフォーム */
	private ScrolledForm form ;

	
    /** ファイル出力ダイアログを開くアクション */
    private OpenDetailsExportDlgAction openDetailsExportDlgAction;
	/** 課題票を保存・送信するアクション */
	private SendReleaseIssueAction sendIssueCmdAction ;
	/** 課題票を保存するアクション */
	private SaveIssueAction saveIssueAction ;
	/** プルダウンメニューを編集するアクション */
	private OpenProjectConfigDlgAction editPullDownMenuAction; 
	/** 添付ファイルを一覧から削除するアクション */
	private RemoveAttachmentAction removeAttachmentAction;
	/** 投票のリンク */
	ImageHyperlink voteLink; 

	/** マルチページエディタの親 */
	IssueMultiPageEditor parentEditor;
	
    /**
     * 当該ページのインスタンスを生成します。
     */
    public OverviewPart(IssueMultiPageEditor parentEditor) {
		super();
    	this.parentEditor = parentEditor;
	}
    
    
	/* (非 Javadoc)
	 * @see org.eclipse.ui.part.WorkbenchPart#createPartControl(org.eclipse.swt.widgets.Composite)
	 */
	public void createPartControl(Composite parent) {
		try {
			// このページで使うアクションを生成
			makeActions();
			
			// このページのFormToolKitを生成
			if (toolkit == null) {
				getFormToolkit(parent);
			}
			
			// 共通レイアウト生成
			createCommonLayout();

			// フォームボディ部生成
			Composite body = createBody(parent);

			// フォームボディ部のGridLayout
			GridLayout bodyLayout = new GridLayout();
			bodyLayout.marginHeight = 0;
			bodyLayout.marginWidth = 0;

			// 「基本セクション（左ペイン）」のGridData
			GridData grid = new GridData(GridData.FILL_HORIZONTAL);
			grid.verticalSpan = 3;
			grid.verticalAlignment = GridData.BEGINNING;
			grid.minimumWidth = WIDTH_PAIN;

			// 「基本情報セクション（左ペイン）」を生成
			Composite generalSection = toolkit.createComposite(body);
			generalSection.setLayout(bodyLayout);
			generalSection.setLayoutData(grid);
			createGeneralSection(generalSection);

			// 「オプションセクション（右ペイン上）」のGridData 
			grid = new GridData(GridData.FILL_HORIZONTAL);
			grid.verticalAlignment = GridData.BEGINNING;
			
			// 「オプションセクション（右ペイン上）」を生成
			Composite optionSection = toolkit.createComposite(body);
			optionSection.setLayout(bodyLayout);
			optionSection.setLayoutData(grid);
			createOptionSection(optionSection);

			// 「更新情報セクション（右ペイン下）」のGridData 
			grid = new GridData(GridData.FILL_HORIZONTAL);
			grid.verticalAlignment = GridData.BEGINNING;
			
			// 「更新情報セクション（右ペイン下）」を生成
			Composite updInfoSection = toolkit.createComposite(body);
			updInfoSection.setLayout(bodyLayout);
			updInfoSection.setLayoutData(grid);
			createUpdateInfoSection(updInfoSection);

			// ボタンの活性／非活性を設定する
			changeActivate();

			// レイアウトやスクロールバーの再処理を行う
			this.form.reflow(false);

		}catch (Exception e) {
			String msg = Messages.getString("OverviewPart.1"); //$NON-NLS-1$
			logger.fatal(msg, e);
			BtsPlugin.getInstance().error(msg, e);
		}
	}
	
	
	/**
	 * アクションクラスの生成。
	 * このビューから呼び出されるアクションクラスを生成します。
	 *
	 */
	private void makeActions() {

		// ファイル出力ダイアログ表示アクション
		openDetailsExportDlgAction = new OpenDetailsExportDlgAction(this, parentEditor);

		IssueMultiPageEditorInput 
			issueEditorInput = (IssueMultiPageEditorInput)getEditorInput();
		if(issueEditorInput.getOpenMode() == IssueMultiPageEditorInput.OPEN_NEW_ISSUE) {
			openDetailsExportDlgAction.setEnabled(false);
		}
		
		// 課題票を保存・送信するアクション
		sendIssueCmdAction = new SendReleaseIssueAction();
		
		// 課題票を保存するアクション
		saveIssueAction = new SaveIssueAction();
		saveIssueAction.setEnabled(false);// 編集があるまで非活性
		
		// プルダウンメニューを編集するアクション
		editPullDownMenuAction = new OpenProjectConfigDlgAction(this);  

		// 添付ファイルを一覧から削除するアクション
		removeAttachmentAction = new RemoveAttachmentAction(); 
	}
	
	
	/**
	 * このページの基本となるボディ部を生成します。
	 * 
	 * @param		composite			親コンポジット
	 * @return		ボディ部のコンポジット
	 */
	private Composite createBody(Composite parent) {

		// 共通レイアウト
		GridLayout layout = new GridLayout();
		layout.marginHeight = 0;

		// フォーム生成
		form = toolkit.createScrolledForm(parent);
		// タイトル設定
		form.setText(Messages.getString("OverviewPart.2")); //$NON-NLS-1$
		
		// レイアウト設定
		form.getBody().setLayout(layout);
		form.getBody().setLayoutData(new GridData(GridData.FILL_BOTH));
		// 右上のボタン設定
		form.getToolBarManager().add(sendIssueCmdAction);		// 課題票を送信するアクション
		form.getToolBarManager().add(saveIssueAction);			// 課題票を保存するアクション
		form.getToolBarManager().add(openDetailsExportDlgAction);		// ファイル出力アクション
		form.updateToolBar();	// ボタンバー更新
		
		// 基本レイアウト
		Composite body = toolkit.createComposite(form.getBody());
		layout = new GridLayout();
		layout.marginHeight = 0;
		layout.numColumns = 2;
		body.setLayout(layout);
		
		body.setLayoutData(new GridData(GridData.FILL_BOTH));
		
		return body;
	}

	
    /**
     * この画面で使う共通レイアウトを生成します。
     *
     */
     private void createCommonLayout() {

  		labelSizeS = new GridData(); 
 		labelSizeS.widthHint = WITH_LABEL_S;
 		
 		labelSizeL = new GridData(GridData.FILL_HORIZONTAL); 
 		labelSizeL.widthHint = WITH_LABEL_L;

 		labelSizeMB = new GridData(); 
 		labelSizeMB.widthHint = WITH_LABEL_S;
 		labelSizeMB.verticalAlignment = GridData.BEGINNING;

 		
 		buttonSizeSS = new GridData();
 		buttonSizeSS.heightHint = HEIGHT_BUTTON_S;
 		
 		buttonSizeM = new GridData();
 		buttonSizeM.widthHint = WIDTH_BUTTON_M;

 		textSizeM = new GridData(); 
 		textSizeM.widthHint = WITH_TEXT_M;

 		textSizeS = new GridData(); 
 		textSizeS.widthHint = WITH_TEXT_S;

 		textSizeL = new GridData(GridData.FILL_HORIZONTAL);
 		textSizeL.widthHint = WIDTH_TEXT_L;

 		textSizeLL = new GridData(GridData.FILL_HORIZONTAL);
 		textSizeLL.heightHint = HEIGHT_TEXT_L;
 		textSizeLL.widthHint = WIDTH_TEXT_L;
 		
		comboSizeM = new GridData();
		comboSizeM.widthHint = WITH_TEXT_M;
     }


     /**
      * 更新情報セクション（右ペイン下）を生成します。 
      * 
 	  * @param		composite			親コンポジット
      */
     private void createUpdateInfoSection(Composite parent) {
      	// セクション生成・ラベル設定 
  		Section section = createSection(parent, Messages.getString("OverviewPart.3")); //$NON-NLS-1$
  		section.setDescription(Messages.getString("OverviewPart.4")); //$NON-NLS-1$
  		Composite composite = toolkit.createComposite(section);
  		
  		// レイアウト設定
  		composite.setLayout(new GridLayout());
  		composite.setLayoutData(new GridData(GridData.FILL_BOTH));
  		section.setClient(composite);
     	 
		// タイムスタンプエリアを表示
		createTimeStampArea(composite);

     }

     
    /**
     * オプションセクション（右ペイン上）を生成します。 
     * 
	 * @param		composite			親コンポジット
     */
     private void createOptionSection(Composite parent) {
    	 
    	// セクション生成・ラベル設定 
		Section section = createSection(parent, Messages.getString("OverviewPart.5")); //$NON-NLS-1$
		section.setDescription(Messages.getString("OverviewPart.6")); //$NON-NLS-1$
		Composite composite = toolkit.createComposite(section);
		
		// レイアウト設定
		composite.setLayout(new GridLayout());
		composite.setLayoutData(new GridData(GridData.FILL_BOTH));
		section.setClient(composite);
		
		// 完了期限エリア表示
		createDeadLineArea(composite);
		// オプションステータスエリア表示
		createOptionStArea(composite);
		// 添付ファイルエリア表示
		createAttachmentArea(composite);
	}
     

	/**
	 * 基本情報セクション（左ペイン）を生成します。
	 * 
	 * @param		composite			親コンポジット
	 * @throws BtsDBException 
	 */
  	private void createGeneralSection(Composite parent) throws BtsDBException {

  		// セクション生成・ラベル設定
		Section section = createSection(parent, Messages.getString("OverviewPart.7")); //$NON-NLS-1$
		section.setDescription(Messages.getString("OverviewPart.8")); //$NON-NLS-1$
		Composite composite = toolkit.createComposite(section);

		// レイアウト設定
		composite.setLayout(new GridLayout());
		composite.setLayoutData(new GridData(GridData.FILL_BOTH));
		section.setClient(composite);

		// セクションのヘッダに「投票」ボタンをつける
		voteLink = new ImageHyperlink(section, SWT.NULL);
		getFormToolkit(parent).adapt(voteLink, true, true);
		voteLink.setImage(BtsPlugin.getInstance().getImage("vote.gif"));
		voteLink.setBackground(section.getTitleBarGradientBackground());
		voteLink.setToolTipText(Messages.getString("OverviewPart.9")); //$NON-NLS-1$
		
		// 開いている課題票の投票総数取得
		VoteDAOFacade voteDAOFacade = new VoteDAOFacade();
		int voteSum = voteDAOFacade.getSum(issueDisplayValue.getIssue().getFingerPrint());
		
		// 投票リンクの表示設定
		String voteStr = Messages.getString("OverviewPart.40") + voteSum + Messages.getString("OverviewPart.41"); //$NON-NLS-1$ //$NON-NLS-2$
		voteLink.setForeground(null);
		voteLink.setFont(null);
		
		// 現在開いているプロジェクト設定情報取得
		ProjectConfig config = CurrentProject.getInsance().getProjectConfig();
		
		// 投票数の警告表示をする場合
		if(config.isVoteCheck()) {
			
			// 投票数警告日数を取得
			int warningCnt = Integer.parseInt(config.getVote());
			
			// 投票数が投票警告日を超えている場合
			if(voteSum > warningCnt) {
				
				// 警告メッセージ
				voteStr = Messages.getString("OverviewPart.42") + voteSum + Messages.getString("OverviewPart.43"); //$NON-NLS-1$ //$NON-NLS-2$

				// 警告メッセージ色設定
				voteLink.setForeground(new Color(Display.getCurrent(), 255, 0, 0));

				// 警告メッセージフォント設定
				voteLink.setFont(new Font(Display.getCurrent(), "", 10, SWT.BOLD));
				
			}
			
		}
		
		// 投票リンクに文字列をセット
		voteLink.setText(voteStr);
		
		voteLink.addHyperlinkListener(new HyperlinkAdapter() {
			public void linkActivated(HyperlinkEvent e) {
				// 「投票」ボタン押下の動作。投票用アクションを呼びます。
				OpenVoteDlgAction action = new OpenVoteDlgAction(OverviewPart.this, voteLink);
				action.setIssue(issueDisplayValue.getIssue());
				action.run();
			}
		});
		
		section.setTextClient(voteLink);
		
		// 課題票エリアを表示
		createIssueIDArea(composite);
		// カテゴリエリアを表示
		createCategoryArea(composite);
		// タイトルエリアを表示
		createTitleArea(composite);
		// 詳細エリアを表示
		createDescriptionArea(composite);
		// 優先度エリアを表示
		createPriorityArea(composite);
		// ステータスエリアを表示
		createStatusArea(composite);
	}
	

  	
	/**
	 * 添付ファイルエリアを生成します。
	 * 
	 * @param		composite			親コンポジット
	 */
	private void createAttachmentArea(Composite composite) {

		// このエリアのコンポジット生成
		Composite thisArea = toolkit.createComposite(composite);
		toolkit.paintBordersFor(thisArea);
		
		// レイアウト設定
		GridLayout layout = new GridLayout();
		layout.numColumns = 2;
		thisArea.setLayout(layout);

		// レイアウト設定
		thisArea.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));

		// ラベル
		Label label = toolkit.createLabel(thisArea, Messages.getString("OverviewPart.10"), SWT.RIGHT); //$NON-NLS-1$
		label.setLayoutData(labelSizeMB);
		
		// 添付ファイル右エリア用コンポジット生成
		Composite right = toolkit.createComposite(thisArea);
		layout = new GridLayout();
		layout.marginHeight = 0;
		layout.marginWidth = 0;
		right.setLayout(layout);
		// レイアウト設定
		right.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));

		// 添付ファイル一覧テーブル
		createAttachementTblArea(right);
		// 添付ファイル操作ボタン
		createAttachementBtnArea(right);
	}
 	
	
	/**
	 * 添付ファイル操作ボタンを表示する部品を生成します。
	 * 
	 * @param		composite			親コンポジット
	 */
	private void createAttachementBtnArea(Composite composite) {

		// このエリアのコンポジット生成
		Composite thisArea = toolkit.createComposite(composite);
		toolkit.paintBordersFor(thisArea);
		
		// レイアウト設定
		GridLayout layout = new GridLayout();
		layout.verticalSpacing = 0;
		layout.marginHeight = 1;
		thisArea.setLayout(layout);

		// レイアウト設定
		thisArea.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		
		// 添付ファイル追加ボタン
		Button addAttachment = toolkit.createButton(thisArea, Messages.getString("OverviewPart.11"), SWT.PUSH); //$NON-NLS-1$
		addAttachment.addSelectionListener(new SelectionAdapter(){
        	public void widgetSelected(SelectionEvent e){

        		// ファイル選択ダイアログを開く
        		FileDialog dialog = new FileDialog(BtsPlugin.getInstance().getShell(), SWT.OPEN);
        		// 選択したファイルのフルパスを取得
        		String path = dialog.open();
    			if( path==null || "".equals(path)) {
        			return;
        		}
        		
        		// 存在するファイルかどうかのチェック
        		File file = new File(path);
        		if(!file.exists()) {
        			return;
        		}

        		// 現在開いている課題票情報を取得する
        		IssueMultiPageEditorInput 
        			issueEditorInput = (IssueMultiPageEditorInput)getEditorInput();
        		Issue issue = issueEditorInput.getIssue();

        		// 添付ファイル情報を生成する。
        		Attachment attachment = new Attachment();
        		attachment.setFingerPrint(issue.getFingerPrint());
        		attachment.setType(issue.getType());
        		attachment.setRealPath(path);
        		attachment.setDisplayName( file.getName() );
        		attachment.setRegisted(false);// DB保存後にtrueになる
        		
    			// 実データの更新
    			((List)attachViewer.getInput()).add(attachment);

    			// テーブル表示の更新
    			attachViewer.insert(attachment, -1);
    			
				// 更新を通知する
				dirty = true;
				firePropertyChange(PROP_DIRTY);
				fireIssueChange();
        	}
        });
	}
	

	/**
	 * テーブルを右クリックしたときの動作です。
	 * 右クリックメニューを表示します。
	 */
	private void hookContextMenu() {
		MenuManager menuMgr = new MenuManager("#PopupMenu"); 
		menuMgr.setRemoveAllWhenShown(true);
		menuMgr.addMenuListener(new IMenuListener() {
			public void menuAboutToShow(IMenuManager manager) {
				// 選択した課題票を取得する
				Attachment attachment = getSelectionRecord();
				if(attachment != null) {
					DownloadIssueAttachmentAction action = new DownloadIssueAttachmentAction();
					action.setAttachment(attachment);
					manager.add(action);
					
					manager.add(removeAttachmentAction);
				}
				// Other plug-ins can contribute there actions here
				manager.add(new Separator(IWorkbenchActionConstants.MB_ADDITIONS));
			}
		});
		Menu menu = menuMgr.createContextMenu(attachViewer.getControl());
		attachViewer.getControl().setMenu(menu);
		getSite().registerContextMenu(menuMgr, attachViewer);
	}

	/**
	 * 選択しているレコードを返します。
	 * 選択しているレコードを{@link Issue}にキャストして返します。
	 * 
	 * @return			キャスト済み選択レコード
	 */
	private Attachment getSelectionRecord() {
		
		// 選択しているレコードを取得します。
		ISelection selection = attachViewer.getSelection();
		Object obj = ((IStructuredSelection)selection).getFirstElement();
		if(obj instanceof Attachment) {
			// キャストして返します。
			return (Attachment)obj;
		}
		return null;
	}

	
	/**
	 * 添付ファイル一覧テーブルを表示する部品を生成します。
	 * 
	 * @param		composite			親コンポジット
	 */
	private void createAttachementTblArea(Composite composite)  {

		// このエリアのコンポジット生成
		Composite thisArea = toolkit.createComposite(composite);
		toolkit.paintBordersFor(thisArea);
		
		// レイアウト設定
		GridLayout layout = new GridLayout();
//		layout.verticalSpacing = 0;
		layout.marginHeight = 1;
		layout.marginWidth = 1;
		thisArea.setLayout(layout);

		// レイアウト設定
		thisArea.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		
		
        // テーブル生成
        Table table = new Table(thisArea , SWT.V_SCROLL | SWT.FULL_SELECTION);
        table.setLinesVisible(true);
        table.setHeaderVisible(true);

        // テーブルレイアウトの設定
        TableLayout tableLayout = new TableLayout();
        table.setLayout(tableLayout);
        
        // 1カラム目（添付ファイル名）の設定
        tableLayout.addColumnData(new ColumnWeightData(200, true));
        TableColumn col1 = new TableColumn(table, SWT.NONE , 0);
        col1.setText(Messages.getString("OverviewPart.12"));  //$NON-NLS-1$

        // テーブルのビューアオブジェクト設定
        attachViewer = new TableViewer(table);

        // テーブルの横幅・高さの調整
        GridData tableGrid = new GridData();
        tableGrid.heightHint = 60;
        tableGrid.widthHint = 200;
        attachViewer.getControl().setLayoutData(tableGrid);
        
        // テーブルにプロバイダを設定
        attachViewer.setContentProvider(new AttachementTableContentProvider());
        attachViewer.setLabelProvider(new AttachementTableLabelProvider());

        // テーブルの右クリックメニュー設定
        hookContextMenu();
        
        // 初期値を設定
        attachViewer.setInput(issueDisplayValue.getAttachmentList());
		
	}
	
	
	/**
	 * オプションのステータスを表示する部品を生成します。
	 * 
	 * @param		composite			親コンポジット
	 */
	private void createOptionStArea(Composite composite) {

		// このエリアのコンポジット生成
		Composite thisArea = toolkit.createComposite(composite);
		toolkit.paintBordersFor(thisArea);
		
		// レイアウト設定
		GridLayout layout = new GridLayout();
		layout.numColumns = 2;
		layout.verticalSpacing = 0;
		layout.marginHeight = 4;
		thisArea.setLayout(layout);

		// レイアウト設定
		thisArea.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		
		// ラベル
		Label label = toolkit.createLabel(thisArea, Messages.getString("OverviewPart.13"), SWT.RIGHT); //$NON-NLS-1$
		// ラベルのレイアウト
		GridData gridData = new GridData(); 
		gridData.widthHint = WITH_LABEL_S;
		gridData.verticalAlignment = GridData.BEGINNING;
		gridData.verticalSpan = 2;
		label.setLayoutData(gridData);

		// ステータス：内緒のバグ
		secretChk = createCheckBox(thisArea, Messages.getString("OverviewPart.14"), issueDisplayValue.isSecretBug()); //$NON-NLS-1$

		// ステータス：保留
		suspendChk = createCheckBox(thisArea, Messages.getString("OverviewPart.15"), issueDisplayValue.isSuspend()); //$NON-NLS-1$
	}

	
	/**
	 * 完了期限を表示する部品を生成します。
	 * 
	 * @param		composite			親コンポジット
	 */
	private void createDeadLineArea(Composite composite) {

//		 このエリアのコンポジット生成
		Composite thisArea = toolkit.createComposite(composite);
		toolkit.paintBordersFor(thisArea);
		
		// レイアウト設定
		GridLayout layout = new GridLayout();
		layout.numColumns = 5;
		layout.marginHeight = 0;
		thisArea.setLayout(layout);
		
		// レイアウト設定
		thisArea.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));

		// ラベル
		Label label = toolkit.createLabel(thisArea, Messages.getString("OverviewPart.16"), SWT.RIGHT); //$NON-NLS-1$
		label.setLayoutData(labelSizeS);
		
		// 完了期限入力テキスト
		deadLineTxt = createText(thisArea, issueDisplayValue.getDeadline());
		deadLineTxt.setLayoutData(textSizeS);

		// カレンダ表示ボタン
		ImageHyperlink calendarLink = new ImageHyperlink(thisArea, SWT.NULL);
		calendarLink.setImage(BtsPlugin.getInstance().getImage("calendar.gif"));
		calendarLink.setBackground(COLOR_WHITE);
		calendarLink.setToolTipText(Messages.getString("OverviewPart.17")); //$NON-NLS-1$
		calendarLink.addHyperlinkListener(new HyperlinkAdapter() {
			public void linkActivated(HyperlinkEvent e) {
				// 日付入力用のカレンダーを開きます。
				OpenCalendarDlgAction action = new OpenCalendarDlgAction(deadLineTxt);
				action.run();
			}
		});

		// 完了期限テキスト文字列取得
		String deadLineStr = deadLineTxt.getText().trim();
		
		// アイコン表示
		limitLabelImg = toolkit.createLabel(thisArea, "");
		limitLabelImg.setLayoutData(new GridData(GridData.HORIZONTAL_ALIGN_END));

		// メッセージ表示
		limitLabel = toolkit.createLabel(thisArea, "");
		GridData data = new GridData();
		limitLabel.setLayoutData(data);

		// 完了期限を判定
		int judge = BTSUtility.judgeLimit(BTSUtility.createDate(deadLineStr)
										 , getIssueStatus().getValue());
		setDeadLineMessage(judge);
		
	}
 	
	/**
	 * 完了期限の参照してそれに応じたメッセージを表示する
	 * 
	 * @param		judge		完了期限判定用フラグ
	 */
	private void setDeadLineMessage(int judge) {
		if(judge == Issue.DEADLINE_OVER) {
			// 完了期限が切れている場合
			limitLabelImg.setImage(IMAGE_DEADLINE_ERROR);
			limitLabel.setText(Messages.getString("OverviewPart.44")); //$NON-NLS-1$
			limitLabel.setForeground(new Color(Display.getCurrent(), 255, 0, 0));

		} else if(judge == Issue.DEADLINE_WARNING) {
			// 完了期限が迫っている場合
			limitLabelImg.setImage(IMAGE_DEADLINE_WORNING);
			limitLabel.setText(Messages.getString("OverviewPart.45")); //$NON-NLS-1$
			limitLabel.setForeground(new Color(Display.getCurrent(), 0, 0, 255));
		} else if(judge == Issue.DEADLINE_SAFE) {
			// 完了期限まで余裕がある場合
			limitLabelImg.setImage(null);
			limitLabelImg.setText("    ");
			limitLabel.setText("                               ");
		}		
	}
	
	/**
	 * ステータスを表示する部品を生成します。
	 * 
	 * @param		composite			親コンポジット
	 */
	private void createStatusArea(Composite composite) {

		// このエリアのコンポジット生成
		Composite thisArea = toolkit.createComposite(composite);
		toolkit.paintBordersFor(thisArea);
		
		// レイアウト設定
		GridLayout layout = new GridLayout();
		layout.numColumns = 4;
		layout.verticalSpacing = 0;
		layout.marginHeight = 4;
		thisArea.setLayout(layout);

		// レイアウト設定
		thisArea.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));

		// ラベル
		Label label = toolkit.createLabel(thisArea, Messages.getString("OverviewPart.19"), SWT.RIGHT); //$NON-NLS-1$
		
		// ラベルレイアウト
		GridData gridData = new GridData(); 
		gridData.widthHint = WITH_LABEL_S;
		gridData.verticalAlignment = GridData.BEGINNING;
		gridData.verticalSpan = 5;
		label.setLayoutData(gridData);

		// 「担当者割り当て」以外はレイアウトにcolspanを設定する
		GridData colspan = new GridData(); 
		colspan.horizontalSpan = 3;
		
		// ステータス：起票
		Button stOpenedRdi = createRadio(thisArea, IssueStatus.OPENED.getDescription(), issueDisplayValue.isStatusOpened());
		stOpenedRdi.setLayoutData(colspan);
		statusRdiList.add(stOpenedRdi);

		// ステータス：担当者割り当て
		Button stAssignedRdi = createRadio(thisArea, IssueStatus.ASSIGNED.getDescription(), issueDisplayValue.isStatusAssigned());
		statusRdiList.add(stAssignedRdi);

		// 担当者入力用コンボ
		userNameLbl = toolkit.createLabel(thisArea, Messages.getString("OverviewPart.20"), SWT.RIGHT); //$NON-NLS-1$
		userNameCmb = createCombo(thisArea, IssueUtility.getAssigned(), issueDisplayValue.getAssigned());
		userNameCmb.setLayoutData(comboSizeM);
		
		// 担当者の文字数制限（20文字）
		userNameCmb.setTextLimit(20);

		// 「ステータス：担当者割り当て」はリスナを追加する
		stAssignedRdi.addSelectionListener(new SelectionListener() {
			public void widgetSelected(SelectionEvent e) {

				// 「担当者割り当て」選択時に「担当者」の活性／非活性を切り替える
				Button assigned = (Button)e.widget;
				if(assigned.getSelection()) {
					userNameCmb.setEnabled(true);
					userNameLbl.setEnabled(true);
				} else {
					userNameCmb.setEnabled(false);
					userNameLbl.setEnabled(false);
				}
			}

			public void widgetDefaultSelected(SelectionEvent e) {
				// 何もしない;
			}
		});

		// 担当者コンボの初期表示。「担当者割り当て」選択時のみ活性化する。
		if(!stAssignedRdi.getSelection()) {
			userNameCmb.setEnabled(false);
			userNameLbl.setEnabled(false);
		}
		
		// ステータス：対応済み
		Button stResolvedRdi = createRadio(thisArea, IssueStatus.RESOLVED.getDescription(), issueDisplayValue.isStatusResolved());
		stResolvedRdi.setLayoutData(colspan);
		statusRdiList.add(stResolvedRdi);
		
		// ステータス：確認済み
		Button stVerifiedRdi = createRadio(thisArea, IssueStatus.VERIFIED.getDescription(), issueDisplayValue.isStatusVerified());
		stVerifiedRdi.setLayoutData(colspan);
		statusRdiList.add(stVerifiedRdi);
		
		// ステータス：完了
		stClosedRdi = createRadio(thisArea, IssueStatus.CLOSED.getDescription(), issueDisplayValue.isStatusClosed());
		stClosedRdi.setLayoutData(colspan);
		statusRdiList.add(stClosedRdi);
	}

	
	/**
	 * 優先順度を表示する部品を生成します。
	 * 
	 * @param		composite			親コンポジット
	 */
	private void createPriorityArea(Composite composite) {

		// このエリアのコンポジット生成
		Composite thisArea = toolkit.createComposite(composite);
		toolkit.paintBordersFor(thisArea);
		
		// レイアウト設定
		GridLayout layout = new GridLayout();
		layout.numColumns = 2;
		layout.marginHeight = 2;
		thisArea.setLayout(layout);

		// レイアウト設定
		thisArea.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		
		// ラベル
		Label label = toolkit.createLabel(thisArea, Messages.getString("OverviewPart.21"), SWT.RIGHT); //$NON-NLS-1$
		label.setLayoutData(labelSizeS);

		// 優先度コンボ
		priorityCmb = createCombo(thisArea, IssueUtility.getPriorities(), issueDisplayValue.getPriority());
		priorityCmb.setLayoutData(comboSizeM);
		
		// 優先度の文字数制限（20文字）
		priorityCmb.setTextLimit(20);
	}
	

	/**
	 * 詳細を表示する部品を生成します。
	 * 
	 * @param		composite			親コンポジット
	 */
	private void createDescriptionArea(Composite composite) {

		// このエリアのコンポジット生成
		Composite thisArea = toolkit.createComposite(composite);
		toolkit.paintBordersFor(thisArea);
		
		// レイアウト設定
		GridLayout layout = new GridLayout();
		layout.numColumns = 2;
		layout.marginHeight = 2;
		thisArea.setLayout(layout);

		// レイアウト設定
		thisArea.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));

		// ラベル
		Label label = toolkit.createLabel(thisArea, Messages.getString("OverviewPart.22"), SWT.RIGHT); //$NON-NLS-1$
		label.setLayoutData(labelSizeMB);
		
		// 詳細入力テキスト
		descTxt = createText(thisArea, issueDisplayValue.getDescription(), SWT.MULTI | SWT.V_SCROLL | SWT.H_SCROLL);
		descTxt.setLayoutData(textSizeLL);
		
	}

	
	/**
	 * タイトルを表示する部品を生成します
	 *  
	 * @param		composite			親コンポジット
	 */
	private void createTitleArea(Composite composite) {

		// このエリアのコンポジット生成
		Composite thisArea = toolkit.createComposite(composite);
		toolkit.paintBordersFor(thisArea);
		
		// レイアウト設定
		GridLayout layout = new GridLayout();
		layout.numColumns = 2;
		layout.marginHeight = 2;
		thisArea.setLayout(layout);

		// レイアウト設定
		thisArea.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));

		// ラベル
		Label label = toolkit.createLabel(thisArea, Messages.getString("OverviewPart.23"), SWT.RIGHT); //$NON-NLS-1$
		label.setLayoutData(labelSizeS);
		
		// タイトル入力テキスト
		titleTxt = createText(thisArea,issueDisplayValue.getTitle());
		titleTxt.setLayoutData(textSizeL);
		
		// タイトルの文字数制限（45文字）
		titleTxt.setTextLimit(45);
	}


	/**
	 * カテゴリを表示する部品を生成します。
	 * 
	 * @param		composite			親コンポジット
	 */
	private void createCategoryArea(Composite composite) {

		// このエリアのコンポジット生成
		Composite thisArea = toolkit.createComposite(composite);
		toolkit.paintBordersFor(thisArea);
		
		// レイアウト設定
		GridLayout layout = new GridLayout();
		layout.numColumns = 3;
		layout.marginHeight = 1;
		thisArea.setLayout(layout);

		// レイアウト設定
		thisArea.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		
		// カテゴリラベル表示
		Label label = toolkit.createLabel(thisArea, Messages.getString("OverviewPart.24"), SWT.RIGHT); //$NON-NLS-1$
		label.setLayoutData(labelSizeS);
		
		// カテゴリコンボ生成
		categoryCmb = createCombo(thisArea, IssueUtility.getCategories(), issueDisplayValue.getCategory());
		categoryCmb.setLayoutData(comboSizeM);
		
		// カテゴリの文字数制限（20文字）
		categoryCmb.setTextLimit(20);

		// プルダウンメニュー設定へのリンク
		Link link = new Link(thisArea, SWT.NONE);
        link.setText(Messages.getString("OverviewPart.25")); //$NON-NLS-1$
        link.addSelectionListener(new org.eclipse.swt.events.SelectionAdapter() {
            public void widgetSelected(org.eclipse.swt.events.SelectionEvent e) {
            	// プロジェクト設定ダイアログを開きます
            	editPullDownMenuAction.run();
            }
        });
        
        // リンクテキストのレイアウト
        GridData gridData = new GridData(GridData.FILL_HORIZONTAL);
        gridData.horizontalIndent = 5;
        link.setLayoutData(gridData);
        link.setBackground(COLOR_WHITE);
	}
	
	/**
	 * 課題票IDを表示する部品を生成します。
	 * 
	 * @param		composite			親コンポジット
	 */
	private void createIssueIDArea(Composite composite) {

		// このエリアのコンポジット生成
		Composite thisArea = toolkit.createComposite(composite);
		toolkit.paintBordersFor(thisArea);
		
		// レイアウト設定
		GridLayout layout = new GridLayout();
		layout.numColumns = 2;
		layout.marginHeight = 0;
		thisArea.setLayout(layout);

		// レイアウト設定
		thisArea.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));

		// 課題票IDラベル表示
		Label label = toolkit.createLabel(thisArea, Messages.getString("OverviewPart.26"), SWT.RIGHT); //$NON-NLS-1$
		label.setLayoutData(labelSizeS);
		
		// 課題票ID入出力用テキスト
		issueIdLbl = new CLabel(thisArea, SWT.LEFT);
		issueIdLbl.setText(issueDisplayValue.getDisplayIssueID());
		issueIdLbl.setBackground(COLOR_WHITE);
		issueIdLbl.setLayoutData(labelSizeL);
	}

	
	/**
	 * 課題票のタイムスタンプを表示します。
	 * 
	 * @param		composite			親コンポジット
	 */
	private void createTimeStampArea(Composite composite) {
		
		// このエリアのコンポジット生成
		Composite thisArea = toolkit.createComposite(composite);
		
		// レイアウト設定
		GridLayout layout = new GridLayout();
		layout.numColumns = 2;
		layout.marginHeight = 0;
		thisArea.setLayout(layout);
		thisArea.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));

		// 起票者
		Label label = toolkit.createLabel(thisArea, Messages.getString("OverviewPart.27"), SWT.RIGHT); //$NON-NLS-1$
		label.setLayoutData(labelSizeS);
		
		createUserLbl = toolkit.createLabel(thisArea, issueDisplayValue.getCreateUser());
		createUserLbl.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));

		// 起票日時
		label = toolkit.createLabel(thisArea, Messages.getString("OverviewPart.28"), SWT.RIGHT); //$NON-NLS-1$
		label.setLayoutData(labelSizeS);
		
		createDateLbl = toolkit.createLabel(thisArea, issueDisplayValue.getCreateDate());
		createDateLbl.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));

		// 更新日時
		label = toolkit.createLabel(thisArea, Messages.getString("OverviewPart.29"), SWT.RIGHT); //$NON-NLS-1$
		label.setLayoutData(labelSizeS);

		updateDateLbl = toolkit.createLabel(thisArea, issueDisplayValue.getUpdateDate());
		updateDateLbl.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		
		// バージョン
		label = toolkit.createLabel(thisArea, Messages.getString("OverviewPart.30"), SWT.RIGHT); //$NON-NLS-1$
		label.setLayoutData(labelSizeS);
		
		versionLbl = toolkit.createLabel(thisArea, issueDisplayValue.getVersion());
		versionLbl.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));

		// 履歴ページへのリンク
		Link link = new Link(thisArea, SWT.NONE);
        link.setText(Messages.getString("OverviewPart.31")); //$NON-NLS-1$
        link.addSelectionListener(new org.eclipse.swt.events.SelectionAdapter() {
            public void widgetSelected(org.eclipse.swt.events.SelectionEvent e) {
            	// 「履歴ページ」に遷移します
                parentEditor.changeHistoryPage();
            }
        });
        
        // リンクテキストのレイアウト
        GridData gridData = new GridData(GridData.FILL_HORIZONTAL);
        gridData.horizontalSpan = 2;
        gridData.verticalIndent = 5;
        
        link.setLayoutData(gridData);
        link.setBackground(COLOR_WHITE);
	}
	

    /**
     * 課題票のステータス取得。
     * 選択状態がtrueの{@link IssueStatus 課題票ステータス}を取得します。
     * 
     * @return			課題票ステータス
     */
    private IssueStatus getIssueStatus() {
    	for (int idx = 0; idx < statusRdiList.size(); idx++) {
    		// ステータスラジオボタン取得
    		Button statusRdi = (Button)statusRdiList.get(idx);
    		if(statusRdi.getSelection()) {
    			// true状態のステータスを返す。
    			String description = statusRdi.getText();
    			return IssueStatus.getEnumByDescription(description);
    		}
		}
    	throw new RuntimeException(Messages.getString("OverviewPart.32")); //$NON-NLS-1$
    }


	/**
	 * 表示情報を更新します。
	 * 課題票を保存した後に、最新の情報（保存した情報）で課題票エディタの表示を更新します。
	 * 
	 * @param			savedIssue			保存した課題票情報
	 */
	public void postSave(Issue savedIssue) {
		IssueMultiPageEditorInput issueEditorInput = (IssueMultiPageEditorInput)getEditorInput();
		
		// 一度保存処理を行ったらオープンモードを「更新」に変える。課題票の表示情報も最新のものを設定する
		// !!!!!! この処理重要。気軽に移動・削除などしないこと !!!!!!
		issueEditorInput.resetIssue(savedIssue);

		// 更新情報の表示を更新する
		versionLbl.setText(issueDisplayValue.getVersion());
		updateDateLbl.setText(issueDisplayValue.getUpdateDate());
		createDateLbl.setText(issueDisplayValue.getCreateDate());
		createUserLbl.setText(issueDisplayValue.getCreateUser());
		
		// 課題票IDの表示を更新する
		int issueID = savedIssue.getIssueID();
		if(issueID != 0) {
			issueIdLbl.setText(savedIssue.getDisplayIssueID());
		}


		// 完了期限を判定
		int judge = BTSUtility.judgeLimit(savedIssue);
		setDeadLineMessage(judge);

		// ボタンの活性／非活性を設定する
		changeActivate();
		
		// 更新をEclipseに通知する
		dirty = false;
		firePropertyChange(PROP_DIRTY);
		fireIssueChange();
    }

	
	/**
	 * 課題票種別によりボタンの活性／非活性を切り替える
	 *
	 */
    private void changeActivate(){
    	// 課題票種別取得
    	String type = issueDisplayValue.getIssue().getType(); 
	    
    	// 課題票種別により処理を分ける
    	if(type.equals(IssueType.RELEASE_VALUE)) {
    		// 課題票種別：配布版
//	    	saveIssueAction.setEnabled(true);		// 保存：活性
	    	sendIssueCmdAction.setEnabled(true);	// 配信：活性
	    	voteLink.setEnabled(true);				// 投票：活性
			openDetailsExportDlgAction.setEnabled(true);
	    	
    	} else if(type.equals(IssueType.DRAFT_VALUE)) {
    		// 課題票種別：下書き
//	    	saveIssueAction.setEnabled(true);		// 保存：活性
	    	sendIssueCmdAction.setEnabled(true);	// 配信：活性
	    	voteLink.setEnabled(false);				// 投票：非活性
			openDetailsExportDlgAction.setEnabled(false);
	    
    	} else if(type.equals(IssueType.CONFLICT_VALUE)) {
    		// 課題票種別：競合
	    	saveIssueAction.setEnabled(false);		// 保存：非活性
	    	sendIssueCmdAction.setEnabled(false);	// 配信：非活性
	    	voteLink.setEnabled(false);				// 投票：非活性
			openDetailsExportDlgAction.setEnabled(true);
	    
    	} else if(type.equals(IssueType.GARBAGE_VALUE)) {
    		// 課題票種別：ごみ箱
	    	saveIssueAction.setEnabled(false);		// 保存：非活性
	    	sendIssueCmdAction.setEnabled(false);	// 配信：非活性
	    	voteLink.setEnabled(false);				// 投票：非活性
			openDetailsExportDlgAction.setEnabled(true);
	    }
    }

	private boolean validate() {
		
		String category = categoryCmb.getText().trim();
		
		// カテゴリは入力必須
		if(category.equals("")) {
			MessageDialog.openError
				( BtsPlugin.getInstance().getShell()
				, Messages.getString("OverviewPart.33") //$NON-NLS-1$
				, Messages.getString("OverviewPart.34")); //$NON-NLS-1$
			return false;
		}
		
		
		String title = titleTxt.getText().trim();
		
		// タイトルは入力必須
		if(title.equals("")) {
			MessageDialog.openError
				( BtsPlugin.getInstance().getShell()
				, Messages.getString("OverviewPart.35") //$NON-NLS-1$
				, Messages.getString("OverviewPart.36")); //$NON-NLS-1$
			return false;
		}


    	String deadLine = deadLineTxt.getText().trim();
    	
    	// 完了期限の日付チェック
    	if(!deadLine.equals("")) {

    		Date date = BTSUtility.createDate(deadLineTxt.getText());
    		if(date==null) {
    			MessageDialog.openError
				( BtsPlugin.getInstance().getShell()
				, Messages.getString("OverviewPart.37") //$NON-NLS-1$
				, Messages.getString("OverviewPart.38")); //$NON-NLS-1$
    			
    			return false;
    		}
    	}
    	
		return true;
	}

	
    /**
     * 入力情報を反映した課題票を取得します。
     * 
     * @return		入力を反映した課題票
     */
    Issue getModifiedIssue() {
    	if(!validate()) {
    		return null;
    	}
    	
    	// 更新情報を格納する課題票。新しくインスタンスを生成する。
    	Issue newIssue = new Issue();

    	// 更新前の情報を取得 
    	Issue oldIssue = issueDisplayValue.getIssue();

    	// 画面の入力値を取得して設定する
    	newIssue.setCategory(categoryCmb.getText());
    	newIssue.setTitle(titleTxt.getText());
    	newIssue.setDescription(descTxt.getText());
    	newIssue.setPriority(priorityCmb.getText());
    	newIssue.setStatus(getIssueStatus().getValue());
    	newIssue.setAssigned(userNameCmb.getText());
    	newIssue.setSecretBug(secretChk.getSelection());
    	newIssue.setSuspend(suspendChk.getSelection());
       	newIssue.setDeadline(BTSUtility.createDate(deadLineTxt.getText()));
    	
    	// 更新情報を格納。以前のものを引き継ぐ。これらは配信時に更新される。
    	newIssue.setCreateUser(oldIssue.getCreateUser());
    	newIssue.setCreateDate(oldIssue.getCreateDate());
    	newIssue.setUpdateDate(oldIssue.getUpdateDate());
    	
    	// 課題票IDは以前のものを引き継ぐ。IDの割り当ては配信時に行われる。
       	newIssue.setIssueID(oldIssue.getIssueID());
       	
       	// 課題票種別は以前のものを引き継ぐ。「配信時」または「下書き保存時」に更新する。
       	newIssue.setType(oldIssue.getType());
       	
       	// 課題票バージョンは以前のものを引き継ぐ。配信時にインクリメントされる。
       	newIssue.setVersion(oldIssue.getVersion());

		// FingerPrintを設定する
       	if(oldIssue.getFingerPrint()==null || "".equals(oldIssue.getFingerPrint())) {
	    	IssueUtility.createFingerPrint(newIssue);
       	} else {
	    	newIssue.setFingerPrint(oldIssue.getFingerPrint());
       	}
		
       	// 最新の添付ファイル情報を設定する。
       	newIssue.setAttachmentList( (ArrayList)attachViewer.getInput() );
       	
		return newIssue;
    }

    
    /*
     *  (非 Javadoc)
     * @see jp.valtech.bts.ui.editor.AbstractIssueEditorPart#notifyModifyInput()
     */
	protected void fireIssueChange() {

    	// 課題票種別取得
    	String type = issueDisplayValue.getIssue().getType(); 
	    
    	// 課題票種別により処理を分ける
    	if(type.equals(IssueType.RELEASE_VALUE)) {

    		// 保存アクションはdirty=trueの時のみ活性化
    		saveIssueAction.setEnabled(dirty);
	    
    	} else if(type.equals(IssueType.DRAFT_VALUE)) {

    		// 保存アクションはdirty=trueの時のみ活性化
    		saveIssueAction.setEnabled(dirty);
	    
    	} else if(type.equals(IssueType.CONFLICT_VALUE)) {
	    
    	} else if(type.equals(IssueType.GARBAGE_VALUE)) {

    	}
	}
   

	/** 
	 * ページ初期表示時のフォーカス。
	 * タイトルのテキストボックスにフォーカスする。
	 * 
	 * @see 		org.eclipse.ui.part.WorkbenchPart#setFocus()
	 */
	public void setFocus() {
		titleTxt.setFocus();
	}
	
	
	/**
 	 * このページに何らかの入力を行った場合はtrueを返します。
 	 * 
 	 * @return		true --このページに何らかの入力を行った場合。
 	 * 				false--このページに何も入力をしてない。
	 * @see 		org.eclipse.ui.ISaveablePart#isDirty()
	 */
    public boolean isDirty() {
    	return this.dirty;
    }

    
//	=============================================================================================
//	================================   ここからインナークラス   =================================
//	=============================================================================================


    /**
     * 「添付ファイル一覧テーブル」のコンテントプロバイダです。
     */
     private class AttachementTableContentProvider implements IStructuredContentProvider  {
    	/*
    	 *  (非 Javadoc)
    	 * @see org.eclipse.jface.viewers.IStructuredContentProvider#getElements(java.lang.Object)
    	 */
        public Object[] getElements(Object inputElement) {
    		return ((List)inputElement).toArray();
        }

        /*
         *  (非 Javadoc)
         * @see org.eclipse.jface.viewers.IContentProvider#dispose()
         */
        public void dispose() {
        	
        }

        /*
         *  (非 Javadoc)
         * @see org.eclipse.jface.viewers.IContentProvider#inputChanged(org.eclipse.jface.viewers.Viewer, java.lang.Object, java.lang.Object)
         */
        public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {
        }
    }
     
     
    /**
     * 「添付ファイル一覧テーブル」のラベルプロバイダです。 
     */
    private class AttachementTableLabelProvider extends LabelProvider implements ITableLabelProvider {

		/* (non-Javadoc)
		 * @see org.eclipse.jface.viewers.ITableLabelProvider#getColumnImage(java.lang.Object, int)
		 */
		public Image getColumnImage(Object element, int columnIndex) {
		    return null;
		}
		
		/* (non-Javadoc)
		 * @see org.eclipse.jface.viewers.ITableLabelProvider#getColumnText(java.lang.Object, int)
		 */
		public String getColumnText(Object element, int columnIndex) {
			
			Attachment attach = (Attachment)element ;
			switch (columnIndex) {
				case 0:
					
					return attach.getDisplayName();
			}
			return "";
		}
    }

    
	/**
	 * 「添付ファイルテーブル」の添付ファイル削除アクションです。
	 */
	private class RemoveAttachmentAction extends Action {
	
		/**
		 * アクションを生成します。
		 *
		 */
		RemoveAttachmentAction() {
			super();
			setText(Messages.getString("OverviewPart.39")); //$NON-NLS-1$
		}
		
		/*
		 *  (非 Javadoc)
		 * @see org.eclipse.jface.action.IAction#run()
		 */
		public void run() {
			Attachment attachment = getSelectionRecord();
			if(attachment!=null) {
				// 実データを削除
				((List)attachViewer.getInput()).remove(attachment);
				
		   		// 表示を更新
		   		attachViewer.remove(attachment);

		   		// 更新を通知する
				dirty = true;
				OverviewPart.this.firePropertyChange(PROP_DIRTY);
				fireIssueChange();
			}

		}
	}
	

}

/*
 * FormToolKit + MultipageEditorを使った画面の参考ソース
 * org.eclipse.wst.server.ui
 *  └org.eclipse.wst.server.ui.internal.editor.OverviewEditorPart
 * （CVS『:pserver:anonymous@proxy.eclipse.org:80/cvsroot/webtools』から取得）
 * 
 */

