/***********************************************************************
 * Copyright(C) 2006 Valtech Co.,Ltd.
 * All Rights Reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 ***********************************************************************/
package jp.valtech.bts.ui.editor;


import jp.valtech.bts.data.CommentHistory;
import jp.valtech.bts.data.Issue;
import jp.valtech.bts.data.IssueHistory;
import jp.valtech.bts.data.IssueType;
import jp.valtech.bts.ui.BtsPlugin;
import jp.valtech.bts.ui.action.SaveIssueAction;
import jp.valtech.bts.util.Logging;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.graphics.Image;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.part.MultiPageEditorPart;

/**
 * 課題票エディタです。
 * 課題票エディタは{@link org.eclipse.ui.part.MultiPageEditorPart マルチページ}を拡張してあり、以下のページで構成されます。<br>
 * <ul>
 *   <li>1ページ目：{@link jp.valtech.bts.ui.editor.OverviewPart 概要ページ}
 *   <li>2ページ目：{@link jp.valtech.bts.ui.editor.CommentPart コメントページ}
 *   <li>3ページ目：{@link jp.valtech.bts.ui.editor.HistoryPart 更新履歴ページ}
 * </ul>
 * 
 * @author		<A href="mailto:m_sugitou@valtech.jp">M.Sugito</A>
 * @version	Ver.0.8
 */
public class IssueMultiPageEditor extends MultiPageEditorPart implements Logging {
	
	/** 課題票で使うアイコン：通常の課題票 */
	public static final Image IMG_ISSUE = BtsPlugin.getInstance().getImage("issue.gif");
	/** 課題票で使うアイコン：競合した課題票 */
	public static final Image IMG_ISSUE_CONFLICT = BtsPlugin.getInstance().getImage("issue_conflict.gif");
	/** 課題票で使うアイコン：下書きの課題票 */
	public static final Image IMG_ISSUE_DRAFT = BtsPlugin.getInstance().getImage("issue_draft.gif");
	/** 課題票で使うアイコン：ゴミ箱の課題票 */
	public static final Image IMG_ISSUE_GARBAGE = BtsPlugin.getInstance().getImage("issue_garbage.gif");
	

	/** 1ページ目のページインデックス */
	private static final int PAGE_INDEX_OVERVIEW = 0;
	/** 2ページ目のページインデックス */
	private static final int PAGE_INDEX_COMMENT = 1;
	/** 3ページ目のページインデックス */
	private static final int PAGE_INDEX_HISTORY = 2;

	/** 1ページ目のインスタンス */
	private OverviewPart overviewPage ;
	/** 2ページ目のインスタンス */
	private CommentPart commentPage ;
	/** 3ページ目のインスタンス */
	private HistoryPart historyPage ;
	
	
	/**
	 * Creates a multi-page editor
	 */
	public IssueMultiPageEditor() {
		super();
	} 
	
	/* (非 Javadoc)
	 * @see org.eclipse.ui.IEditorPart#init(org.eclipse.ui.IEditorSite, org.eclipse.ui.IEditorInput)
	 */
	public void init(IEditorSite site, IEditorInput input)	throws PartInitException {
		setSite(site);
		setInput(input);
		// IssueMultiPageEditorInputのみ受け付けます。
		if(input instanceof IssueMultiPageEditorInput) {

			// 課題票タイトルをタブに表示
			IssueMultiPageEditorInput issueEditorInput = (IssueMultiPageEditorInput)input;
			
			setPartName(getTitleStr(issueEditorInput.getIssue()) );
			setTitleImage( getImage(issueEditorInput.getIssue()) );
			
		} else {
			throw new PartInitException(Messages.getString("IssueMultiPageEditor.0")); //$NON-NLS-1$
		}
	}
	
	
	/**
	 * 課題票エディタを生成します。
	 * 
	 */
	protected void createPages() {
		try {
			// マルチページ１つ目のタブ
			overviewPage = new OverviewPart(this);
			addPage(overviewPage, getEditorInput());
			setPageText(PAGE_INDEX_OVERVIEW, OverviewPart.TAB_TITLE);

			// マルチページ２つ目のタブ
			commentPage = new CommentPart(this);
			addPage(commentPage, getEditorInput());
			setPageText(PAGE_INDEX_COMMENT, CommentPart.TAB_TITLE);
			
			// マルチページ２つ目のタブ
			historyPage = new HistoryPart();
			addPage(historyPage, getEditorInput());
			setPageText(PAGE_INDEX_HISTORY, HistoryPart.TAB_TITLE);

		} catch (Exception e) {
			String msg = Messages.getString("IssueMultiPageEditor.1"); //$NON-NLS-1$
			logger.fatal(msg, e);
			BtsPlugin.getInstance().error(msg, e);
		}
	}
	
	
	/**
	 * 課題票エディタに入力した値で課題票情報を取得します。
	 * 
	 * @return		課題票情報
	 */
	public Issue getModifiedIssue() {
        return overviewPage.getModifiedIssue();
	}
	
	public String getNewComment(){
		return commentPage.getNewComment();
	}
	

	/**
	 * {@link OverviewPart 概要ページ}を最新表示にします。
	 * 課題票の更新情報を保存後に呼ばれる処理です。
	 * 保存には{@link SaveIssueAction 「下書きとして保存」処理}と{@link jp.valtech.bts.ui.action.SendReleaseIssueAction 「配信」処理}があります。
	 * 
	 * @param			issue			最新の課題票情報
	 */
	public void postIssueSave(Issue issue) {
		// 「概要」ページ更新
		overviewPage.postSave(issue);
		setTitleImage(getImage(issue));
		setPartName(getTitleStr(issue));
	}
	
	
	/**
	 * {@link CommentPart コメントページ}を最新表示に更新します。
	 * 課題票配信処理後に呼ばれます。 
	 * 
	 * @param		commentHistories		最新のコメント履歴
	 */
	public void postCommentHistorySave(CommentHistory[] commentHistories) {
		commentPage.postSave(commentHistories);
	}
	
	
	/**
	 * {@link HistoryPart 履歴ページ}を最新表示に更新します。
	 * 課題票配信処理後に呼ばれます。 
	 * 
	 * @param		histories		最新の課題票履歴
	 */
	public void postIssueHistorySave(IssueHistory[] issueHistories) {
		historyPage.postSave(issueHistories);
	}

	
    /**
     * 「履歴」ページに遷移します。
     *
     */
    void changeHistoryPage() {
    	setActivePage(PAGE_INDEX_HISTORY);
    	pageChange(PAGE_INDEX_HISTORY);
    }
	
    /**
     * 「概要」ページに遷移します。
     *
     */
    void changeOverviewPage() {
    	setActivePage(PAGE_INDEX_OVERVIEW);
    	pageChange(PAGE_INDEX_OVERVIEW);
    }

 
    /**
	 * The <code>MultiPageEditorPart</code> implementation of this 
	 * <code>IWorkbenchPart</code> method disposes all nested editors.
	 * Subclasses may extend.
	 */
	public void dispose() {
		super.dispose();
	}
	
	
	/**
	 * 課題票を保存します。
	 */
	public void doSave(IProgressMonitor monitor) {

		// 課題票種別取得
		IssueMultiPageEditorInput editorInput = (IssueMultiPageEditorInput)getEditorInput();
		String type = editorInput.getIssue().getType();
	
    	// 課題票種別により処理を分ける
    	if(type.equals(IssueType.RELEASE_VALUE)) {
    		// 課題票種別：配布版
        	SaveIssueAction saveIssueAction = new SaveIssueAction();
    		saveIssueAction.run();
	    
    	} else if(type.equals(IssueType.DRAFT_VALUE)) {
    		// 課題票種別：下書き
        	SaveIssueAction saveIssueAction = new SaveIssueAction();
    		saveIssueAction.run();
	    
    	} else if(type.equals(IssueType.CONFLICT_VALUE)) {
    		// 課題票種別：競合
			MessageDialog.openInformation(
					BtsPlugin.getInstance().getShell(),
					"Infomation", 
					Messages.getString("IssueMultiPageEditor.2")); //$NON-NLS-1$
	    
    	} else if(type.equals(IssueType.GARBAGE_VALUE)) {
    		// 課題票種別：ごみ箱
			MessageDialog.openInformation(
					BtsPlugin.getInstance().getShell(),
					"Infomation", 
					Messages.getString("IssueMultiPageEditor.3")); //$NON-NLS-1$
	    }
	}

	
	/*
	 *  (非 Javadoc)
	 * @see org.eclipse.ui.ISaveablePart#doSaveAs()
	 */
	public void doSaveAs() {
		// 何もしない
	}

	/**
	 * {@link #doSaveAs()}は使わないので必ずfalseを返す。
	 * 
	 * @see org.eclipse.ui.ISaveablePart#isSaveAsAllowed()
	 */
	public boolean isSaveAsAllowed() {
		// #doSaveAs()を使わない
		return false;
	}
	
	/**
	 * エディタのタブに表示するアイコンイメージを返します。
	 * 課題票の種類によってアイコンを変えます。
	 *
	 * @param		issue			課題票			
	 * @return		課題票エディタ用アイコンイメージ
	 */
	private Image getImage(Issue issue) {
		
		if(IssueType.CONFLICT_VALUE.equals(issue.getType())) {
			// 衝突の課題票
			return IMG_ISSUE_CONFLICT;
		
		} else if (IssueType.DRAFT_VALUE.equals(issue.getType())) {
			// 下書きの課題票
			return IMG_ISSUE_DRAFT;
		
		} else if (IssueType.GARBAGE_VALUE.equals(issue.getType())) {
			// ごみ箱の課題票
			return IMG_ISSUE_GARBAGE;
		
		} else {
			// 配布済みの課題票
			return IMG_ISSUE;
		}
	}

	/**
	 * エディタのタブに表示するタイトル文字列を返します。
	 * 
	 * @param		issue			課題票			
	 * @return		タイトル文字列
	 */
	private String getTitleStr(Issue issue) {
		
		IssueMultiPageEditorInput issueEditorInput = (IssueMultiPageEditorInput)getEditorInput();
		
		// 「既存の課題票」かつ「下書きの課題票」は『【下書き】』と頭につける
		if(issueEditorInput.getOpenMode()== IssueMultiPageEditorInput.OPEN_EXIST_ISSUE
		 && IssueType.DRAFT_VALUE.equals(issue.getType()) ) {
			return Messages.getString("IssueMultiPageEditor.4") + issueEditorInput.getName(); //$NON-NLS-1$
		}
		return issueEditorInput.getName();
	}
}

/*
 * 参考ソース
 * org.eclipse.pde.ui
 * 　└org.rclipse.pde.ui_3.1.1.jar
 * 　　　└org.eclipse.pde.internal.ui.editor.plugin.OverViewPage.class　
 * 
 * 
 */
