/***********************************************************************
 * Copyright(C) 2006 Valtech Co.,Ltd.
 * All Rights Reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 ***********************************************************************/
package jp.valtech.bts.ui.editor;

import jp.valtech.bts.data.Issue;
import jp.valtech.bts.ui.BtsPlugin;
import jp.valtech.bts.util.Logging;

import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.window.IShellProvider;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;

/**
 * 課題票詳細出力設定用のダイアログです。
 * {@link jp.valtech.bts.ui.editor.OverviewPart 課題票エディタの「概要」ページ}から呼ばれるダイアログです。
 * 
 * @author		<A href="mailto:s_imai@valtech.jp">S.Imai</A>
 * @version	Ver.0.8
 */
public class IssueDetailsExportDialog extends Dialog implements Logging {

    /** 「ヘッダ文字列」用テキストボックス */
	Text headerTxt;
    /** 「フッタ文字列」用テキストボックス */
	Text footerTxt;
    
	/** 「ファイル形式：HTML」用ラジオボタン */
    Button fileHTMRdi;
	/** 「ファイル形式：PDF」用ラジオボタン */
    private Button filePDFRdi;
	/** 「ファイル形式：XML」用ラジオボタン */
    Button fileXMLRdi;
	/** 「ファイル形式：CSV」用ラジオボタン */
    private Button fileCSVRdi;
	
    /**「添付ファイル一覧テーブル」用テーブルビューア */
    TableViewer attachViewer;
    
	/** マルチページエディタの親 */
	private IssueMultiPageEditor parentEditor;


	/**
	 * 親クラスにシェルを設定します。
	 * 
	 * @param			parentShell			シェル
	 * @param			attachViewer		添付Viewer
	 * @param			mode				呼び出し元識別モード
	 */
	public IssueDetailsExportDialog(Shell parentShell, TableViewer attachViewer, IssueMultiPageEditor parentEditor) {
		super(parentShell);
		this.attachViewer = attachViewer;
		this.parentEditor = parentEditor;
	}

	/**
	 * 親クラスにプロバイダを設定します。
	 * 
	 * @param			parentShell
	 */
	public IssueDetailsExportDialog(IShellProvider parentShell) {
		super(parentShell);
	}

    /**
     * 当ダイアログの設定します。タイトルを設定します。
     * 
     * @param			newShell			該ダイアログのシェル
     */
    protected void configureShell(Shell newShell) {
        super.configureShell(newShell);
        newShell.setText(Messages.getString("IssueDetailsExportDialog.0"));  //$NON-NLS-1$
    }



    /**
     * フィルターパターン編集用のダイアログを生成します。
     * 
     * @param			parent					親コンポジット
     * @return			当ダイアログのコントロールインスタンス
     * @see				Dialog#createDialogArea(org.eclipse.swt.widgets.Composite)
     */
    protected Control createDialogArea(Composite parent) {
        Composite composite = (Composite) super.createDialogArea(parent);
        try {
        	
        	// 「ファイル出力形式」設定のUIを生成
        	createFileFormatArea(composite);
        	
        	// 「ヘッダ」「フッタ」設定のUIを生成
            createHeaderArea(composite);

            // セパレータ生成
            createSeparatorLine(composite);
            
        }catch (Exception e) {
			String msg = Messages.getString("IssueDetailsExportDialog.9"); //$NON-NLS-1$
			logger.fatal(msg, e);
			BtsPlugin.getInstance().error(msg, e);
		}
        return composite;
    }
    
    /**
     * 「出力項目選択」のUIを生成します。
     * 親コンポジットに対して「出力項目選択」のUIを生成します。<br>
     * 
     * @param			parent						親コンポジット
     * @throws			Exception
     */
    private void createFileFormatArea(Composite parent) throws Exception {

    	// このエリアのコンポジットを生成。
        Composite thisArea = new Composite(parent, SWT.NONE);

        // レイアウトを設定。
        thisArea.setLayout(new GridLayout());
        thisArea.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
        
        // 「出力ファイル形式」グループの生成
    	Group group = new Group(thisArea, SWT.NONE);
        group.setText(Messages.getString("IssueDetailsExportDialog.1")); //$NON-NLS-1$

        // 「出力ファイル形式」のレイアウト設定
    	GridLayout layout = new GridLayout();
        layout.numColumns = 4;
    	group.setLayout(layout);
    	group.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));

    	// 「HTML」のラジオボタン
        fileHTMRdi = new Button(group, SWT.RADIO);
        fileHTMRdi.setText("HTML");

    	// 「XML」のラジオボタン
        fileXMLRdi = new Button(group, SWT.RADIO);
        fileXMLRdi.setText("XML");

    	// 「PDF」のラジオボタン
        filePDFRdi = new Button(group, SWT.RADIO);
        filePDFRdi.setText("PDF");
        filePDFRdi.setEnabled(false);

    	// 「CSV」のラジオボタン
        fileCSVRdi = new Button(group, SWT.RADIO);
        fileCSVRdi.setText("CSV");
        fileCSVRdi.setEnabled(false);

    }
    
    
    /**
     * 「ヘッダ」「フッタ」のUIを生成。
     * 親コンポジットに「ヘッダ」「フッタ」設定のUIを生成します。
     * 
     * @param			parent					親コンポジット
     * @throws			Exception
     */
    private void createHeaderArea(Composite parent) throws Exception {
    	
    	// このエリアのコンポジットを生成。
    	Composite thisArea = new Composite(parent, SWT.NONE);

    	// レイアウトを設定。
    	GridLayout layout = new GridLayout();
    	layout.numColumns = 2;
    	thisArea.setLayout(layout);
        thisArea.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));

        // テキストボックスのサイズ指定
        GridData gridData = new GridData(GridData.FILL); 
        gridData.widthHint = 220;

        // 「ヘッダ」ラベル
        Label label = new Label(thisArea, SWT.NONE);
        label.setText(Messages.getString("IssueDetailsExportDialog.2")); //$NON-NLS-1$

        // 「ヘッダ」テキストボックス
        headerTxt = new Text(thisArea, SWT.BORDER);
        headerTxt.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
        headerTxt.setLayoutData(gridData);

    
        // 「フッタ」ラベル
        label = new Label(thisArea, SWT.NONE);
        label.setText(Messages.getString("IssueDetailsExportDialog.3")); //$NON-NLS-1$

        // 「フッタ」テキストボックス
        footerTxt = new Text(thisArea, SWT.BORDER);
        footerTxt.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
        footerTxt.setLayoutData(gridData);
    }

    /**
     * セパレータラインを生成します。
     * 親コンポジットに対してセパレータラインを生成します。
     * 
     * @param			parent						親コンポジット
     */
    private void createSeparatorLine(Composite parent) {
        // 親コンポーネントにラベルを生成
        Label separator = new Label(parent, SWT.HORIZONTAL | SWT.SEPARATOR);
        
        // レイアウトを設定。グリッドレイアウトを使う。
        GridData gd = new GridData(GridData.FILL_HORIZONTAL);
        gd.horizontalSpan = 1;
        separator.setLayoutData(gd);
    }

    /**
     * ボタンバーにボタンを生成します。
     * 左側のボタンのラベルを「ファイル出力」にするためにオーバーライドしています。
     * 
     * @param			parent						親コンポジット
     */
    protected void createButtonsForButtonBar(Composite parent) {
    	
        createButton(parent, IDialogConstants.OK_ID, Messages.getString("IssueDetailsExportDialog.4"), true); //$NON-NLS-1$
        
        createButton(parent, IDialogConstants.CANCEL_ID,
                IDialogConstants.CANCEL_LABEL, false);
        
    }

    /**
     * 「ファイル出力」ボタンを押したときの処理です。
     */
	protected void okPressed() {
		
		// ファイル保存ダイアログ生成
		FileDialog fileDialog = new FileDialog(this.getShell(), SWT.SAVE);

		// 拡張子
		String extension;
		
		// 出力形式がHTMLの場合
		if(fileHTMRdi.getSelection()) {
			
			// 拡張子設定
			extension = ".html";		
			String[] extensions = { "*.html", "*.*" };
			fileDialog.setFilterExtensions(extensions);
			
			// 「ファイルの種類」設定
			String[] filterNames = { Messages.getString("IssueDetailsExportDialog.5"), Messages.getString("IssueDetailsExportDialog.6") }; //$NON-NLS-1$ //$NON-NLS-2$
			fileDialog.setFilterNames(filterNames);
			
		}
		
		// 出力形式がXMLの場合
		else {
			
			// 拡張子設定
			extension = ".xml";
			String[] extensions = { "*.xml", "*.*" };
			fileDialog.setFilterExtensions(extensions);
			
			// 「ファイルの種類」設定
			String[] filterNames = { Messages.getString("IssueDetailsExportDialog.7"), Messages.getString("IssueDetailsExportDialog.8") }; //$NON-NLS-1$ //$NON-NLS-2$
			fileDialog.setFilterNames(filterNames);
			
		}
		
		// タイトルをファイル名のデフォルト値とする
    	Issue issue = parentEditor.getModifiedIssue();
		fileDialog.setFileName(issue.getTitle() + extension);
		
		// ファイル保存ダイアログを開き、ファイルパスを取得
		String fileName = fileDialog.open();
		
		// フォルダパス取得
		String filterPath = fileDialog.getFilterPath();

		// ファイルパスがnullの場合は処理を抜けます
		if(fileName == null) {
			return;
		}
		
		// ファイルに出力
		IssueDetailsExport issueDetailsExport = new IssueDetailsExport(this, fileName, filterPath);
		issueDetailsExport.exportRun();
		
		super.okPressed();
		
	}
    
    
}
