/***********************************************************************
 * Copyright(C) 2006 Valtech Co.,Ltd.
 * All Rights Reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 ***********************************************************************/
package jp.valtech.bts.ui.action;

import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.List;

import jp.valtech.bts.command.client.Join;
import jp.valtech.bts.command.client.SyncIssue;
import jp.valtech.bts.data.NetworkConfig;
import jp.valtech.bts.network.ClientInfo;
import jp.valtech.bts.network.ClientList;
import jp.valtech.bts.ui.BtsPlugin;
import jp.valtech.bts.util.BTSUtility;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.dialogs.ProgressMonitorDialog;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.ColumnWeightData;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.TableLayout;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;

/**
 * 他の端末と同期をとり、ローカルデータを最新状態にします。
 * 同期相手の端末は、ユーザがダイアログで指定します。
 * 
 * @author		<A href="mailto:m_sugitou@valtech.jp">M.Sugito</A>
 * @version	Ver.0.8
 */
public class SyncSelectionRemoteAction extends Action {

	/** 活性時のアイコン */
	private static final ImageDescriptor THIS_IMAGE = BtsPlugin.getInstance().getImageDescriptor("sync2.gif"); 
	/** 非活性時のアイコン */
	private static final ImageDescriptor THIS_IMAGE_DISABLE = BtsPlugin.getInstance().getImageDescriptor("sync2_disable.gif"); 


	/** 同期コマンドが成功したかどうか */
	private boolean finished = false;

	/**
	 * アクションを生成します。
	 * 
	 */
	public SyncSelectionRemoteAction() {
		super();
		setText(Messages.getString("SyncSelectionRemoteAction.0")); //$NON-NLS-1$
		setToolTipText(Messages.getString("SyncSelectionRemoteAction.1"));  //$NON-NLS-1$

		setImageDescriptor(THIS_IMAGE);
		setDisabledImageDescriptor(THIS_IMAGE_DISABLE);
	}

	
	/**
	 * リモート端末選択のダイアログを開きます。
	 */
	public void run() {
		// 正確な配信先を取得するため、一旦クライアントリストをクリアする。
		ClientList.clear();
		ClientList.addMyInfo();
		
		// クリア後Joinコマンドを送る
    	Join join = new Join( new NetworkConfig());
		join.execute();

		// 参加中の端末からのレスポンスが返ってくるまで、プログレスバーを表示して時間を稼ぐ。
		if(!progress()){
			// 途中キャンセルボタンを押した場合
			return;
		}

		// リモート端末選択ダイアログを開きます
		SyncSelectionRemoteDialog dialog 
			= new SyncSelectionRemoteDialog(BtsPlugin.getInstance().getShell());
		dialog.open();
	}
	
	/**
	 * リモート端末検索中に時間を稼ぐプログレスバーを表示する。
	 *
	 */
	private boolean progress() {
		try {
			
	    	// プログレスダイアログを生成
			IRunnableWithProgress progress = new IRunnableWithProgress() {
				public void run(IProgressMonitor monitor) throws InvocationTargetException, InterruptedException {

					// プログレス・モニタの開始の開始
					monitor.beginTask(Messages.getString("SyncSelectionRemoteAction.3"), 100); //$NON-NLS-1$
					monitor.subTask(Messages.getString("SyncSelectionRemoteAction.4")); //$NON-NLS-1$
					for (int i = 0; i < 100; i++) {
						
						if (monitor.isCanceled()) {
							throw new InterruptedException(Messages.getString("SyncSelectionRemoteAction.5")); //$NON-NLS-1$
						}
						
						Thread.sleep(40);
						monitor.worked(1);
					}
					monitor.done();
				}
			};

			// プログレスダイアログの表示
			ProgressMonitorDialog dialog = new ProgressMonitorDialog(BtsPlugin.getInstance().getShell());
			dialog.run(true, true, progress);

			return true;
		} catch (InvocationTargetException e) {
			BtsPlugin.getInstance().errorlog(e);
		} catch (InterruptedException e) {
			// キャンセル後の処理
			MessageDialog.openWarning
				 ( BtsPlugin.getInstance().getShell()
				 , Messages.getString("SyncSelectionRemoteAction.3") //$NON-NLS-1$
				 , e.getMessage()); 
		}
		return false;
	}

	
	
	/**
	 * リモート端末の選択ダイアログを生成します。
	 * 
	 * @author <A href="mailto:s_imai@valtech.jp">$Author$</A>
	 * @version $Rev$
	 */
	public class SyncSelectionRemoteDialog extends Dialog {

		/** リモート端末ビューア */
		private TableViewer viewer;

		/**
		 * @param parentShell
		 */
		public SyncSelectionRemoteDialog(Shell parentShell) {
			super(parentShell);
		}

		/*
		 * (非 Javadoc)
		 * 
		 * @see org.eclipse.jface.window.Window#configureShell(org.eclipse.swt.widgets.Shell)
		 */
		protected void configureShell(Shell newShell) {
			super.configureShell(newShell);
			newShell.setText(Messages.getString("SyncSelectionRemoteDialog.0")); //$NON-NLS-1$
		}

		/*
		 * (非 Javadoc)
		 * 
		 * @see org.eclipse.jface.dialogs.Dialog#createDialogArea(org.eclipse.swt.widgets.Composite)
		 */
		protected Control createDialogArea(Composite parent) {

			// コンポジット生成
			Composite composite = (Composite) super.createDialogArea(parent);

			// リモート端末選択テーブルのUIを生成
			createTableArea(composite);

			return composite;

		}

		/**
		 * リモート端末選択テーブルのUIを生成します。
		 * 
		 * @param parent 親コンポジット
		 */
		private void createTableArea(Composite parent) {

			// このパーツのレイアウト設定。グリッドレイアウトを使う
			Composite thisArea = new Composite(parent, SWT.NONE);
			thisArea.setLayout(new GridLayout());

			// テーブルタイトルの設定
			Label label = new Label(thisArea, SWT.NONE);
			label.setText(Messages.getString("SyncSelectionRemoteDialog.1")); //$NON-NLS-1$

			// テーブルの表示設定
			Table table = new Table(thisArea, SWT.H_SCROLL | SWT.V_SCROLL | SWT.FULL_SELECTION | SWT.BORDER);
			table.setLinesVisible(false);
			table.setHeaderVisible(true);

			// テーブルレイアウトの設定
			TableLayout tableLayout = new TableLayout();
			table.setLayout(tableLayout);

			// 1カラム目（アドレス）の設定
			tableLayout.addColumnData(new ColumnWeightData(80, true));
			TableColumn tcAddress = new TableColumn(table, SWT.NONE, 0);
			tcAddress.setText(Messages.getString("SyncSelectionRemoteDialog.2")); //$NON-NLS-1$

			// 2カラム目（ユーザ）の設定
			tableLayout.addColumnData(new ColumnWeightData(110, true));
			TableColumn tcUser = new TableColumn(table, SWT.NONE, 1);
			tcUser.setText(Messages.getString("SyncSelectionRemoteDialog.3")); //$NON-NLS-1$

			// テーブルのビューアオブジェクト設定
			viewer = new TableViewer(table);

			// テーブルの横幅・高さの調整
			GridData gridData = new GridData(GridData.FILL_HORIZONTAL);
			gridData.heightHint = 150;
			gridData.widthHint = 230;
			viewer.getControl().setLayoutData(gridData);

			// テーブルにプロバイダークラスを設定
			viewer.setContentProvider(getTableContentProvider());
			viewer.setLabelProvider(getTableLabelProvider());
			
			// ユーザ情報取得
			ClientInfo[] clientInfo = ClientList.getClientInfoList();
			
			// 自分自身を除外したユーザ情報
			List clientInfoExceptMe = new ArrayList();
			
			// 取得したユーザ分処理する
			for(int i=0; i<clientInfo.length; i++) {
				
				// ユーザ情報取得
				ClientInfo info = clientInfo[i];
				
				// ユーザ情報が自分自身では無い場合
				if(!BTSUtility.isLocalHost(info)) {
					// 自分自身を除外したユーザ情報にセット
					clientInfoExceptMe.add(info);
				}
				
			}

			// テーブルにデータをセット
			viewer.setInput(clientInfoExceptMe);
			
			// テーブルの1行目を選択状態にする
			viewer.getTable().setSelection(0);

			// ビューアに選択変更リスナを追加
			viewer.addSelectionChangedListener(new ISelectionChangedListener(){

				/*
				 *  (非 Javadoc)
				 * @see org.eclipse.jface.viewers.ISelectionChangedListener#selectionChanged(org.eclipse.jface.viewers.SelectionChangedEvent)
				 */
				public void selectionChanged(SelectionChangedEvent event) {
					
					// 「接続」ボタンの活性・非活性切換え
					getButton(IDialogConstants.OK_ID).setEnabled(viewer.getTable().getSelectionCount() != 0);
					
				}
				
			});
			
		}


		/**
		 * 「リモート端末テーブル」のコンテントプロバイダを返します。
		 * 
		 * @return 「リモート端末テーブル」のコンテントプロバイダ
		 */
		private IStructuredContentProvider getTableContentProvider() {
			return new IStructuredContentProvider() {

				public Object[] getElements(Object inputElement) {
					return ((List)inputElement).toArray();
				}

				public void dispose() {
				}

				public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {
				}
			};
		}


		/**
		 * {@link TableLabelProvider 「リモート端末一覧テーブル」のラベルプロバイダ}を返します。
		 * 
		 * @return 「リモート端末一覧テーブル」のラベルプロバイダ
		 */
		private ILabelProvider getTableLabelProvider() {
			return new TableLabelProvider();
		}

		/**
		 * 「リモート端末一覧テーブル」のラベルプロバイダです。
		 */
		private class TableLabelProvider extends LabelProvider implements ITableLabelProvider {

			private Image image = BtsPlugin.getInstance().getImage("sync1.gif");

			/*
			 * (非 Javadoc)
			 * 
			 * @see org.eclipse.jface.viewers.ITableLabelProvider#getColumnImage(java.lang.Object,
			 *      int)
			 */
			public Image getColumnImage(Object element, int columnIndex) {
				switch (columnIndex) {
				case 0:
					return image;
				}
				return null;
			}

			/*
			 * (非 Javadoc)
			 * 
			 * @see org.eclipse.jface.viewers.ITableLabelProvider#getColumnText(java.lang.Object,
			 *      int)
			 */
			public String getColumnText(Object element, int columnIndex) {
				switch (columnIndex) {
				case 0:
					return ((ClientInfo) element).getClientAddress();
				case 1:
					return ((ClientInfo) element).getClientName();
				}
				return "";

			}

		}

		/**
		 * ボタンバーのボタンを生成します。 左ボタンのラベルを「接続」にするためにオーバーライドしています。
		 * この「接続」ボタンは送信先が存在するときのみ活性します。
		 * 
		 * @param parent 親コンポジット
		 */
		protected void createButtonsForButtonBar(Composite parent) {
			createButton(parent, IDialogConstants.OK_ID, Messages.getString("SyncSelectionRemoteDialog.4"), true); //$NON-NLS-1$
			getButton(IDialogConstants.OK_ID).setEnabled(viewer.getTable().getSelectionCount() != 0);
			createButton(parent, IDialogConstants.CANCEL_ID, IDialogConstants.CANCEL_LABEL, false);
		}

		/**
		 * 接続ボタン押下。
		 * 指定された端末と同期をとります。
		 * 
		 */
		protected void okPressed() {

			// 選択しているレコードを取得します。
			ISelection selection = viewer.getSelection();
			Object obj = ((IStructuredSelection)selection).getFirstElement();
			ClientInfo clientInfo = (ClientInfo)obj;
			if(clientInfo == null) {
				return;
			}

			super.okPressed();

			// 同期処理のクライアントコマンド実行
			SyncIssue syncIssue = new SyncIssue( new NetworkConfig()
					   , clientInfo.getClientAddress()
					   , clientInfo.getPort().intValue());

			syncIssue.setRemoteUserName(clientInfo.getClientName());
			syncIssue.execute();
			
			finished = syncIssue.isFinished(); 
		}

		/**
		 * キャンセルボタン押下。
		 * 
		 */
		protected void cancelPressed() {
	    	super.cancelPressed();
	    	
			// 同期処理を行いませんでした。のダイアログ
			 MessageDialog.openWarning
	 			( BtsPlugin.getInstance().getShell()
	            , Messages.getString("SyncSelectionRemoteDialog.5") //$NON-NLS-1$
	            , Messages.getString("SyncSelectionRemoteDialog.6")); //$NON-NLS-1$
	    }
	}


	/**
	 * 同期処理の実行結果を取得する。
	 * 
	 * @return			true --同期処理実行成功。
	 * 					false--同期処理実行失敗。
	 */
	public boolean isFinished() {
		return this.finished;
	}
}
