/***********************************************************************
 * Copyright(C) 2006 Valtech Co.,Ltd.
 * All Rights Reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 ***********************************************************************/
package jp.valtech.bts.ui.action;

import java.util.ArrayList;

import jp.valtech.bts.data.Issue;
import jp.valtech.bts.data.IssueStatus;
import jp.valtech.bts.data.IssueType;
import jp.valtech.bts.facade.ModifyIssueFacade;
import jp.valtech.bts.ui.BtsPlugin;
import jp.valtech.bts.ui.editor.IssueMultiPageEditorInput;
import jp.valtech.bts.util.Logging;

import org.eclipse.jface.action.Action;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.resource.ImageDescriptor;
 
/**
 * 課題票を開くアクションです。
 * 
 * @author		<A href="mailto:m_sugitou@valtech.jp">M.Sugito</A>
 * @version	Ver.0.8
 */
public class OpenIssueAction extends Action implements Logging {

	/** 課題票のアイコン */
	private static final ImageDescriptor EXIST_ISSUE = BtsPlugin.getInstance().getImageDescriptor("issue.gif"); 
	/** 課題票起票のアイコン */
	private static final ImageDescriptor NEW_ISSUE = BtsPlugin.getInstance().getImageDescriptor("issue_new.gif"); 
	/** 課題票起票(DISABLE)のアイコン */
	private static final ImageDescriptor NEW_ISSUE_DISABLE = BtsPlugin.getInstance().getImageDescriptor("issue_new_disable.gif"); 

	/** 新規に課題票を開く(起票) */
	public static final int OPEN_NEW_ISSUE = 0;
	/** 既存の課題票を開く */
	public static final int OPEN_EXIST_ISSUE = 1;
	/** 新規に課題票を開く(開いた課題票にカテゴリを入れる) */
	public static final int OPEN_CATEGORY_ISSUE = 2;
	
	/** 課題票のオープンモード */
	private int	mode = OPEN_NEW_ISSUE;
	
	/** 課題票のカテゴリ */
	private String category = "";

	/** 課題票のFingerPrint */
	private String fingerPrint = "";

	/** 課題票種別 */
	private String type="";

	/**
	 * 呼び出し元のビューインスタンスを設定します。
	 * 課題票を新規で開くときに呼ばれるコンストラクタです。
	 * 
	 */
	public OpenIssueAction() {
		this(OPEN_NEW_ISSUE);
	}

	/**
	 * 課題票のオープンモードを設定します。
	 * 
	 * @param		mode			 課題票のオープンモード。
	 * 								{@link #OPEN_NEW_ISSUE OPEN_NEW_ISSUE--新規に課題票を開く}、
	 * 								{@link #OPEN_EXIST_ISSUE OPEN_EXIST_ISSUE--既存の課題票を開く}、
	 * 								{@link #OPEN_CATEGORY_ISSUE OPEN_CATEGORY_ISSUE--指定のカテゴリで新規に課題票を開く}、
	 */
	public OpenIssueAction(int mode) {
		super();
		this.mode = mode;

		
		// オープンモードにより処理を分ける
		if(mode == OPEN_NEW_ISSUE) {
			// 課題票の起票
			setText(Messages.getString("OpenIssueAction.0")); //$NON-NLS-1$
			setToolTipText(Messages.getString("OpenIssueAction.1")); //$NON-NLS-1$

			setImageDescriptor(NEW_ISSUE);
			setDisabledImageDescriptor(NEW_ISSUE_DISABLE);

		} else if(mode==OPEN_EXIST_ISSUE) {
			// 既存の課題票を開く
			setText(Messages.getString("OpenIssueAction.2")); //$NON-NLS-1$
			setToolTipText(Messages.getString("OpenIssueAction.3")); //$NON-NLS-1$
			setImageDescriptor(EXIST_ISSUE);
		} else if(mode==OPEN_CATEGORY_ISSUE) {
			// 指定のカテゴリで起票
			setText(Messages.getString("OpenIssueAction.4")); //$NON-NLS-1$
			setToolTipText(Messages.getString("OpenIssueAction.5")); //$NON-NLS-1$
			setImageDescriptor(NEW_ISSUE);
		}
	}

	/**
	 * 課題票を開きます。
	 */
	public void run() {
		try {
			// エディタに設定する情報を生成
			IssueMultiPageEditorInput input = new IssueMultiPageEditorInput();
			
			if(mode == OPEN_NEW_ISSUE) {

				// 新規に起票の場合。デフォルト値だけを設定しておく
				Issue openIssue = new Issue();
				openIssue.setVersion(0);
				openIssue.setType(IssueType.DRAFT_VALUE);
				openIssue.setStatus(IssueStatus.OPENED_VALUE);
				openIssue.setAttachmentList(new ArrayList());

				input.setIssue(openIssue);
				input.setOpenMode(IssueMultiPageEditorInput.OPEN_NEW_ISSUE);
				
			} else if(mode==OPEN_EXIST_ISSUE) {
				
				// 既存の課題票の場合。念のためDBから最新情報を取得する。
				ModifyIssueFacade issueModifyFacade = new ModifyIssueFacade(); 
				Issue openIssue = issueModifyFacade.getByFingerPrint(fingerPrint, type);

				// 課題票を取得できなかったら開かない。
				if(openIssue == null) {
					MessageDialog.openInformation(
							BtsPlugin.getInstance().getShell(),
							"Infomation", 
							Messages.getString("OpenIssueAction.6") //$NON-NLS-1$
							+ Messages.getString("OpenIssueAction.7") //$NON-NLS-1$
							+ Messages.getString("OpenIssueAction.8")); //$NON-NLS-1$
					return;
				} else {
					input.setIssue(openIssue);
					input.setOpenMode(IssueMultiPageEditorInput.OPEN_EXIST_ISSUE);
				}
			} else if(mode==OPEN_CATEGORY_ISSUE) {
				
				// カテゴリを指定して新規に起票の場合。デフォルト値とカテゴリを設定する
				Issue openIssue = new Issue();
				openIssue.setVersion(0);
				openIssue.setType(IssueType.DRAFT_VALUE);
				openIssue.setStatus(IssueStatus.OPENED_VALUE);
				openIssue.setCategory(this.category);
				openIssue.setAttachmentList(new ArrayList());

				input.setIssue(openIssue);
				input.setOpenMode(IssueMultiPageEditorInput.OPEN_NEW_ISSUE);
			}

			// エディタを開く
			BtsPlugin.getInstance().openEditor(input, true);
		}catch (Exception e) {
			String msg = Messages.getString("OpenIssueAction.9"); //$NON-NLS-1$
			logger.fatal(msg, e);
			BtsPlugin.getInstance().error(msg, e);
		}
	}

	/**
	 * オープンする課題票のカテゴリを設定します。
	 * 
	 * @param			category				オープンする課題票のカテゴリ
	 */
	public void setCategory(String category) {
		this.category = category;
	}

	
	/**
	 * オープンする課題票情報を設定します。。
	 * 
	 * @param			issue				オープンする課題票
	 */
	public void setIssue(Issue issue) {
		this.fingerPrint = issue.getFingerPrint();
		this.type = issue.getType();
	}
	
	
	/**
	 * オープンする課題票の一意識別するキーとなる情報のみを設定します。
	 * 
	 * @param		fingerPrint			課題票のFingerPrint
	 * @param		type				課題票種別
	 */
	public void setIssueKeys(String fingerPrint, String type) {
		this.fingerPrint = fingerPrint;
		this.type = type;
	}
}
