/***********************************************************************
 * Copyright(C) 2006 Valtech Co.,Ltd.
 * All Rights Reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 ***********************************************************************/
package jp.valtech.bts.ui;

import java.io.BufferedInputStream;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;

import jp.valtech.bts.command.client.Leave;
import jp.valtech.bts.data.CurrentProject;
import jp.valtech.bts.data.NetworkConfig;
import jp.valtech.bts.ui.editor.IssueMultiPageEditor;
import jp.valtech.bts.ui.editor.IssueMultiPageEditorInput;
import jp.valtech.bts.ui.issuelist.IssueListView;
import jp.valtech.bts.ui.msglist.MessageListView;
import jp.valtech.bts.ui.navigator.IProjectDialogSettings;
import jp.valtech.bts.ui.navigator.NavigatorView;
import jp.valtech.bts.ui.navigator.ProjectDialogSettings;
import jp.valtech.bts.util.Logging;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.IExtensionRegistry;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Status;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IEditorReference;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.ide.IDE;
import org.eclipse.ui.plugin.AbstractUIPlugin;
import org.osgi.framework.Bundle;
import org.osgi.framework.BundleContext;

/**
 * 
 * The main plugin class to be used in the desktop.
 * 
 * @author		<A href="mailto:m_sugitou@valtech.jp">M.Sugito</A>
 * @version	Ver.0.8
 */
public class BtsPlugin extends AbstractUIPlugin implements Logging {
 
	/** The shared instance.  */
	private static BtsPlugin plugin;
	
	/** リスナーのID  */
	public static final String ID_LISTENER= "jp.valtech.bts.listeners";
	
	/** 課題票一覧ビューのID  */
	public static final String ID_VIEW_ISSUELIST= "jp.valtech.bts.ui.issuelist.issueListView";
	
	/** メッセージ一覧ビューのID  */
	public static final String ID_VIEW_MESSAGELIST= "jp.valtech.bts.ui.msglist.messageListView";
	
	/** ナビゲータビューのID  */
	public static final String ID_VIEW_NAVIGATOR= "jp.valtech.bts.ui.navigator.navigatorView";
	
	/** 課題票エディタのID  */
	public static final String ID_EDITOR_ISSUE= "jp.valtech.bts.ui.editor.issueMultiPageEditor";

	/** リスナーを保持するリスト */
	private List listeners = new ArrayList();
	
	/** プロジェクト設定情報 */
	private ProjectDialogSettings projectDialogSettings = null;
	
	/** プロジェクト設定情報のファイル名 */
	private static final String FN_DIALOG_SETTINGS = "project_settings.xml";

	/**
	 * The constructor.
	 */
	public BtsPlugin() {
		plugin = this;
		
	}

	/**
	 * This method is called upon plug-in activation
	 */
	public void start(BundleContext context) throws Exception {
		try {
			super.start(context);
			
			// plubin.xmlに定義してあるリスナのインスタンスを生成します。
			computeListeners();
			
		} catch (Exception e) {
			String msg = Messages.getString("BtsPlugin.0"); //$NON-NLS-1$
			logger.fatal(msg, e);
			error(msg, e);
		}
	}


	/**
	 * 当プラグインで持つファイルへの参照をのURLインスタンスにて取得します。
	 *  
	 * @param		filePath		当プラグインで持つファイルのパス文字列
	 * @return		ファイル参照のURLインスタンス
	 */
    public URL getURL(String filePath) {
    	
    	try {
            // if the bundle is not ready then there is no image
            Bundle bundle = Platform.getBundle(getBundle().getSymbolicName());

            // look for the image (this will check both the plugin and fragment folders
            URL url = Platform.find(bundle, new Path(filePath));
            if (url == null) {
            	url = new URL(filePath);
            }

            return url;
    	} catch (Exception e) {
			error(e);
    		return null;
		}
    }

	/**
	 * ファイルをコピーします。
	 * 当プラグインの中で持つファイルを指定のパスに書き出します。
	 * 当プラグイン以外のファイルのコピーを行う場合は{@link jp.valtech.bts.util.BTSUtility#copy(String, String)}の方を使ってください。
	 * 
	 * @param		inputPath		当プラグインで持つファイルのPATH文字列		
	 * @param		outputPath		出力先のファイルPATH文字列（フルパス指定）
	 * @see			jp.valtech.bts.util.BTSUtility#copy(String, String)
	 */
    public void copy(String inputPath, String outputPath) {
    	
    	InputStream in = null;
    	FileOutputStream out = null;
    	
    	try {
            URL url = getURL(inputPath);
            if (url == null) {
            	return ;
            }

            in =  new BufferedInputStream(url.openStream());
            out = new FileOutputStream(outputPath);

            int c;
    	    while ((c = in.read()) != -1) {
    	    	out.write(c);
    	    }
            
    	} catch (Exception e) {
			error(e);
			
		} finally {
			
			if(in != null) {
				try {
					in.close();
				} catch (IOException e) {
					error(e);
				}
			}
			
			if(out != null) {
				try {
					out.close();
				} catch (IOException e) {
					error(e);
				}
			}
			
		}
    }
	

    /**
	 * This method is called when the plug-in is stopped
	 */
	public void stop(BundleContext context) throws Exception {
		super.stop(context);
		
		// プロジェクト情報の保存
		saveProjectDialogSettings();

		// オープンしているプロジェクトがある場合は、ネットワークからの離脱をメンバに通知します。
		CurrentProject current = CurrentProject.getInsance();
		if(current.isOpen()) {
			Leave leave = new Leave( new NetworkConfig() );
			leave.execute();
		}
		
		plugin = null;
	}

	/**
	 * Returns the shared instance.
	 */
	public static BtsPlugin getInstance() {
		return plugin;
	}
	

	/**
	 * IWorkbenchPage（アクティブなEclipseウィンドウのパースペクティブ）を取得する。
	 * 
	 * @return			アクティブなページ
	 */
	public IWorkbenchPage getPage() {
		// ワークベンチの取得
		IWorkbench workbench = PlatformUI.getWorkbench();
		// ワークベンチウィンドウの取得
		IWorkbenchWindow window = workbench.getActiveWorkbenchWindow();
		
		// アクティブなページの取得
		return window.getActivePage();
	}
	
	

	/**
	 * 拡張ポイントに定義されている{@link IBtsListener リスナークラス}を取得して当該インスタンスに保持します
	 *
	 */
	private void computeListeners() throws CoreException {
		IExtensionRegistry registry = Platform.getExtensionRegistry();
		IExtensionPoint extensionPoint= registry.getExtensionPoint(ID_LISTENER);
		IExtension[] extensions= extensionPoint.getExtensions();
		for (int i= 0; i < extensions.length; i++) {
			IConfigurationElement[] elements= extensions[i].getConfigurationElements();
			for (int j= 0; j < elements.length; j++) {
				Object listener= elements[j].createExecutableExtension("class");
				if (listener instanceof IBtsListener){
					listeners.add(listener);
				}
			}
		}
	}
	
	/**
	 * リスナクラスを登録します。
	 * 
	 * @param		listener			登録するリスナ
	 */
	public void addListener(IBtsListener listener) {
		listeners.add(listener);
	}

	/**
	 * 指定のリスナを削除します。
	 * 
	 * @param		listener			削除するリスナ
	 */
	public void removeListener(IBtsListener listener) {
		listeners.remove(listener);
	}
	
	/**
	 * 当該インスタンスで保持するリスナのリストを返します。
	 *  
	 * @return		リスナのリスト
	 */
	public List getListeners() {
		return listeners;
	}

	
	/**
	 * Returns an image descriptor for the image file at the given
	 * plug-in relative path.
	 *
	 * @param path the path
	 * @return the image descriptor
	 */
	public ImageDescriptor getImageDescriptor(String path) {
		path = "icons/" + path;
		ImageDescriptor image 
			= AbstractUIPlugin.imageDescriptorFromPlugin(getBundle().getSymbolicName(), path); 
		
		return image;
	}

	/**
	 * イメージファイルをかえします。
	 * 指定のイメージファイルは「%PROJECT_ROOT%/icons/」に存在する必要があります。
	 * 
	 * @param		path			イメージファイル名
	 * @return		指定したイメージ。
	 */
	public Image getImage(String path) {
		return getImageDescriptor(path).createImage();
	}
	
	
	/**
	 * シェルを取得します。
	 * 
	 * @return		シェル
	 */
	public Shell getShell() {
		IWorkbench workbench = PlatformUI.getWorkbench();
		IWorkbenchWindow window = workbench.getActiveWorkbenchWindow();
		return window.getShell();
	}
	
	
	/**
	 * アクティブなエディタを取得します。
	 * 
	 * 取得したエディタが{@link IssueMultiPageEditor}でない場合はnullを返します。
	 * 
	 * @return		アクティブなエディタのインスタンス
	 */
	public IssueMultiPageEditor getActiveEditor() {
		IEditorPart editor = getPage().getActiveEditor();
		if(editor != null && editor instanceof IssueMultiPageEditor) {
			return (IssueMultiPageEditor)editor;
		}
		return null;
	}
	
 
	/**
	 * 課題票エディタを開きます。
	 * 
	 * @param			input					課題票エディタ用のIEditorInputオブジェクト
	 * @param			activate				if <code>true</code> the editor will be activated				
	 * @throws			PartInitException
	 */
	public void openEditor(IssueMultiPageEditorInput input, boolean activate) throws PartInitException {
		IDE.openEditor(getPage(), input, ID_EDITOR_ISSUE, activate);
	}
	

    /**
     * Closes the given editor. The editor must belong to this workbench page.
     * <p>
     * If the editor has unsaved content and <code>save</code> is <code>true</code>,
     * the user will be given the opportunity to save it.
     * </p>
     * 
     * @param			editor		the editor to close				
     * @param 			save		<code>true</code> to save the editor contents if required
     *            					(recommended), and <code>false</code> to discard any
     *            					unsaved changes
     * @return 			<code>true</code> if the editor was successfully closed, and
     *         			<code>false</code> if the editor is still open
     */
	public boolean closeEditor(IEditorPart editor, boolean save) {
		return getPage().closeEditor(editor, save);
	}
	
	
	/**
	 * 指定の課題票エディタを閉じます。
	 * 
	 * @param		fingerprint		課題票のFingerPrint
	 * @param		type			課題票の種別
	 */
	public void closeIssueEditor(String fingerprint, String type) {
		
		// 開いている課題票エディタを取得します。
		IssueMultiPageEditor[] editor = getIssueEditors();
		if(editor ==null) {
			return;
		}
		for (int i = 0; i < editor.length; i++) {
			IssueMultiPageEditorInput editorInput
				= (IssueMultiPageEditorInput)editor[i].getEditorInput();
			
			// 課題票エディタから課題表情を取得し、引数の課題票情報と比較します。
			if(fingerprint.equals(editorInput.getIssue().getFingerPrint()) 
				   && type.equals(editorInput.getIssue().getType()) ) {

				// 比較結果が同じであれば当該エディタを閉じます。 
				closeEditor(editor[i], false);
				continue;
			}
		}
	}
	
	
	/**
	 * 現在開いている全ての課題票エディタを閉じます。
	 *
	 */
	public void closeAllIssueEditors() {
		
		// 開いている課題票エディタを取得します。
		IssueMultiPageEditor[] editor = getIssueEditors();
		if(editor ==null) {
			return;
		}
		
		// 課題票エディタを全部閉じる
		for (int i = 0; i < editor.length; i++) {
			closeEditor(editor[i], false);
		}
	}

	
	
	/**
	 * 開いている課題票エディタを配列で取得します。
	 * 
	 * @return			開いている課題票エディタの配列
	 */
	public IssueMultiPageEditor[] getIssueEditors() {
		ArrayList list = new ArrayList();
		
		IEditorReference[] editorRef = getPage().getEditorReferences();
		for (int i = 0; i < editorRef.length; i++) {
			try {
				IEditorPart editorPart = editorRef[i].getEditor(false);
				if(editorPart instanceof IssueMultiPageEditor) {
					list.add(editorPart);
				}
			} catch (Exception e) {
				errorlog(e);
			}
		}
		return (IssueMultiPageEditor[])list.toArray(new IssueMultiPageEditor[0]);
	}

	/**
	 * 指定の課題票のエディタを指定します。
	 * 指定の課題票のエディタが開いてないときはnullを返します。
	 * 
	 * @return			開いている課題票エディタの配列
	 */
	public IssueMultiPageEditor getIssueEditor(String fingerPrint, String type) {
		// 開いている課題票エディタを取得します。
		IssueMultiPageEditor[] editors = getIssueEditors();
		if(editors ==null) {
			return null;
		}
		
		for (int i = 0; i < editors.length; i++) {
			IssueMultiPageEditorInput editorInput
				= (IssueMultiPageEditorInput)editors[i].getEditorInput();
			
			// 課題票エディタから課題表情を取得し、引数の課題票情報と比較します。
			if(fingerPrint.equals(editorInput.getIssue().getFingerPrint()) 
				   && type.equals(editorInput.getIssue().getType()) ) {

				// 比較結果が同じであれば当該エディタを閉じます。 
				return editors[i];
			}
		}
		return null;
	}

	
	/**
	 * ナビゲータビューインスタンスを取得します。
	 * 
	 * @return		ナビゲータビューインスタンス
	 */
	public NavigatorView getNavigatorView() {
        return (NavigatorView)getPage().findView(ID_VIEW_NAVIGATOR);
	}
	
	/**
	 * 課題票一覧ビューインスタンスを取得します。
	 * 
	 * @return		課題票一覧ビューインスタンス
	 */
	public IssueListView getIssueView() {
        return (IssueListView)getPage().findView(ID_VIEW_ISSUELIST);
	}
	
	/**
	 * メッセージ一覧ビューインスタンスを取得します。
	 * 
	 * @return		課題票一覧ビューインスタンス
	 */
	public MessageListView getMessageView() {
        return (MessageListView)getPage().findView(ID_VIEW_MESSAGELIST);
	}

	/**
	 * エラー処理です。
	 * 
	 * 例外を取得したときには当メソッドを呼びます。
	 * 以下のエラー処理を行います。
	 * <pre>
	 * 	・エラーログ出力。
	 * 　　ワークスペースの.metadata/.logに書き出されます。
	 * 　　EclipseのUIからは[ウィンドウ]-[ビューを表示-[エラーログ]で確認できます。
	 *    
	 * 	・エラーダイアログ表示。
	 * 　　エラー概要を示したダイアログを表示します。
	 * </pre>
	 * @param		msg			エラーメッセージ
	 */
	public void error(Throwable ex) {
		error(ex.getMessage(), ex);
	}
	

	/**
	 * エラー処理です。
	 * 
	 * 例外を取得したときには当メソッドを呼びます。
	 * 以下のエラー処理を行います。
	 * <pre>
	 * 	・エラーログ出力。
	 * 　　ワークスペースの.metadata/.logに書き出されます。
	 * 　　EclipseのUIからは[ウィンドウ]-[ビューを表示-[エラーログ]で確認できます。
	 *    
	 * 	・エラーダイアログ表示。
	 * 　　エラー概要を示したダイアログを表示します。
	 * </pre>
	 * @param		msg			エラーメッセージ
	 * @param		ex			例外
	 */
	public void error(String msg, Throwable ex) {

		// エラー情報をログに出力
		IStatus status= new Status
			( IStatus.ERROR
			, getBundle().getSymbolicName()
			, 0
			, msg
			, ex);
		getLog().log(status);
		
		// ダイアログ表示
		ErrorDialog.openError
			( getShell()
			, "ERROR!"
			, Messages.getString("BtsPlugin.1") //$NON-NLS-1$
			, status);

		ex.printStackTrace();
	}
	
	/**
	 * エラー処理です。
	 * ログだけを残してエラーダイアログを表示しない場合に呼びます。
	 * 
	 * @param		ex			例外
	 */
	public void errorlog(Throwable ex) {
		errorlog(ex.getMessage(), ex);
	}
	
	/**
	 * エラー処理です。
	 * ログだけを残してエラーダイアログを表示しない場合に呼びます。
	 * 
	 * @param		msg			エラーメッセージ
	 * @param		ex			例外
	 */
	public void errorlog(String msg, Throwable ex) {

		// エラー情報をログに出力
		IStatus status= new Status
			( IStatus.ERROR
			, getBundle().getSymbolicName()
			, 0
			, msg
			, ex);
		getLog().log(status);

		ex.printStackTrace();
	}

	
	/**
	 * プロジェクト設定情報を取得します。
	 * プロジェクト設定情報の取得には{@link AbstractUIPlugin}クラスを利用しません。
	 */
    public IProjectDialogSettings getProjectDialogSettings() {
    	
        if (projectDialogSettings == null) {
            loadProjectDialogSettings();
        }

        return projectDialogSettings;
    }
	
    
    /**
     * ProjectDialogSetting情報を読み込みます。
	 * プロジェクト設定情報の読み込みには{@link AbstractUIPlugin}クラスを利用しません。
     */
    public void loadProjectDialogSettings() {
    	
		projectDialogSettings = new ProjectDialogSettings("Workbench");

		String readWritePath = getStateLocation().append(FN_DIALOG_SETTINGS).toOSString();
		
		File settingsFile = new File(readWritePath);
		
		if (settingsFile.exists()) {
			
			try {
				projectDialogSettings.load(readWritePath);
			} catch (IOException e) {
				projectDialogSettings = new ProjectDialogSettings("Workbench");
			}
			
		} else {
			
            // if the bundle is not ready then there is no image
            Bundle bundle = Platform.getBundle(getBundle().getSymbolicName());

            // look for the image (this will check both the plugin and fragment folders
            URL dsURL = Platform.find(bundle, new Path(FN_DIALOG_SETTINGS));
			
			if (dsURL == null) {
				return;
			}

			InputStream is = null;
			
			try {
				
				is = dsURL.openStream();
				
				BufferedReader reader = new BufferedReader(new InputStreamReader(is, "utf-8"));
				
				projectDialogSettings.load(reader);
				
			} catch (IOException e) {
				
				// load failed so ensure we have an empty settings
				projectDialogSettings = new ProjectDialogSettings("Workbench");
				
			} finally {
				
				try {
					
					if (is != null) {
						is.close();
					}
					
				} catch (IOException e) {
					// do nothing
				}
				
			}
			
		}

	}
    
    /**
     * ProjectDialogSetting情報を保存します。
	 * プロジェクト設定情報の保存には{@link AbstractUIPlugin}クラスを利用しません。
     */
    public void saveProjectDialogSettings() {
    	
        if (projectDialogSettings == null) {
            return;
        }

        try {
            String readWritePath = getStateLocation().append(FN_DIALOG_SETTINGS).toOSString();
            projectDialogSettings.save(readWritePath);
        } catch (IOException e) {
            // spec'ed to ignore problems
        }
        
    }
	
}	

