/***********************************************************************
 * Copyright(C) 2006 Valtech Co.,Ltd.
 * All Rights Reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 ***********************************************************************/
package jp.valtech.bts.network.server;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.net.ServerSocket;
import java.net.Socket;
import java.net.SocketException;

import jp.valtech.bts.data.NetworkConfig;
import jp.valtech.bts.network.Request;
import jp.valtech.bts.network.command.TCPServerCommand;
import jp.valtech.bts.util.BTSUtility;
import jp.valtech.bts.util.Logging;

/**
 * TCP通信サーバプロセス。
 * TCP通信で通知されるコマンドの受信処理を行います。
 * 
 * @author		<A href="mailto:m_sugitou@valtech.jp">M.Sugito</A>
 * @version	Ver.0.8
 */
class TCPServer implements Runnable, Logging {

	/** ネットワーク情報 */
	private NetworkConfig myconfig	= null;
	
	/** TCP通信のレシーバインスタンス */
	private ServerSocket serverSocket = null;


	/**
	 * TCP通信サーバインスタンスを生成します。
	 * 
	 * @param	config		設定情報
	 */
	public TCPServer( NetworkConfig myconfig ) throws IOException {
		super();
		this.myconfig		= myconfig;

		// TCP通信で使用するポートを取得
		int port = myconfig.getTCPPort();
		
		// 通信開始。失敗したら例外を投げます。
		serverSocket = new ServerSocket( port );
		logger.info( Messages.getString("TCPServer.0") + port + "}"); //$NON-NLS-1$
	}
	
	/**
	 * TCP通信受信処理スレッド開始
	 * 
	 * 他のクライアントから接続要求を受け付けると、受信データから実行されるコマンドクラスを判定して
	 * {@link ServerCommand#getCommand(String, String)}よりコマンドクラスを取得して実行します。
	 * 
	 * @see java.lang.Runnable#run()
	 */
	public void run() {

		// 通信を待ち受けます。受信後、次の待ち受けに入るようにループします。
		boolean active = true;
		while(active) {
			try{
				// 接続が確立されるまで、このメソッドの中でブロックします。
				Socket socket	= this.serverSocket.accept(); 

				// 接続要求がきたので受信オブジェクトを生成します。
				ObjectInputStream is = new ObjectInputStream( socket.getInputStream() );
				Request request = (Request)is.readObject(); 

				// 実行するコマンドのインスタンスを生成します。
				Object command = BTSUtility.getCommand(request.getCommand());

				//	コマンドがTCPServerCommandのインスタンスでれば実行します。
				if( command != null && command instanceof TCPServerCommand){
					// 接続情報をコマンドインスタンスに格納
					TCPServerCommand tcpCommand = (TCPServerCommand)command;
					tcpCommand.setRequest( request );
					tcpCommand.setSocket( socket );
					tcpCommand.setConfig(myconfig);
				
					// 受信コマンドをスレッドで実行
					Thread t = new Thread(tcpCommand, "TCPServerCommand");
					t.start();

				}else{
					throw new ClassNotFoundException(Messages.getString("TCPServer.1")); //$NON-NLS-1$
				}

			} catch ( SocketException sx) {
				// この例外を受け取るとループを抜けます。
				logger.info(Messages.getString("TCPServer.2")); //$NON-NLS-1$
				active = false;
			} catch ( Exception e ) {
				logger.error(Messages.getString("TCPServer.3"), e ); //$NON-NLS-1$
			}
		}
		logger.info(Messages.getString("TCPServer.4")); //$NON-NLS-1$
	}
	
	
	/**
	 * TCP通信サーバプロセスを停止します。 
	 * このメソッドを実行することで{@link #run()}のループ処理内で使っている{@link java.net.ServerSocket}
	 * が{@link SocketException}をスローします。
	 * それにより同ループ処理を抜け出してスレッドが終了します。
	 */
	public void stop() throws IOException {
		if(serverSocket != null) {
			serverSocket.close();
			serverSocket = null;
		}
	}
}