/***********************************************************************
 * Copyright(C) 2006 Valtech Co.,Ltd.
 * All Rights Reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 ***********************************************************************/
package jp.valtech.bts.network;

import java.net.InetAddress;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.Map;
import java.util.TreeMap;

import jp.valtech.bts.data.CurrentProject;
import jp.valtech.bts.ui.BtsPlugin;
import jp.valtech.bts.ui.preference.BtsPreference;

/**
 * <dl><dt><b>他のクライアント一覧</b></dt>
 * <dd>
 * 	他のクライアントから、{@link jp.valtech.bts.command.client.Join}コマンドを送信して自身のクライアント情報を通知した場合、クライアント一覧に送信元のIPアドレスおよび名前が追加されます。
 * 	追加された名前は、このクライアント一覧クラスより取得することが可能です。<br>
 * 	また、{@link jp.valtech.bts.command.client.Leave}コマンドを送信した場合は、クライアント一覧クラスより送信元のIPアドレスおよび名前が削除されます。<br>
 * <br>
 * このクライアント一覧から必要な情報を取得する場合には、{@link #addresses()}で登録されている他のクライアントのIPアドレスの一覧を取得後、
 * {@link #getClientName(String)}で、IPアドレスを引数に指定して名前を取得してください。
 * </dd>
 * </dl>
 * 
 * @see			jp.valtech.bts.command.client.Join
 * @see			jp.valtech.bts.command.client.Leave
 * @see			jp.valtech.bts.command.server.JoinServer
 * @see			jp.valtech.bts.command.server.LeaveServer
 * @author		<A href="mailto:iinuma@valtech.jp">K.Iinuma</A>
 * @version	Ver.0.8
 */
public class ClientList {

	/** クライアントの一覧 */
	private static Map		clients			= null;

	/** インスタンス */
	private static ClientList clientList	= null;

	/**
	 * <DL><DT><B>コンストラクタ</B>
	 *  <DD></DD>
	 * </DL>
	 */
	private ClientList() {
		super();
	}

	/**
	 * <DL><DT><B>クライアント一覧のインスタンスを取得します</B></DL>
	 *  <DD></DD>
	 * </DL>
	 * @return	クライアントリストのインスタンス
	 */
	public static synchronized ClientList getInstance(){
		if( clientList == null ){
			clientList = new ClientList();
		}
		return clientList;
	}

	/**
	 * <DL><DT><B>IPアドレスにクライアント情報を関連付けて保存します</B></DL>
	 *  <DD>
	 *  クライアント情報は{@link ClientInfo#getClientAddress()}で取得できるIPアドレス情報をキーとして保存されます。
	 *  重複したIPアドレスは上書きされます。<br><br>
	 *  保管しているIPアドレスの一覧は{@link #addresses()}で取得可能です。
	 *  </DD>
	 * </DL>
	 * @param ipAddress 		IPアドレス
	 * @param clientName		クライアント名
	 * @param port 			ポート番号
	 */
	public static synchronized void add( InetAddress ipAddress, String clientName, int port ){
		ClientInfo clientInfo = new ClientInfo();
		clientInfo.setClientAddress( ipAddress.getHostAddress() );
		clientInfo.setClientHostName( ipAddress.getHostName() );
		clientInfo.setClientName( clientName );
		clientInfo.setPort( new Integer( port ) );

		add( clientInfo );
	}

	/**
	 * <DL><DT><B>クライアント情報を保存します</B></DL>
	 *  <DD>
	 *  クライアント情報は{@link ClientInfo#getClientAddress()}で取得できるIPアドレス情報をキーとして保存されます。
	 *  重複したIPアドレスは上書きされます。<br><br>
	 *  保管しているIPアドレスの一覧は{@link #addresses()}で取得可能です。
	 *  </DD>
	 * </DL>
	 * @param clientInfo	クライアント情報
	 */
	public static synchronized void add( ClientInfo clientInfo ){
		if( clients == null ){
			clients	= Collections.synchronizedMap( new TreeMap() );
		}

		clients.put( clientInfo.getClientAddress(), clientInfo );		
	}
	

	/**
	 * <DL><DT><B>自分自身のクライアント情報を保存します</B></DL>
	 *  <DD>
	 *  </DD>
	 * </DL>
	 */
	public static synchronized void addMyInfo() {
		try {
			InetAddress inetAddress = InetAddress.getLocalHost();		// ローカルホスト情報
			String userName = CurrentProject.getInsance().getProjectConfig().getUserName();	// ユーザ名
			int tcpPort = BtsPreference.getInstance().getTCPPort();		// TCPポート番号
			
			add( inetAddress, userName, tcpPort) ;
		} catch (Exception e) {
			BtsPlugin.getInstance().errorlog(e);
		}
	}

	/**
	 * <DL><DT><B>IPアドレスに関連付けられたクライアント名を取得</B></DL>
	 *  <DD></DD>
	 * </DL>
	 * @param ipAddress	IPアドレス
	 * @return				クライアント名
	 */
	public static synchronized String getClientName( InetAddress ipAddress ){
		return getClientName( ipAddress.getHostAddress() );
	}

	/**
	 * <DL><DT><B>IPアドレスに関連付けられたクライアント名を取得</B></DL>
	 *  <DD></DD>
	 * </DL>
	 * @param ipAddress	IPアドレス
	 * @return				クライアント名
	 */
	public static synchronized String getClientName( String ipAddress ){
		ClientInfo clientInfo = getClientInfo( ipAddress );
		String result = null;
		if( clientInfo != null ){
			result = clientInfo.getClientName();
		}
		return result;
	}

	
	/**
	 * <DL><DT><B>{@link ClientInfo クライアント情報}の配列を取得</B></DL>
	 *  <DD></DD>
	 * </DL>
	 * @return				クライアント情報の配列を取得
	 */
	public static ClientInfo[] getClientInfoList() {
		Collection col = clients.values();
		return (ClientInfo[])col.toArray(new ClientInfo[0]);
	}
	

	/**
	 * <DL><DT><B>IPアドレスに関連付けられたポート番号を取得</B></DL>
	 *  <DD></DD>
	 * </DL>
	 * @param ipAddress	IPアドレス
	 * @return				ポート番号
	 */
	public static synchronized int getPort( String ipAddress ) {
		ClientInfo clientInfo = getClientInfo( ipAddress );
		int result = -1;
		if( clientInfo != null ){
			result = clientInfo.getPort().intValue();
		}
		return result;
	}

	/**
	 * <DL><DT><B>IPアドレスに関連付けられたクライアント情報を取得</B></DL>
	 *  <DD></DD>
	 * </DL>
	 * @param ipAddress	IPアドレス
	 * @return				クライアント情報
	 */
	public static synchronized ClientInfo getClientInfo( String ipAddress ){
		ClientInfo clientInfo = null;
		if( ( clients != null ) && ( clients.containsKey( ipAddress ) ) ){
			clientInfo = (ClientInfo)clients.get( ipAddress );
		}
		
		return clientInfo;
	}

	
	public static ClientInfo getClientInfoByClientName(String userName) {
		Collection col = clients.values();
		Iterator iterator = col.iterator();
		while(iterator.hasNext()) {
			ClientInfo info = (ClientInfo)iterator.next();
			if(userName.equals(info.getClientName())) {
				return info;
			}
		}
		return null;
	}

	/**
	 * <DL><DT><B>IPアドレスに関連付けられたクライアント情報を取得</B></DL>
	 *  <DD></DD>
	 * </DL>
	 * @param ipAddress	IPアドレス
	 * @return				クライアント情報
	 */
	public static synchronized ClientInfo getClientInfo( InetAddress inetAddress ){
		return getClientInfo(inetAddress.getHostAddress());
	}

	/**
	 * <DL><DT><B>IPアドレスに関連付けられたクライアント情報を取得</B></DL>
	 *  <DD></DD>
	 * </DL>
	 * @param ipAddress	IPアドレス
	 * @return				クライアント情報
	 */
	public static synchronized ClientInfo getMyInfo(){
		try {
			InetAddress inetAddress = InetAddress.getLocalHost();
			return getClientInfo(inetAddress.getHostAddress());
		} catch (Exception e) {
			BtsPlugin.getInstance().errorlog(e);
			return null;
		}
	}

	/**
	 * <DL><DT><B>IPアドレスに関連付けられたクライアント情報を削除します</B></DL>
	 *  <DD></DD>
	 * </DL>
	 * @param ipAddress	IPアドレス
	 */
	public static synchronized void remove( InetAddress ipAddress ){
		if( clients != null ){
			clients.remove( ipAddress.getHostAddress() );
		}
	}

	/**
	 * <DL><DT><B>クライアント一覧に登録されているIPアドレスを取得します。</B></DL>
	 *  <DD></DD>
	 * </DL>
	 * @return	クライアントの一覧
	 */
	public static synchronized String[] addresses(){
		String[] result = new String[]{};
		if( clients != null ){
			result = (String[])clients.keySet().toArray( result );
		}
		return result;
	}

	/**
	 * <DL><DT><B></B></DL>
	 *  <DD></DD>
	 * </DL>
	 */
	public static synchronized void clear() {
		if( clients != null ){
			clients.clear();
			clients = null;
		}
	}

}
